(*  Title:      CoW/Submonoids.thy
    Author:     Štěpán Holub, Charles University
    Author:     Martin Raška, Charles University
    Author:     Štěpán Starosta, CTU in Prague

Part of Combinatorics on Words Formalized. See https://gitlab.com/formalcow/combinatorics-on-words-formalized/

*)


theory Submonoids
  imports CoWBasic "HOL.Hull"
begin

chapter \<open>Submonoids of a free monoid\<close>

text\<open>This chapter deals with properties of submonoids of a free monoid, that is, with monoids of words.
See more in Chapter 1 of \cite{ Lo83}.
\<close>

section \<open>Generated monoid (also called hull)\<close>

text\<open>First, we define the hull of a set of words, that is, the monoid generated by them.\<close>

definition word_monoid where
  "word_monoid M \<equiv> (\<forall> w1 w2. w1 \<in> M \<longrightarrow> w2 \<in> M \<longrightarrow> w1 \<cdot> w2 \<in> M) \<and> \<epsilon> \<in> M"

lemma word_monoidI[intro]: assumes "\<And> w1 w2. w1 \<in> M \<Longrightarrow> w2 \<in> M \<Longrightarrow> w1 \<cdot> w2 \<in> M" "\<epsilon> \<in> M"
  shows "word_monoid M"
  by (simp add: assms word_monoid_def)

inductive_set hull :: "'a list set \<Rightarrow> 'a list set" ("\<langle>_\<rangle>")
  for G where
    emp_in[simp]:  "\<epsilon> \<in> \<langle>G\<rangle>" |
    prod_cl:  "w1 \<in> G \<Longrightarrow> w2 \<in> \<langle>G\<rangle> \<Longrightarrow> w1 \<cdot> w2 \<in> \<langle>G\<rangle>"

lemmas [intro] = hull.intros

lemma hull_closed[intro]: "w1 \<in> \<langle>G\<rangle> \<Longrightarrow> w2 \<in> \<langle>G\<rangle> \<Longrightarrow> w1 \<cdot> w2 \<in> \<langle>G\<rangle>"
  by (rule hull.induct[of w1 G "\<lambda> x. (x\<cdot>w2)\<in>\<langle>G\<rangle>"]) auto+

lemma gen_in [intro]: "w \<in> G \<Longrightarrow> w \<in> \<langle>G\<rangle>"
  using hull.prod_cl by force

lemma gen_monoid_monoid: "word_monoid \<langle>G\<rangle>"
  unfolding word_monoid_def using hull_closed emp_in by blast

lemma gen_monoid_hull: "word_monoid hull G = \<langle>G\<rangle>"
proof (rule hull_unique, blast, use gen_monoid_monoid in blast,rule subsetI)
  fix M x
  assume "G \<subseteq> M" and "word_monoid M" "x \<in> \<langle>G\<rangle>"
  show "x \<in> M"
  by (induction rule: hull.induct[OF \<open>x \<in> \<langle>G\<rangle>\<close>])
  (use \<open>word_monoid M\<close>[unfolded word_monoid_def] \<open>G \<subseteq> M\<close> in blast)+
qed

lemma gen_monoid_induct: assumes "x \<in> \<langle>G\<rangle>" "P \<epsilon>" "\<And>w. w \<in> G \<Longrightarrow> P w"
  "\<And>w1 w2. w1 \<in> \<langle>G\<rangle> \<Longrightarrow> P w1 \<Longrightarrow> w2 \<in> \<langle>G\<rangle> \<Longrightarrow> P w2 \<Longrightarrow> P (w1 \<cdot> w2)" shows  "P x"
  using hull.induct[of _ _ P, OF \<open>x \<in> \<langle>G\<rangle>\<close>  \<open>P \<epsilon>\<close>]
    assms by (simp add: gen_in)

lemma genset_sub[simp]: "G \<subseteq> \<langle>G\<rangle>"
  using gen_in ..

lemma lists_sub: "ws \<in> lists G \<Longrightarrow> ws \<in> lists \<langle>G\<rangle>"
  using sub_lists_mono[OF genset_sub].

lemma in_lists_conv_set_subset: "set ws \<subseteq> G \<longleftrightarrow> ws \<in> lists G"
  unfolding in_lists_conv_set subset_code(1)..

lemma concat_in_hull [intro]:
  assumes "set ws \<subseteq> G"
  shows   "concat ws \<in> \<langle>G\<rangle>"
  using assms by (induction ws) auto

lemma concat_in_hull' [intro]:
  assumes "ws \<in> lists G"
  shows   "concat ws \<in> \<langle>G\<rangle>"
  using assms by (induction ws) auto

lemma hull_concat_lists0:
   "w \<in> \<langle>G\<rangle> \<Longrightarrow> \<exists> ws \<in> lists G. concat ws = w" (is "w \<in> \<langle>G\<rangle> \<Longrightarrow> ?P w")
proof(rule hull.induct[of _ G])
  show "\<exists>ws\<in>lists G. concat ws = w1 \<cdot> w2" if "w1 \<in> G" "w2 \<in> \<langle>G\<rangle>"
 "\<exists>ws\<in>lists G. concat ws = w2"  for w1 w2
    using that Cons_in_lists_iff concat.simps(2) by metis
qed (use concat_nemp in blast)+

lemma hull_concat_listsE[elim]: assumes "w \<in> \<langle>G\<rangle>"
  obtains ws where "ws \<in> lists G" and "concat ws = w"
  using assms hull_concat_lists0 by blast

lemma hull_concat_lists: "\<langle>G\<rangle> = concat ` lists G"
  using hull_concat_lists0 by blast

lemma hull_concat_lists': "\<langle>G\<rangle> = {concat xs | xs. xs \<in> lists G}"
  using Setcompr_eq_image hull_concat_lists by blast

lemma hull_mono: "A \<subseteq> B \<Longrightarrow> \<langle>A\<rangle> \<subseteq> \<langle>B\<rangle>"
  using gen_monoid_hull hull_mono by blast

lemma hull_concat_lists_nempE[elim]: assumes "w \<in> \<langle>G\<rangle>"
  obtains ws where "ws \<in> lists (G - {\<epsilon>})" and "concat ws = w"
proof-
  obtain ws where "ws \<in> lists G" "concat ws = w"
    using assms by blast
  with that[of "filter (\<lambda> x. x \<noteq> \<epsilon>) ws"]
  show thesis
    by force
qed

lemma hull_nemp_eq_hull[simp]: "\<langle>G - {\<epsilon>}\<rangle> = \<langle>G\<rangle>"
  by (rule equalityI, use hull_mono in fast, rule subsetI, blast)



lemma emp_gen_set: "\<langle>{}\<rangle> = {\<epsilon>}"
  unfolding hull_concat_lists by auto

lemma concat_lists_minus[simp]: "concat ` lists (G - {\<epsilon>}) = concat ` lists G"
proof
  show "concat ` lists G \<subseteq> concat ` lists (G - {\<epsilon>})"
  proof
    fix x assume "x \<in> concat ` lists G"
    from imageE[OF this]
    obtain y where "x = concat y" "y \<in> lists G".
    from lists_minus'[OF \<open>y \<in> lists G\<close>] del_emp_concat[of y, folded \<open>x = concat y\<close>]
    show "x \<in> concat ` lists (G - {\<epsilon>})"
      by blast
  qed
qed (simp add: image_mono lists_mono)

lemma hull_drop_one: "\<langle>G - {\<epsilon>}\<rangle> = \<langle>G\<rangle>"
proof (intro equalityI subsetI)
  fix x assume "x \<in> \<langle>G\<rangle>" thus "x \<in> \<langle>G - {\<epsilon>}\<rangle>"
    unfolding  hull_concat_lists by simp
next
  fix x assume "x \<in> \<langle>G - {\<epsilon>}\<rangle>" thus "x \<in> \<langle>G\<rangle>"
    unfolding  hull_concat_lists image_iff by auto
qed

lemma sing_gen_power: "u \<in> \<langle>{x}\<rangle> \<Longrightarrow> \<exists>k. u = x\<^sup>@k"
  unfolding hull_concat_lists  using one_generated_list_power by auto


lemma pow_sing_gen[simp]: "x\<^sup>@k \<in> \<langle>{x}\<rangle>"
  using concat_in_hull[OF sing_pow_set_sub, of _  k x] concat_pow_list_single by simp

lemma sing_gen_pow_ex_conv: "u \<in> \<langle>{x}\<rangle> \<longleftrightarrow> (\<exists>k. u = x\<^sup>@k)"
  using sing_gen_power pow_sing_gen by blast

lemma sing_gen_primroot_gen:  assumes "w \<in> \<langle>{x}\<rangle>"
  shows "w \<in> \<langle>{\<rho> x}\<rangle>"
  using sing_gen_power[OF assms] pow_sing_gen
  by (subst (asm) (1) primroot_exp_eq[of x, symmetric],
  unfold pow_mult[symmetric]) blast

lemma sing_gen_primroot_eq:
  assumes "w \<in> \<langle>{x}\<rangle>" "w \<noteq> \<epsilon>"
  shows "\<rho> w = \<rho> x"
proof-
  have "\<rho> x \<noteq> \<epsilon>"
    using assms concat.simps(1) by fastforce
  from sing_gen_primroot_gen[OF \<open>w \<in> \<langle>{x}\<rangle>\<close>]
  show "\<rho> w = \<rho> x"
    using primroot_unique[OF \<open>w \<noteq> \<epsilon>\<close> primroot_prim[OF \<open>\<rho> x \<noteq> \<epsilon>\<close>, unfolded primroot_idemp]]  sing_gen_power[OF \<open>w \<in> \<langle>{\<rho> x}\<rangle>\<close>] by blast
qed

lemma sing_gen_pow_conv: "w \<in> \<langle>{\<rho> x}\<rangle> \<longleftrightarrow> w = \<rho> x\<^sup>@e\<^sub>\<rho> w"
proof (rule iffI, cases "w = \<epsilon>", force)
  assume "w \<in> \<langle>{\<rho> x}\<rangle>" "w \<noteq> \<epsilon>"
  have "\<rho> w = \<rho> x"
    using sing_gen_primroot_eq[OF \<open>w \<in> \<langle>{\<rho> x}\<rangle>\<close> \<open>w \<noteq> \<epsilon>\<close>, unfolded primroot_idemp].
  show "w = \<rho> x \<^sup>@ e\<^sub>\<rho> w"
    using primroot_exp_eq[of w, unfolded \<open>\<rho> w = \<rho> x\<close>, symmetric].
qed (metis pow_sing_gen)

lemma two_primroots_comm: assumes "w \<noteq> \<epsilon>" and  "w \<in> \<langle>{\<rho> x}\<rangle>" and "w \<in> \<langle>{\<rho> y}\<rangle>"
  shows "x \<cdot> y = y \<cdot> x"
proof (rule comm_primrootI)
  from assms[unfolded sing_gen_pow_conv]
  show "\<rho> x \<cdot> \<rho> y = \<rho> y \<cdot> \<rho> x"
    using comm_drop_exps primroot_exp_nemp by metis
qed

lemma comm_rootI: "x \<in> \<langle>{r}\<rangle> \<Longrightarrow> y \<in> \<langle>{r}\<rangle> \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  unfolding comm sing_gen_pow_ex_conv by blast


lemma sing_genE[elim]:
  assumes "u \<in> \<langle>{x}\<rangle>"
  obtains k where "x\<^sup>@k = u"
  using assms using sing_gen_power by blast


lemma lists_gen_to_hull: "us \<in> lists (G - {\<epsilon>}) \<Longrightarrow> us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})"
  using lists_mono genset_sub  by blast

lemma rev_hull: "rev`\<langle>G\<rangle> = \<langle>rev`G\<rangle>"
proof
  show "rev ` \<langle>G\<rangle> \<subseteq> \<langle>rev ` G\<rangle>"
  proof
    fix x assume "x \<in> rev ` \<langle>G\<rangle>"
    then obtain xs where "x = rev (concat xs)" and "xs \<in> lists G"
      unfolding hull_concat_lists by auto
    from rev_in_lists[OF \<open>xs \<in> lists G\<close>]
    have "(map rev (rev xs)) \<in> lists (rev ` G)"
      by fastforce
    thus "x \<in> \<langle>rev ` G\<rangle>"
      unfolding image_iff hull_concat_lists
      using \<open>x = rev (concat xs)\<close>[unfolded rev_concat] by blast
  qed
  show "\<langle>rev ` G\<rangle> \<subseteq> rev ` \<langle>G\<rangle>"
  proof
    fix x assume  "x \<in>  \<langle>rev ` G\<rangle>"
    then obtain xs where "x = concat xs" and "xs \<in> lists (rev ` G)"
      unfolding hull_concat_lists by blast
    from rev_in_lists[OF \<open>xs \<in> lists ((rev ` G))\<close>]
    have "map rev (rev xs) \<in> lists G"
      by fastforce
    hence "rev x \<in> \<langle>G\<rangle>"
      unfolding \<open>x = concat xs\<close> rev_concat
      by fast
    thus "x \<in> rev ` \<langle>G\<rangle>"
      unfolding rev_in_conv.
  qed
qed

lemma power_in[intro]: "x \<in> \<langle>G\<rangle> \<Longrightarrow> x\<^sup>@k \<in> \<langle>G\<rangle>"
  by (induction k, auto)

lemma hull_closed_lists:  "us \<in> lists \<langle>G\<rangle> \<Longrightarrow> concat us \<in> \<langle>G\<rangle>"
  by (induct us, auto)

lemma hull_I [intro]:
  "\<epsilon> \<in> H \<Longrightarrow> (\<And> x y. x \<in> H \<Longrightarrow> y \<in> H \<Longrightarrow> x \<cdot> y \<in> H) \<Longrightarrow> \<langle>H\<rangle> = H"
  by (standard, use hull.induct[of _ H "\<lambda> x. x \<in> H"] in force) (simp only: genset_sub)

lemma gen_idemp: "\<langle>\<langle>G\<rangle>\<rangle> = \<langle>G\<rangle>"
  using image_subsetI[of "lists \<langle>G\<rangle>" concat "\<langle>G\<rangle>", unfolded hull_concat_lists[of "\<langle>G\<rangle>", symmetric],
      THEN subset_antisym[OF _ genset_sub[of "\<langle>G\<rangle>"]]] hull_closed_lists[of _ G] by blast

lemma hull_mono'[intro]: "A \<subseteq> \<langle>B\<rangle> \<Longrightarrow> \<langle>A\<rangle> \<subseteq> \<langle>B\<rangle>"
  using hull_mono[of A "\<langle>B\<rangle>"] unfolding gen_idemp.

lemma hull_conjug [elim]: "w \<in> \<langle>{r\<cdot>s,s\<cdot>r}\<rangle> \<Longrightarrow> w \<in> \<langle>{r,s}\<rangle>"
  using hull_mono[of "{r\<cdot>s,s\<cdot>r}" "\<langle>{r,s}\<rangle>", unfolded gen_idemp] by blast

lemma root_divisor: assumes "period w k" and  "k dvd \<^bold>|w\<^bold>|" shows "w \<in> \<langle>{take k w}\<rangle>"
  using pow_sing_gen[of "(\<^bold>|w\<^bold>| div k)" "take k w"] unfolding
    take_several_pers[OF \<open>period w k\<close>, of "\<^bold>|w\<^bold>| div k", unfolded dvd_div_mult_self[OF \<open>k dvd \<^bold>|w\<^bold>|\<close>] take_self, OF , OF order_refl].

text\<open>Intersection of hulls is a hull.\<close>

lemma hulls_inter: "\<langle>\<Inter> {\<langle>G\<rangle> | G. G \<in> S}\<rangle> = \<Inter> {\<langle>G\<rangle> | G. G \<in> S}"
proof
  {fix G assume "G \<in> S"
    hence "\<langle>\<Inter> {\<langle>G\<rangle> |G. G \<in> S}\<rangle> \<subseteq> \<langle>G\<rangle>"
      using Inter_lower[of "\<langle>G\<rangle>" "{\<langle>G\<rangle> |G. G \<in> S}"] mem_Collect_eq[of "\<langle>G\<rangle>" "\<lambda> A. \<exists> G. G \<in> S \<and> A = \<langle>G\<rangle>"]
        hull_mono[of "\<Inter> {\<langle>G\<rangle> |G. G \<in> S}" "\<langle>G\<rangle>"] unfolding gen_idemp by auto}
  thus "\<langle>\<Inter> {\<langle>G\<rangle> |G. G \<in> S}\<rangle> \<subseteq> \<Inter> {\<langle>G\<rangle> |G. G \<in> S}"  by blast
next
  show "\<Inter> {\<langle>G\<rangle> |G. G \<in> S} \<subseteq> \<langle>\<Inter> {\<langle>G\<rangle> |G. G \<in> S}\<rangle>"
    by simp
qed

theorem hull_minimal: "\<langle>G\<rangle> = \<Inter> {S . G \<subseteq> S \<and> \<langle>S\<rangle> = S}"
proof
  show " \<langle>G\<rangle> \<subseteq> \<Inter> {S. G \<subseteq> S \<and> \<langle>S\<rangle> = S}"
    by (rule Inter_greatest, unfold mem_Collect_eq) (use hull_mono'[of G] in blast)
  show " \<Inter> {S. G \<subseteq> S \<and> \<langle>S\<rangle> = S} \<subseteq> \<langle>G\<rangle>"
    by (rule Inter_lower, unfold mem_Collect_eq gen_idemp) (use genset_sub in blast)
qed



lemma all_gen_comm_hull_comm: assumes "\<And> x y. x \<in> G \<Longrightarrow> y \<in> G  \<Longrightarrow> x \<cdot> y = y \<cdot> x"
      "u \<in> \<langle>G\<rangle>" "v \<in> \<langle>G\<rangle>"
    shows "u \<cdot> v = v \<cdot> u"
proof (rule hull.induct[OF \<open>u \<in> \<langle>G\<rangle>\<close>], blast)
  fix w1 w2
  assume "w1 \<in> G" "w2 \<in> \<langle>G\<rangle>" "w2 \<cdot> v = v \<cdot> w2"
  show "(w1 \<cdot> w2) \<cdot> v = v \<cdot> w1 \<cdot> w2"
    unfolding rassoc \<open>w2 \<cdot> v = v \<cdot> w2\<close> unfolding lassoc cancel_right
  proof (rule hull.induct[OF \<open>v \<in> \<langle>G\<rangle>\<close>], blast)
    fix w1' w2'
    assume "w1' \<in> G" "w2' \<in> \<langle>G\<rangle>" "w1 \<cdot> w2' = w2' \<cdot> w1"
    show "w1 \<cdot> w1' \<cdot> w2' = (w1' \<cdot> w2') \<cdot> w1"
      unfolding rassoc \<open>w1 \<cdot> w2' = w2' \<cdot> w1\<close>[symmetric] unfolding lassoc cancel_right
      using \<open>w1' \<in> G\<close> \<open>w1 \<in> G\<close> assms(1) by force
  qed
qed

lemma bin_comm_hull_comm: assumes "x \<cdot> y = y \<cdot> x" "u \<in> \<langle>{x,y}\<rangle>" "v \<in> \<langle>{x,y}\<rangle>"
  shows "u \<cdot> v = v \<cdot> u"
  by (rule all_gen_comm_hull_comm[OF _ assms(2-3)]) (use assms(1) in fastforce)

lemma[reversal_rule]: "rev ` \<langle>{rev u, rev v}\<rangle> = \<langle>{u,v}\<rangle>"
  by (simp add: rev_hull)

lemma[reversal_rule]: "rev w \<in>  \<langle>rev ` G\<rangle> \<equiv> w \<in> \<langle>G\<rangle>"
  unfolding rev_in_conv rev_hull rev_rev_image_eq.


section "Factorization into generators"

text\<open>We define a decomposition (or a factorization) of a into elements of a given generating set. Such a decomposition is well defined only
if the decomposed word is an element of the hull. Even int that case, however, the decomposition need not be unique.\<close>

definition decompose :: "'a list set  \<Rightarrow> 'a list \<Rightarrow> 'a list list" ("Dec _ _" [55,55] 56) where
  "decompose G u = (SOME us. us \<in> lists (G - {\<epsilon>}) \<and> concat us = u)"

lemma dec_ex:  assumes "u \<in> \<langle>G\<rangle>" shows "\<exists> us. (us \<in> lists (G - {\<epsilon>}) \<and> concat us = u)"
  using assms unfolding image_def  hull_concat_lists[of G] mem_Collect_eq
  using del_emp_concat lists_minus' by metis

lemma dec_in_lists': "u \<in> \<langle>G\<rangle> \<Longrightarrow> (Dec G u) \<in> lists (G - {\<epsilon>})"
  unfolding decompose_def using someI_ex[OF dec_ex] by blast

lemma concat_dec[simp, intro] : "u \<in> \<langle>G\<rangle> \<Longrightarrow> concat (Dec G u) = u"
  unfolding decompose_def using someI_ex[OF dec_ex] by blast

lemma dec_emp [simp]: "Dec G \<epsilon> = \<epsilon>"
proof-
  have ex:  "\<epsilon> \<in> lists (G - {\<epsilon>}) \<and> concat \<epsilon> = \<epsilon>"
    by simp
  have all: "(us \<in> lists (G - {\<epsilon>}) \<and> concat us = \<epsilon>) \<Longrightarrow> us = \<epsilon>" for us
    using emp_concat_emp by auto
  show  ?thesis
    unfolding decompose_def
    using all[OF someI[of "\<lambda> x. x \<in> lists (G - {\<epsilon>}) \<and> concat x = \<epsilon>", OF ex]].
qed

lemma dec_nemp: "u \<in> \<langle>G\<rangle> - {\<epsilon>} \<Longrightarrow>  Dec G u \<noteq> \<epsilon>"
  using concat_dec[of u G] by force

lemma dec_nemp'[simp, intro]: "u \<noteq> \<epsilon> \<Longrightarrow> u \<in> \<langle>G\<rangle> \<Longrightarrow> Dec G u \<noteq> \<epsilon>"
  using dec_nemp by blast

lemma dec_eq_emp_iff [simp]: assumes "u \<in> \<langle>G\<rangle>" shows "Dec G u = \<epsilon> \<longleftrightarrow> u = \<epsilon>"
  using dec_nemp'[OF _ \<open>u \<in> \<langle>G\<rangle>\<close>] by auto

lemma dec_in_lists[simp]: "u \<in> \<langle>G\<rangle> \<Longrightarrow> Dec G u \<in> lists G"
  using dec_in_lists' by auto

lemma set_dec_sub: "x \<in> \<langle>G\<rangle> \<Longrightarrow> set (Dec G x) \<subseteq> G"
  using dec_in_lists by blast

lemma dec_hd: "u \<noteq> \<epsilon> \<Longrightarrow> u \<in> \<langle>G\<rangle> \<Longrightarrow> hd (Dec G u) \<in> G"
  by simp

lemma non_gen_dec: assumes  "u \<in> \<langle>G\<rangle>" "u \<notin> G" shows "Dec G u  \<noteq> [u]"
  using dec_in_lists[OF \<open>u \<in> \<langle>G\<rangle>\<close>] Cons_in_lists_iff[of u \<epsilon> G] \<open>u \<notin> G\<close> by argo

lemma fac_fac_interpE_dec: assumes  "w \<in> \<langle>G\<rangle>"  "u \<le>f w" "u \<noteq> \<epsilon>"
  obtains p s ws where "ws \<in> lists (G - {\<epsilon>})" "p u s \<sim>\<^sub>\<I> ws" "ws \<le>f Dec G w"
proof-
  from fac_fac_interpE'[OF  _  \<open>u \<noteq> \<epsilon>\<close>, of "Dec G w", unfolded concat_dec[OF \<open>w \<in> \<langle>G\<rangle>\<close>], OF \<open>u \<le>f w\<close>]
  obtain p s ws where *: "p u s \<sim>\<^sub>\<I> ws" "ws \<le>f Dec G w".
  have "ws \<in> lists (G - {\<epsilon>})"
    using  fac_in_lists[OF dec_in_lists'[OF \<open>w \<in> \<langle>G\<rangle>\<close>] \<open>ws \<le>f Dec G w\<close>].
  from that[OF this *]
  show thesis.
qed

lemma set_len_mod_concat: assumes "\<forall> x \<in> set us. \<^bold>|x\<^bold>| mod n = 0" "0 < n"
  shows "\<^bold>|concat us\<^bold>| mod n = 0"
  using \<open>\<forall> x \<in> set us. \<^bold>|x\<^bold>| mod n = 0\<close>
  by (induct us, force, unfold concat_simps lenmorph, force)

lemma hull_len_mod_concat: assumes "\<forall> x \<in> G. \<^bold>|x\<^bold>| mod n = 0" "0 < n" "w \<in> \<langle>G\<rangle>"
  shows "\<^bold>|w\<^bold>| mod n = 0"
  by (rule set_len_mod_concat[of "Dec G w", OF _ \<open>0 < n\<close>, unfolded concat_dec[OF \<open>w \<in> \<langle>G\<rangle>\<close>]] concat_dec[OF \<open>w \<in> \<langle>G\<rangle>\<close>])
  (use \<open>\<forall> x \<in> G. \<^bold>|x\<^bold>| mod n = 0\<close> \<open>w \<in> \<langle>G\<rangle>\<close> set_dec_sub in blast)

subsection \<open>Refinement into a specific decomposition\<close>

text\<open>We extend the decomposition to lists of words. This can be seen as a refinement of a previous decomposition of some word.\<close>

definition refine :: "'a list set \<Rightarrow> 'a list list \<Rightarrow> 'a list list" ("Ref _ _" [55,56] 56) where
  "refine G us = concat(map (decompose G) us)"

lemma ref_morph: "us \<in> lists \<langle>G\<rangle> \<Longrightarrow> vs \<in> lists \<langle>G\<rangle> \<Longrightarrow> refine G (us \<cdot> vs) = refine G us \<cdot> refine G vs"
  unfolding refine_def by simp

lemma ref_morph_plus: "us \<in> lists (\<langle>G\<rangle> - {\<epsilon>}) \<Longrightarrow> vs \<in> lists (\<langle>G\<rangle> - {\<epsilon>}) \<Longrightarrow> refine G (us \<cdot> vs) = refine G us \<cdot> refine G vs"
  unfolding refine_def by simp

lemma ref_pref_mono: "ws \<in> lists \<langle>G\<rangle> \<Longrightarrow> us \<le>p ws \<Longrightarrow> Ref G us \<le>p Ref G ws"
  unfolding prefix_def using ref_morph append_in_lists_dest' append_in_lists_dest by metis

lemma ref_suf_mono: "ws \<in> lists \<langle>G\<rangle> \<Longrightarrow> us \<le>s ws \<Longrightarrow> (Ref G us) \<le>s Ref G ws"
  unfolding suffix_def using ref_morph append_in_lists_dest' append_in_lists_dest by metis

lemma ref_fac_mono: "ws \<in> lists \<langle>G\<rangle> \<Longrightarrow> us \<le>f ws \<Longrightarrow> (Ref G us) \<le>f Ref G ws"
  unfolding sublist_altdef' using ref_pref_mono ref_suf_mono  suf_in_lists by metis

lemma ref_pop_hd: "us \<noteq> \<epsilon> \<Longrightarrow> us \<in> lists \<langle>G\<rangle> \<Longrightarrow> refine G us = decompose G (hd us) \<cdot> refine G (tl us)"
  unfolding  refine_def  using list.simps(9)[of "decompose G" "hd us" "tl us"] by simp

lemma ref_in: "us \<in> lists \<langle>G\<rangle> \<Longrightarrow> (Ref G us) \<in> lists (G - {\<epsilon>})"
proof (induction us)
  case (Cons a us)
  then show ?case
    using Cons.IH Cons.prems dec_in_lists' by (auto simp add: refine_def)
qed (simp add: refine_def)

lemma ref_in'[intro]: "us \<in> lists \<langle>G\<rangle> \<Longrightarrow> (Ref G us) \<in> lists G"
  using ref_in by auto

lemma concat_ref: "us \<in> lists \<langle>G\<rangle> \<Longrightarrow> concat (Ref G us) = concat us"
proof (induction us)
  case (Cons a us)
  then show ?case
    using Cons.IH Cons.prems concat_dec refine_def by (auto simp add: refine_def)
qed (simp add: refine_def)

lemma ref_gen: "us \<in> lists B \<Longrightarrow> B \<subseteq> \<langle>G\<rangle> \<Longrightarrow> Ref G us \<in> \<langle>decompose G ` B\<rangle>"
  by (induct us, auto simp add: refine_def)

lemma ref_set: "ws \<in> lists \<langle>G\<rangle> \<Longrightarrow> set (Ref G ws) = \<Union> (set`(decompose G)`set ws)"
  by (simp add: refine_def)

lemma emp_ref: assumes "us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})" and  "Ref G us = \<epsilon>" shows "us = \<epsilon>"
  using emp_concat_emp[OF \<open>us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})\<close>]
    concat_ref [OF lists_minus[OF assms(1)], unfolded \<open>Ref G us = \<epsilon>\<close> concat.simps(1),symmetric] by blast

lemma sing_ref_sing:
  assumes "us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})" and "refine G us = [b]"
  shows "us = [b]"
proof-
  have "us \<noteq> \<epsilon>"
    using \<open>refine G us = [b]\<close> by (auto simp add: refine_def)
  have "tl us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})" and "hd us \<in> \<langle>G\<rangle> - {\<epsilon>}"
    using list.collapse[OF \<open>us \<noteq> \<epsilon>\<close>]  \<open>us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})\<close> Cons_in_lists_iff[of "hd us" "tl us" "\<langle>G\<rangle> - {\<epsilon>}"]
    by auto
  have "Dec G (hd us) \<noteq> \<epsilon>"
    using dec_nemp[OF \<open>hd us \<in> \<langle>G\<rangle> - {\<epsilon>}\<close>].
  have "us \<in> lists \<langle>G\<rangle>"
    using \<open>us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})\<close> lists_minus by auto
  have "concat us = b"
    using \<open>us \<in> lists \<langle>G\<rangle>\<close> assms(2) concat_ref by force
  have "refine G (tl us) = \<epsilon>"
    using ref_pop_hd[OF \<open>us \<noteq> \<epsilon>\<close> \<open>us \<in> lists \<langle>G\<rangle>\<close>]  unfolding  \<open>refine G us = [b]\<close>
    using \<open>Dec G (hd us) \<noteq> \<epsilon>\<close> Cons_eq_append_conv[of b \<epsilon> "(Dec G (hd us))" "(Ref G (tl us))"]
      Cons_eq_append_conv[of b \<epsilon> "(Dec G (hd us))" "(Ref G (tl us))"]  append_is_Nil_conv[of _ "(Ref G (tl us))"]
    by blast
  from  emp_ref[OF \<open>tl us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})\<close> this, symmetric]
  have "\<epsilon> = tl us".
  from this[unfolded Nil_tl]
  show ?thesis
    using \<open>us \<noteq> \<epsilon>\<close> \<open>concat us = b\<close> by auto
qed

lemma ref_ex: assumes "Q \<subseteq> \<langle>G\<rangle>" and "us \<in> lists Q"
  shows "Ref G us \<in> lists (G - {\<epsilon>})" and "concat (Ref G us) = concat us"
  using ref_in[OF sub_lists_mono[OF assms]] concat_ref[OF sub_lists_mono[OF assms]].

lemma ref_emp [simp]: "Ref G \<epsilon> = \<epsilon>"
  unfolding refine_def by force

section "Basis"

text\<open>An important property of monoids of words is that they have a unique minimal generating set. Which is the set consisting of indecomposable elements.\<close>

text\<open>Indecomposable element is an element that is not generated by other ones.\<close>

definition ungenerated :: "'a list \<Rightarrow> 'a list set  \<Rightarrow> bool" (" _ \<in>U _ " [51,51] 50) where
  "ungenerated b G \<equiv> b \<in> G \<and> b \<notin> \<langle>G - {b}\<rangle>"

lemma ungen_nemp[simp]: "b \<in>U G \<Longrightarrow> b \<noteq> \<epsilon>"
  unfolding ungenerated_def by blast

lemma ungen_in[intro]: "ungenerated b G \<Longrightarrow> b \<in> G" and
      ungenD[intro]: "ungenerated b G \<Longrightarrow> b \<notin> \<langle>G - {b}\<rangle>"
  unfolding ungenerated_def by blast+

text \<open>An ungenerated element has no nontrivial decomposition\<close>

lemma ungen_dec_triv: assumes "u \<in> \<langle>G\<rangle>" "v \<in> \<langle>G\<rangle>" "u \<cdot> v \<in>U \<langle>G\<rangle>" shows " u = \<epsilon> \<or> v = \<epsilon>"
proof (rule ccontr)
  assume "\<not> (u = \<epsilon> \<or> v = \<epsilon>)"
  hence "u \<cdot> v \<noteq> u" "u \<cdot> v \<noteq> v"
    by blast+
  hence "u \<in> \<langle>\<langle>G\<rangle> - {u \<cdot> v}\<rangle>" "v \<in> \<langle>\<langle>G\<rangle> - {u \<cdot> v}\<rangle>"
    using assms(1-2) by blast+
  from hull_closed[OF this]
  show False
    using \<open>u \<cdot> v \<in>U \<langle>G\<rangle>\<close> unfolding ungenerated_def by blast
qed

lemma ungen_dec_triv': assumes "us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})" "concat us \<in>U \<langle>G\<rangle>" shows "\<^bold>|us\<^bold>| = 1"
proof-
  have "us \<noteq> \<epsilon>"
    using \<open>concat us \<in>U \<langle>G\<rangle>\<close> ungen_nemp by force
  hence "hd us \<noteq> \<epsilon>" "hd us \<in> \<langle>G\<rangle>"
    using  \<open>us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})\<close> by force+
  have "concat (tl us) \<in> \<langle>G\<rangle>"
    using  concat_in_hull'[OF tl_in_lists, OF \<open>us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})\<close>, unfolded gen_idemp]
    by (simp add: gen_idemp)
  have "hd us \<cdot>  concat (tl us) \<in>U \<langle>G\<rangle>"
    using \<open>concat us \<in>U \<langle>G\<rangle>\<close>  by (subst (asm) (1) hd_tl[OF \<open>us \<noteq> \<epsilon>\<close>, symmetric], simp)
  from ungen_dec_triv[of "hd us" _ "concat (tl us)", OF \<open>hd us \<in> \<langle>G\<rangle>\<close> \<open>concat (tl us) \<in> \<langle>G\<rangle>\<close>  this]
  have "tl us = \<epsilon>"
    using tl_in_lists[OF \<open>us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})\<close>] \<open>hd us \<noteq> \<epsilon>\<close> by force
  then show "\<^bold>|us\<^bold>| = 1"
    using nemp_le_len[OF \<open>us \<noteq> \<epsilon>\<close>] long_list_tl by force
qed

text \<open>Conversely, any nonempty  element that is not ungenerated is a product of at least two shorter elements\<close>

lemma gen_elem_list: assumes "u \<in> \<langle>G - {u}\<rangle>" "u \<noteq> \<epsilon>"
  obtains us where "us \<in> lists (G - {u} - {\<epsilon>})" "concat us = u" "1 < \<^bold>|us\<^bold>|"
    "\<And> c. c \<in> set us \<Longrightarrow> \<^bold>|c\<^bold>| < \<^bold>|u\<^bold>|"
proof-
  from  hull_concat_lists_nempE[OF \<open>u \<in> \<langle>G - {u}\<rangle>\<close>]
  obtain ws where cond:  "ws \<in> lists (G - {u} - {\<epsilon>})" "concat ws = u".
  have "1 < \<^bold>|ws\<^bold>|"
  proof (rule ccontr)
    assume "\<not> 1 < \<^bold>|ws\<^bold>| "
    hence "\<^bold>|ws\<^bold>| = 1"
      using nemp_len[of ws] \<open>u \<noteq> \<epsilon>\<close>[folded \<open>concat ws = u\<close>]
      by (metis concat_nemp less_one linorder_neqE_nat nemp_len_not0)
    hence "ws = [u]"
      using \<open>concat ws = u\<close> sing_word_concat by fastforce
    thus False
      using \<open>ws \<in> lists (G - {u} - {\<epsilon>})\<close> by force
  qed
  have "\<^bold>|c\<^bold>| < \<^bold>|u\<^bold>|" if "c \<in> set ws" for c
  proof-
    obtain ws1 ws2 where "ws = ws1 \<cdot> [c] \<cdot> ws2"
      using \<open>c \<in> set ws\<close> split_listE by meson
    hence ws_lists: "ws1 \<in> lists (G - {u} - {\<epsilon>})" "ws2 \<in> lists (G - {u} - {\<epsilon>})"
      using \<open>ws \<in> lists (G - {u} - {\<epsilon>})\<close> by simp_all
    have "ws1 \<cdot>  ws2 \<noteq> \<epsilon>"
      using \<open>1 < \<^bold>|ws\<^bold>|\<close>[unfolded \<open>ws = ws1 \<cdot> [c] \<cdot> ws2\<close>] by force
    hence "\<^bold>|concat ws1\<^bold>| + \<^bold>|concat ws2\<^bold>| \<noteq> 0"
      using ws_lists by force
    thus "\<^bold>|c\<^bold>| < \<^bold>|u\<^bold>|"
      using lenarg[OF arg_cong[OF \<open>ws = ws1 \<cdot> [c] \<cdot> ws2\<close>, of concat]]
      unfolding concat_morph lenmorph concat_sing' \<open>concat ws = u\<close> by linarith
  qed
  from that[OF cond \<open>1 < \<^bold>|ws\<^bold>|\<close> this]
  show thesis.
qed

lemma gen_elem_dec: assumes "b \<in> \<langle>G - {b}\<rangle>" "b \<noteq> \<epsilon>"
  obtains u v where "u \<in> \<langle>G\<rangle>" "u \<noteq> \<epsilon>" "v \<in> \<langle>G\<rangle>" "v \<noteq> \<epsilon>" "u \<cdot> v = b"
proof-
  from gen_elem_list[OF assms]
  obtain us where "us \<in> lists (G - {b} - {\<epsilon>})" "concat us = b" "1 < \<^bold>|us\<^bold>|".
  have "us \<noteq> \<epsilon>" "tl us \<noteq> \<epsilon>" "[hd us] \<noteq> \<epsilon>"
    using long_list_tl[OF \<open>1 < \<^bold>|us\<^bold>|\<close>] by fastforce+
  note b =  arg_cong[OF hd_tl[OF \<open>us \<noteq> \<epsilon>\<close>], of concat, unfolded \<open>concat us = b\<close> concat_morph]
  have "concat (tl us) \<noteq> \<epsilon>"
    using \<open>tl us \<noteq> \<epsilon>\<close> \<open>us \<in> lists (G - {b} - {\<epsilon>})\<close> emp_concat_emp tl_in_lists by meson
  have "concat (tl us) \<in> \<langle>G\<rangle>"
    using \<open>us \<in> lists (G - {b} - {\<epsilon>})\<close> concat_in_hull' lists_minus tl_in_lists by meson
  have "concat [hd us] \<noteq> \<epsilon>"
    using \<open>us \<in> lists (G - {b} - {\<epsilon>})\<close> \<open>us \<noteq> \<epsilon>\<close> by fastforce
  have "concat [hd us] \<in> \<langle>G\<rangle>"
    using \<open>us \<in> lists (G - {b} - {\<epsilon>})\<close> Diff_iff \<open>us \<noteq> \<epsilon>\<close> concat_sing' gen_in lists_hd_in_set by metis
  show thesis
    by (rule that[OF _ _ _ _ b]) fact+
qed

text \<open>This yields several criteria for being ungenerated\<close>

lemma ungeneratedI:
  assumes "b \<in> G" and "b \<noteq> \<epsilon>"
  and all: "\<And> u v. u \<noteq> \<epsilon> \<Longrightarrow>  u \<in> \<langle>G\<rangle> \<Longrightarrow>  v \<noteq> \<epsilon> \<Longrightarrow> v \<in> \<langle>G\<rangle> \<Longrightarrow> u \<cdot> v \<noteq> b"
  shows "b \<in>U G"
  unfolding ungenerated_def
proof
  show "b \<notin> \<langle>G - {b}\<rangle>"
  proof
    assume "b \<in> \<langle>G - {b}\<rangle>"
    from gen_elem_dec[OF this \<open>b \<noteq> \<epsilon>\<close>]
    show False
      using all by metis
  qed
qed fact

lemma ungeneratedI':
  assumes "b \<in> G" and "b \<noteq> \<epsilon>"
  and all: "\<And> us. us \<in> lists (G - {b} - {\<epsilon>})  \<Longrightarrow> concat us = b  \<Longrightarrow> \<^bold>|us\<^bold>| \<le> 1"
  shows "b \<in>U G"
  unfolding ungenerated_def
proof
  show "b \<notin> \<langle>G - {b}\<rangle>"
  proof
    assume "b \<in> \<langle>G - {b}\<rangle>"
    from gen_elem_list[OF this \<open>b \<noteq> \<epsilon>\<close>]
    show False
      using all le_antisym nless_le by metis
  qed
qed fact

lemma ungenerated_shortest:
  assumes "b \<in> G" and "b \<noteq> \<epsilon>"
  and all: "\<And> c. c \<in> G - {\<epsilon>}  \<Longrightarrow> \<^bold>|b\<^bold>| \<le> \<^bold>|c\<^bold>|"
  shows "b \<in>U G"
  unfolding ungenerated_def
proof
  show "b \<notin> \<langle>G - {b}\<rangle>"
  proof
    assume "b \<in> \<langle>G - {b}\<rangle>"
    from gen_elem_list[OF this \<open>b \<noteq> \<epsilon>\<close>]
    obtain us where "us \<in> lists (G - {b} - {\<epsilon>})" "concat us = b" and all': "(\<And>c. c \<in> set us \<Longrightarrow> \<^bold>|c\<^bold>| < \<^bold>|b\<^bold>|)"
      by metis
    have "c \<in> set us \<Longrightarrow> \<^bold>|b\<^bold>| \<le> \<^bold>|c\<^bold>|" for c
      using all \<open>us \<in> lists (G - {b} - {\<epsilon>})\<close> by blast
    then show False
      using all' \<open>b \<noteq> \<epsilon>\<close> \<open>concat us = b\<close> unfolding linorder_not_less[symmetric]
      by fastforce
  qed
qed fact

lemma ungenerated_sing:
  assumes "[a] \<in> G"
  shows "[a] \<in>U G"
  using ungenerated_shortest[OF assms] nemp_le_len unfolding sing_len
   by blast

\<comment> \<open>Ungeneretad element is ungenerated by the whole monoid\<close>

lemma ungen_hull_ungen:  "b \<in>U \<langle>G\<rangle> \<longleftrightarrow> b \<in>U G"
proof
  assume "b \<in>U G"
  show " b \<in>U \<langle>G\<rangle>"
    unfolding ungenerated_def
  proof
    show "b \<notin> \<langle>\<langle>G\<rangle> - {b}\<rangle>"
    proof
      assume "b \<in> \<langle>\<langle>G\<rangle> - {b}\<rangle>"
    from gen_elem_list[OF this ungen_nemp[OF \<open>b \<in>U G\<close>]]
    obtain us where "us \<in> lists (\<langle>G\<rangle> - {b} - {\<epsilon>})" and "concat us = b"
      and "1 < \<^bold>|us\<^bold>|" and short: "\<And>c. c \<in> set us \<Longrightarrow> \<^bold>|c\<^bold>| < \<^bold>|b\<^bold>|"
      by blast
    have no_b: "b \<notin> set (Dec G u)" if  "u \<in> set us" for u
    proof
      assume "b \<in> set (Dec G u)"
      have "concat (Dec G u) = u"
        using \<open>us \<in> lists (\<langle>G\<rangle> - {b} - {\<epsilon>})\<close> \<open>u \<in> set us\<close> by blast
      hence "\<^bold>|b\<^bold>| \<le> \<^bold>|u\<^bold>|"
        using split_list_first[OF \<open>b \<in> set (Dec G u)\<close>] by force
      thus False
        using short[OF \<open>u \<in> set us\<close>] by force
    qed
    define vs where "vs = refine G us"
    have  "concat vs = b" "vs \<in> lists (G - {\<epsilon>})"
      using ref_ex[OF _ \<open>us \<in> lists (\<langle>G\<rangle> - {b} - {\<epsilon>})\<close>, of G]
      unfolding vs_def \<open>concat us = b\<close> by blast+
    have "b \<notin> set vs"
      using no_b unfolding vs_def refine_def by simp
    hence "vs \<in> lists (G - {b})"
      using \<open>vs \<in> lists (G - {\<epsilon>})\<close> by blast
    with ungenD[OF \<open>b \<in>U G\<close>]
    show False
      unfolding ungenerated_def using \<open>concat vs = b\<close> by blast
  qed
  show "b \<in> \<langle>G\<rangle>"
    using ungen_in[OF \<open>b \<in>U G\<close>] by blast
qed
next
  assume "b \<in>U \<langle>G\<rangle> "
  show "b \<in>U G"
    unfolding ungenerated_def
  proof
    have "G - {b} \<subseteq> \<langle>G\<rangle> - {b}"
      by blast
    from hull_mono[OF this]
    show "b \<notin> \<langle>G - {b}\<rangle>"
      using ungenD[OF \<open>b \<in>U \<langle>G\<rangle>\<close>] by blast
    then show "b \<in> G"
      using ungen_in[OF \<open>b \<in>U \<langle>G\<rangle>\<close>] by auto
  qed
qed

text\<open>The \emph{basis} is the set of all ungenerated elements.\<close>

definition basis :: "'a list set  \<Rightarrow> 'a list set" ("\<BB> _" [51] ) where
  "basis G = {x. x \<in>U G}"

lemma basisD: "x \<in> \<BB> G \<Longrightarrow> x \<in>U G"
  unfolding basis_def by simp

lemma basis_in: "x \<in> \<BB> G \<Longrightarrow> x \<in> G"
  unfolding basis_def ungenerated_def by simp

lemma emp_not_basis: "x \<in> \<BB> G \<Longrightarrow> x \<noteq> \<epsilon>"
  unfolding basis_def ungenerated_def by blast

lemma basis_sub_gen: "\<BB> G \<subseteq> G"
  unfolding  basis_def ungenerated_def by simp

text\<open>The basis generates the generating set and therefore also the whole monoid\<close>

lemma gen_sub_basis: "G \<subseteq> \<langle>\<BB> G\<rangle>"
proof
  fix w show "w \<in> G \<Longrightarrow> w \<in> \<langle>\<BB> G\<rangle>"
  proof (induct "length w" arbitrary: w rule: less_induct)
    case less
    show ?case
    proof (cases "w \<in> \<BB> G \<or> w = \<epsilon>", blast)
      assume "\<not> (w \<in> \<BB> G \<or> w = \<epsilon>)"
      hence "w \<in> \<langle>G - {w}\<rangle>" "w \<noteq> \<epsilon>"
        using \<open>w \<in> G\<close> unfolding basis_def ungenerated_def  by blast+
      from gen_elem_list[OF this[unfolded ungenerated_def]]
      obtain us where "us \<in> lists (G - {w} - {\<epsilon>})" "concat us = w" "1 < \<^bold>|us\<^bold>|" and
        small: "(\<And>c. c \<in> set us \<Longrightarrow> \<^bold>|c\<^bold>| < \<^bold>|w\<^bold>|)"
        by blast
      have "c \<in> \<langle>\<BB> G\<rangle>" if "c \<in> set us" for c
        using  less.hyps[of c, OF small, OF that] \<open>c \<in> set us\<close> \<open>us \<in> lists (G - {w} - {\<epsilon>})\<close> by blast
      thus "w \<in> \<langle>\<BB> G\<rangle> "
        unfolding \<open>concat us = w\<close>[symmetric]
        using hull_closed_lists[OF in_listsI] by blast
    qed
  qed
qed

lemma basis_concat_listsE:
  assumes "w \<in> G"
  obtains ws where "ws \<in> lists \<BB> G"  and "concat ws = w"
  using assms by (rule hull_concat_listsE[OF subsetD, OF  gen_sub_basis])

theorem basis_gen_hull: "\<langle>\<BB> G\<rangle> = \<langle>G\<rangle>"
  by (rule equalityI; simp only: hull_mono[OF basis_sub_gen] hull_mono[OF gen_sub_basis, unfolded gen_idemp])

theorem basis_of_hull: "\<BB> \<langle>G\<rangle> = \<BB> G"
unfolding basis_def ungen_hull_ungen..

lemma basis_gen_hull': "\<langle>\<BB> \<langle>G\<rangle>\<rangle> = \<langle>G\<rangle>"
  unfolding basis_of_hull using basis_gen_hull.

lemma basis_hull_sub: "\<BB> \<langle>G\<rangle> \<subseteq> G"
  unfolding basis_of_hull using basis_sub_gen.

text\<open>The basis is the smallest generating set.\<close>
theorem gen_basis_sub:  "\<langle>S\<rangle> = \<langle>G\<rangle> \<Longrightarrow> \<BB> G \<subseteq> S"
  using basis_hull_sub[of S] basis_of_hull[of G] by simp

lemma basis_min_gen: "S \<subseteq> \<BB> G \<Longrightarrow> \<langle>S\<rangle> = G \<Longrightarrow> S = \<BB> G"
  using basis_of_hull basis_sub_gen by blast

lemma basisI: "(\<And> B. \<langle>B\<rangle> = \<langle>C\<rangle> \<Longrightarrow> C \<subseteq> B) \<Longrightarrow> \<BB> \<langle>C\<rangle> = C"
  using basis_gen_hull basis_min_gen basis_of_hull by metis

text\<open>An arbitrary set between basis and the hull is generating...\<close>
lemma gen_sets: assumes "\<BB> G \<subseteq> S" and "S \<subseteq> \<langle>G\<rangle>" shows "\<langle>S\<rangle> = \<langle>G\<rangle>"
  using hull_mono[OF \<open>S \<subseteq> \<langle>G\<rangle>\<close>, unfolded gen_idemp] hull_mono[OF \<open>\<BB> G \<subseteq> S\<close>, unfolded basis_gen_hull] by blast

text\<open>... and has the same basis\<close>
lemma basis_sets: "\<BB> G \<subseteq> S \<Longrightarrow> S \<subseteq> \<langle>G\<rangle> \<Longrightarrow> \<BB> G = \<BB> S"
  by (metis basis_of_hull  gen_sets)

section "Code"

locale nemp_words =
  fixes G
  assumes emp_not_in: "\<epsilon> \<notin> G"

begin
lemma drop_empD: "G - {\<epsilon>} = G"
  using emp_not_in by simp

lemmas emp_concat_emp' = emp_concat_emp[of _ G, unfolded drop_empD]

thm disjE[OF ruler[OF take_is_prefix take_is_prefix]]


lemma nemp: "x \<in> G \<Longrightarrow> x \<noteq> \<epsilon>"
  using emp_not_in by blast

lemma concat_eq_emp_conv [simp]: "us \<in> lists G \<Longrightarrow> concat us = \<epsilon> \<longleftrightarrow> us = \<epsilon>"
  unfolding in_lists_conv_set concat_eq_Nil_conv
  by (simp add: nemp)

lemma root_dec_inj_on: "inj_on (\<lambda> x. [\<rho> x]\<^sup>@(e\<^sub>\<rho> x)) G"
  unfolding inj_on_def using primroot_exp_eq
  unfolding concat_pow_list_single[of _ "\<rho> _", symmetric] by metis

lemma concat_len_ruler: assumes "ws \<in> lists G" "us \<le>p ws" "vs \<le>p ws" "\<^bold>|concat us\<^bold>| \<le> \<^bold>|concat vs\<^bold>|"
  shows "us \<le>p vs"
proof (rule ccontr)
  assume "\<not> us \<le>p vs"
  with ruler[OF assms(2-3)]
  have "vs <p us"
    by blast
  from arg_cong[OF lq_spref[OF this], of "\<lambda> x. \<^bold>|concat x\<^bold>|", unfolded concat_morph lenmorph]
  have "\<^bold>|concat (vs\<inverse>\<^sup>>us)\<^bold>| = 0"
    using assms(4) by linarith
  hence "vs\<inverse>\<^sup>>us = \<epsilon>"
    using emp_concat_emp'[OF lq_in_lists[OF pref_in_lists[OF \<open>us \<le>p ws\<close> \<open>ws \<in> lists G\<close>]]] by blast
  thus False
    using lq_spref_nemp[OF \<open>vs <p us\<close>] by contradiction
qed


end

lemma concat_emp:
   "\<epsilon> \<notin> G \<Longrightarrow> us \<in> lists G \<Longrightarrow> concat us = \<epsilon> \<Longrightarrow> us = \<epsilon>"
 using nemp_words.concat_eq_emp_conv[OF nemp_words.intro] by blast

text\<open>A basis freely generating its hull is called a \emph{code}. By definition,
this means that generated elements have unique factorizations into the elements of the code.\<close>

locale code =
  fixes \<C>
  assumes is_code: "xs \<in> lists \<C> \<Longrightarrow> ys \<in> lists \<C> \<Longrightarrow> concat xs = concat ys \<Longrightarrow> xs = ys"
begin

lemma code_comm_eq: "x \<in> \<C> \<Longrightarrow> y \<in> \<C> \<Longrightarrow> x \<cdot> y = y \<cdot> x \<Longrightarrow> x = y"
  using is_code[of "[x,y]" "[y,x]", THEN arg_cong[of _ _ hd]] by simp

lemma emp_not_in: "\<epsilon> \<notin> \<C>"
proof
  assume "\<epsilon> \<in> \<C>"
  hence "[] \<in> lists \<C>" and "[\<epsilon>] \<in> lists \<C>" and "concat [] = concat [\<epsilon>]" and "[] \<noteq> [\<epsilon>]"
    by simp+
  thus False
    using is_code by blast
qed

lemma nemp: "u \<in> \<C> \<Longrightarrow> u \<noteq> \<epsilon>"
  using emp_not_in by force

sublocale nemp_words \<C>
  using emp_not_in by unfold_locales

lemma code_elem_dec: "us \<in> lists \<C> \<Longrightarrow> concat us = c \<Longrightarrow> c \<in> \<C> \<Longrightarrow> us = [c]"
  using is_code[of us "[c]"] by simp

lemma code_ungen: assumes "c \<in> \<C>" shows "c \<in>U \<C>"
  unfolding ungenerated_def
proof
  show "c \<notin> \<langle>\<C> - {c}\<rangle>"
  proof
    assume \<open>c \<in> \<langle>\<C> - {c}\<rangle>\<close>
    from gen_elem_list[OF this nemp[OF \<open>c \<in> \<C>\<close>]]
    obtain us where "us \<in> lists (\<C> - {c} - {\<epsilon>})" "concat us = c" "1 < \<^bold>|us\<^bold>|".
    show False
      using code_elem_dec[OF _ \<open>concat us = c\<close> \<open>c \<in> \<C>\<close>]
        \<open>1 < \<^bold>|us\<^bold>|\<close> sing_len[of c] \<open>us \<in> lists (\<C> - {c} - {\<epsilon>})\<close> by fastforce
  qed
qed fact

lemma code_is_basis: "\<BB> \<C> = \<C>"
  using code_ungen basis_def[of \<C>] basis_sub_gen by blast

lemma code_unique_dec': "us \<in> lists \<C> \<Longrightarrow> Dec \<C> (concat us) = us"
  using dec_in_lists[of "concat us" \<C>, THEN is_code, of us]
    concat_dec[of "concat us" \<C>] hull_concat_lists[of \<C>] image_eqI[of "concat us" concat us "lists \<C>"]
  by argo

lemma code_unique_dec [intro!]: "us \<in> lists \<C> \<Longrightarrow> concat us = u \<Longrightarrow>  Dec \<C> u = us"
  using code_unique_dec' by blast

lemma triv_refine[intro!] : "us \<in> lists \<C> \<Longrightarrow> concat us = u \<Longrightarrow>  Ref \<C> [u] = us"
  using code_unique_dec' by (auto simp add: refine_def)

lemma code_unique_ref: "us \<in> lists \<langle>\<C>\<rangle> \<Longrightarrow> refine \<C> us = decompose \<C> (concat us)"
proof-
  assume "us \<in> lists \<langle>\<C>\<rangle>"
  hence "concat (refine \<C> us) = concat us"
    using concat_ref by blast
  hence eq: "concat (refine \<C> us) = concat (decompose \<C> (concat us))"
    using  concat_dec[OF hull_closed_lists[OF \<open>us \<in> lists \<langle>\<C>\<rangle>\<close>]] by auto
  have dec: "Dec \<C> (concat us) \<in> lists \<C>"
    using \<open>us \<in> lists \<langle>\<C>\<rangle>\<close> dec_in_lists hull_closed_lists
    by metis
  have "Ref \<C> us \<in> lists \<C>"
    using lists_minus[OF ref_in[OF \<open>us \<in> lists \<langle>\<C>\<rangle>\<close>]].
  from  is_code[OF this dec eq]
  show ?thesis.
qed

lemma refI [intro]: "us \<in> lists \<langle>\<C>\<rangle> \<Longrightarrow> vs \<in> lists \<C> \<Longrightarrow> concat vs = concat us \<Longrightarrow> Ref \<C> us = vs"
  unfolding code_unique_ref code_unique_dec..

lemma code_dec_morph: assumes "x \<in> \<langle>\<C>\<rangle>" "y \<in> \<langle>\<C>\<rangle>"
  shows "(Dec \<C> x) \<cdot> (Dec \<C> y) = Dec \<C> (x\<cdot>y)"
proof-
  have eq: "(Dec \<C> x) \<cdot> (Dec \<C> y) = Dec \<C> (concat ((Dec \<C> x) \<cdot> (Dec \<C> y)))"
    using dec_in_lists[OF \<open>x \<in> \<langle>\<C>\<rangle>\<close>] dec_in_lists[OF \<open>y \<in> \<langle>\<C>\<rangle>\<close>]
      code.code_unique_dec[OF code_axioms, of "(Dec \<C> x) \<cdot> (Dec \<C> y)", unfolded append_in_lists_conv, symmetric]
    by presburger
  moreover have "concat ((Dec \<C> x) \<cdot> (Dec \<C> y)) = (x \<cdot> y)"
    using concat_morph[of "Dec \<C> x" "Dec \<C> y"]
    unfolding concat_dec[OF \<open>x \<in> \<langle>\<C>\<rangle>\<close>] concat_dec[OF \<open>y \<in> \<langle>\<C>\<rangle>\<close>].
  ultimately show "(Dec \<C> x) \<cdot> (Dec \<C> y) = Dec \<C> (x\<cdot>y)"
    by argo
qed

lemma dec_pow: "rs \<in> \<langle>\<C>\<rangle> \<Longrightarrow> Dec \<C> (rs\<^sup>@k) = (Dec \<C> rs)\<^sup>@k"
proof(induction k arbitrary: rs, fastforce)
  case (Suc k)
  then show ?case
    using code_dec_morph pow_Suc power_in by metis
qed

lemma code_el_dec: "c \<in> \<C> \<Longrightarrow> decompose \<C> c = [c]"
  by fastforce

lemma code_ref_list: "us \<in> lists \<C> \<Longrightarrow> refine \<C> us = us"
proof (induct us)
  case (Cons a us)
  then show ?case using code_el_dec
    unfolding refine_def by simp
qed (simp add: refine_def)

lemma code_ref_gen: assumes "G \<subseteq> \<langle>\<C>\<rangle>" "u \<in> \<langle>G\<rangle>"
  shows "Dec \<C> u \<in> \<langle>decompose \<C> ` G\<rangle>"
proof-
  have "refine \<C> (Dec G u) = Dec \<C> u"
    using  dec_in_lists[OF \<open>u \<in> \<langle>G\<rangle>\<close>]  \<open>G \<subseteq> \<langle>\<C>\<rangle>\<close> code_unique_ref[of "Dec G u", unfolded concat_dec[OF \<open>u \<in> \<langle>G\<rangle>\<close>]] by blast
  from ref_gen[of "Dec G u" G, OF dec_in_lists[OF \<open>u \<in> \<langle>G\<rangle>\<close>], of \<C>, unfolded this, OF \<open>G \<subseteq> \<langle>\<C>\<rangle>\<close>]
  show ?thesis.
qed

find_theorems "\<rho> ?x \<^sup>@ ?k = ?x" "0 < ?k"

lemma code_rev_code: "code (rev ` \<C>)"
proof
  fix xs ys assume "xs \<in> lists (rev ` \<C>)" "ys \<in> lists (rev ` \<C>)" "concat xs = concat ys"
  have "map rev (rev xs) \<in> lists \<C>" and "map rev (rev ys) \<in> lists \<C>"
    using rev_in_lists[OF \<open>xs \<in> lists (rev ` \<C>)\<close>] rev_in_lists[OF \<open>ys \<in> lists (rev ` \<C>)\<close>] map_rev_lists_rev
    by (metis imageI)+
  moreover have "concat (map rev (rev xs)) = concat (map rev (rev ys))"
    unfolding rev_concat[symmetric] using \<open>concat xs = concat ys\<close> by blast
  ultimately have "map rev (rev xs) = map rev (rev ys)"
    using is_code by blast
  thus "xs = ys"
     using \<open>concat xs = concat ys\<close> by simp
qed

lemma dec_rev [simp, reversal_rule]:
  "u \<in> \<langle>\<C>\<rangle> \<Longrightarrow> Dec rev ` \<C> (rev u) = rev (map rev (Dec \<C> u))"
  by (auto simp only: rev_map lists_image rev_in_lists rev_concat[symmetric] dec_in_lists
      intro!: code_rev_code code.code_unique_dec imageI del: in_listsI)

lemma elem_comm_sing_set: assumes "ws \<in> lists \<C>" and "ws \<noteq> \<epsilon>" and "u \<in> \<C>" and "concat ws \<cdot> u = u \<cdot> concat ws"
  shows  "set ws = {u}"
  using assms
proof-
  have "concat (ws \<cdot> [u]) = concat ([u] \<cdot> ws)"
    using assms by simp
  have "ws \<cdot> [u] = [u] \<cdot> ws"
    using  \<open>u \<in> \<C>\<close> \<open>ws \<in> lists \<C>\<close> is_code[OF _ _  \<open>concat (ws \<cdot> [u]) = concat ([u] \<cdot> ws)\<close>]
    by simp
  from comm_nemp_pows_posE[OF this \<open>ws \<noteq> \<epsilon>\<close> not_Cons_self2[of u \<epsilon>]]
  obtain t k m where "ws = t\<^sup>@k" "[u] = t\<^sup>@m" "0 < k" "0 < m" "primitive t".
  hence "t = [u]"
    by force
  show "set ws = {u}"
    using  \<open>ws = t\<^sup>@k\<close>[unfolded \<open>t = [u]\<close>]  set_sing_nemp_eq[OF  sing_pow_set_sub  \<open>ws \<noteq> \<epsilon>\<close>] by blast
qed

lemma  pure_code_pres_prim:  assumes pure: "\<forall>u \<in> \<langle>\<C>\<rangle>. \<rho> u \<in> \<langle>\<C>\<rangle>" and
  "w \<in> \<langle>\<C>\<rangle>" and "primitive (Dec \<C> w)"
shows "primitive w"
proof-
  obtain k where "(\<rho> w)\<^sup>@k = w"
    using primroot_expE by blast

  have "\<rho> w \<in> \<langle>\<C>\<rangle>"
    using assms(2) pure by auto

  have "(Dec \<C> (\<rho> w))\<^sup>@k \<in> lists \<C>"
    by (metis \<open>\<rho> w \<in> \<langle>\<C>\<rangle>\<close> concat_pow_list_single dec_in_lists flatten_lists order_refl sing_pow_lists)

  have "(Dec \<C> (\<rho> w))\<^sup>@k = Dec \<C> w"
    using \<open>(Dec \<C> (\<rho> w)) \<^sup>@ k \<in> lists \<C>\<close>  code.code_unique_dec code_axioms concat_morph_power \<open>(\<rho> w) \<^sup>@ k = w\<close> concat_dec[OF \<open>\<rho> w \<in> \<langle>\<C>\<rangle>\<close>] by metis
  hence "k = 1"
    using \<open>primitive (Dec \<C> w)\<close> unfolding primitive_def by blast
  thus "primitive w"
    by (metis pow_list_1 \<open>\<rho> w \<^sup>@ k = w\<close> assms(3) dec_emp prim_nemp primroot_prim)
qed

lemma inj_on_dec: "inj_on (decompose \<C>) \<langle>\<C>\<rangle>"
  by (rule inj_on_inverseI[of _ concat]) simp

lemma ref_disj_interp: assumes "vs \<in> lists \<langle>\<C>\<rangle>" "p Ref \<C> vs s \<sim>\<^sub>\<D> ws"
  shows "p vs s \<sim>\<^sub>\<D> ws"
proof(rule disj_interpI)
  show "p (concat vs) s \<sim>\<^sub>\<I> ws"
    using disj_interpD0[OF \<open>p Ref \<C> vs s \<sim>\<^sub>\<D> ws\<close>]
    unfolding concat_ref[OF \<open>vs \<in> lists \<langle>\<C>\<rangle>\<close>].
  show "\<forall>u v. u \<le>p vs \<and> v \<le>p ws \<longrightarrow> p \<cdot> concat u \<noteq> concat v"
  proof (rule allI, rule allI, rule impI, elim conjE)
    fix u v assume "u \<le>p vs" "v \<le>p ws"
    have "Ref \<C> u \<le>p Ref \<C> vs"
      using ref_pref_mono[OF \<open>vs \<in> lists \<langle>\<C>\<rangle>\<close> \<open>u \<le>p vs\<close>].
    have "concat (Ref \<C> u) = concat u"
      using  concat_ref[OF pref_in_lists[OF \<open>u \<le>p vs\<close> \<open>vs \<in> lists \<langle>\<C>\<rangle>\<close>]].
    then show "p \<cdot> concat u \<noteq> concat v"
      using disj_interpD1[OF \<open>p Ref \<C> vs s \<sim>\<^sub>\<D> ws\<close> \<open>Ref \<C> u \<le>p Ref \<C> vs\<close> \<open>v \<le>p ws\<close>]
      by simp
  qed
qed

end \<comment> \<open>end context code\<close>

lemma emp_is_code: "code {}"
  using code.intro empty_iff insert_iff lists_empty by metis

lemma code_rev_code_iff [reversal_rule]: "code (rev ` C) \<longleftrightarrow> code C"
  by (rule iffI[OF code.code_rev_code[of "rev ` C", unfolded rev_rev_image_eq] code.code_rev_code])

lemma code_induct_hd: assumes "\<epsilon> \<notin> C" and
  "\<And> xs ys. xs \<in> lists C \<Longrightarrow> ys \<in> lists C \<Longrightarrow> concat xs = concat ys \<Longrightarrow> hd xs = hd ys"
shows "code C"
proof
  show "xs \<in> lists C \<Longrightarrow> ys \<in> lists C \<Longrightarrow> concat xs = concat ys \<Longrightarrow> xs = ys" for xs ys
  proof (induct xs ys rule: list_induct2')
    case (4 x xs y ys)
    from assms(2)[OF "4.prems"]
    have "x = y" by force
    from "4.prems"[unfolded this]
    have "xs \<in> lists C" and "ys \<in> lists C" and "concat xs = concat ys"
      by simp_all
    from "4.hyps"[OF this] \<open>x = y\<close>
    show ?case
      by simp
  qed (auto simp add: \<open>\<epsilon> \<notin> C\<close>)
qed

lemma ref_set_primroot: assumes "ws \<in> lists (G - {\<epsilon>})" and "code (\<rho>`G)"
  shows "set (Ref \<rho>`G ws) = \<rho>`(set ws)"
proof-
  have "G \<subseteq> \<langle>\<rho>`G\<rangle>"
  proof
    fix x
    assume "x \<in> G"
    show "x \<in> \<langle>\<rho> ` G\<rangle>"
      by (metis \<open>x \<in> G\<close> genset_sub image_subset_iff power_in primroot_expE)
  qed
  hence "ws \<in> lists \<langle>\<rho>`G\<rangle>"
    using assms by blast

  have "set (decompose (\<rho>`G) a) = {\<rho> a}" if "a \<in> set ws" for a
  proof-
    have "\<rho> a \<in> \<rho>`G" "a \<noteq> \<epsilon>"
      using \<open>a \<in> set ws\<close> \<open>ws \<in> lists (G - {\<epsilon>})\<close> by force+
    obtain k where "(Dec (\<rho>`G) a) = [\<rho> a]\<^sup>@k" "0 < k"
      using code.code_unique_dec[OF \<open>code (\<rho> ` G)\<close> sing_pow_lists concat_pow_list_single, OF \<open>\<rho> a \<in> \<rho> ` G\<close>]
        primroot_expE by metis
    hence "Dec (\<rho>`G) a \<noteq> \<epsilon>"
      by simp
    from set_sing_nemp_eq[OF _ this]
    show "set (decompose (\<rho>`G) a) = {\<rho> a}"
      unfolding \<open>Dec \<rho> ` G a = [\<rho> a] \<^sup>@ k\<close> using sing_pow_set' by metis
  qed

  have "(set`(decompose (\<rho>`G))`set ws) = {{\<rho> a} |a. a \<in> set ws}" (is "?L = ?R")
  proof
    show "?L \<subseteq> ?R"
      using \<open>\<And>a. a \<in> set ws \<Longrightarrow> set (Dec \<rho> ` G a) = {\<rho> a}\<close> by blast
    show "?R \<subseteq> ?L"
      using \<open>\<And>a. a \<in> set ws \<Longrightarrow> set (Dec \<rho> ` G a) = {\<rho> a}\<close> by blast
  qed

  show ?thesis
    using ref_set[OF \<open>ws \<in> lists \<langle>\<rho>`G\<rangle>\<close>]
      Setcompr_eq_image \<open>set ` decompose (\<rho> ` G) ` set ws = {{\<rho> a} |a. a \<in> set ws}\<close> by (auto simp add: refine_def)
qed

section \<open>Prefix code\<close>

locale prefix_code =
  fixes \<C>
  assumes
    emp_not_in: "\<epsilon> \<notin> \<C>" and
    pref_free: "u \<in> \<C> \<Longrightarrow> v \<in> \<C> \<Longrightarrow> u \<le>p v \<Longrightarrow> u = v"

begin

lemma nemp: "u \<in> \<C> \<Longrightarrow> u \<noteq> \<epsilon>"
  using emp_not_in by force

lemma concat_pref_concat:
  assumes "us \<in> lists \<C>" "vs \<in> lists \<C>" "concat us \<le>p concat vs"
  shows "us \<le>p vs"
using assms proof (induction us vs rule: list_induct2')
  case (4 x xs y ys)
  from "4.prems"
  have "x = y"
    by (auto elim!: ruler_prefE intro: pref_free sym del: in_listsD)
  with "4" show "x # xs \<le>p y # ys"
  by simp
qed (simp_all add: nemp)

lemma concat_pref_concat_conv:
  assumes "us \<in> lists \<C>" "vs \<in> lists \<C>"
  shows "concat us \<le>p concat vs \<longleftrightarrow> us \<le>p vs"
using concat_pref_concat[OF assms] pref_concat_pref..

sublocale code
  by standard (simp_all add: pref_antisym concat_pref_concat)

lemmas is_code = is_code and
  code = code_axioms

lemma dec_pref_unique:
  "w \<in> \<langle>\<C>\<rangle> \<Longrightarrow> p \<in> \<langle>\<C>\<rangle> \<Longrightarrow> p \<le>p w \<Longrightarrow> Dec \<C> p \<le>p Dec \<C> w"
  using concat_pref_concat_conv[of "Dec \<C> p" "Dec \<C> w", THEN iffD1]
  by simp

lemma concat_suf_eq: assumes
  "us \<in> lists \<C>" "ws \<in> lists \<C>" and
  "concat us \<cdot> s = concat ws" and "s <s last ws"
shows "us = ws" and "s = \<epsilon>"
proof-
  from concat_pref_concat_conv[OF assms(1,2), folded \<open>concat us \<cdot> s = concat ws\<close>]
  have "us \<le>p ws"
    by blast
  hence "concat (us\<inverse>\<^sup>>ws) = s"
    using assms(3) concat_morph_lq lq_triv by metis
  from concat_butlast_last[of "us\<inverse>\<^sup>>ws", unfolded this]
  show "us = ws"
    using empty_lq_eq[OF \<open>us \<le>p ws\<close>] assms(4)  last_appendR lq_pref[OF \<open>us \<le>p ws\<close>]
    ssuf_extD suffix_order.strict_iff_not by metis
  show "s = \<epsilon>"
    using \<open>concat us \<cdot> s = concat ws\<close>[unfolded \<open>us = ws\<close>] by blast
qed

end

thm prefix_code.concat_suf_eq[reversed]

subsection \<open>Suffix code\<close>

locale suffix_code = prefix_code "(rev ` \<C>)" for \<C>
begin

thm dec_rev
    code

sublocale code
  using code_rev_code unfolding rev_rev_image_eq.

lemmas concat_suf_concat = concat_pref_concat[reversed] and
       concat_suf_concat_conv = concat_pref_concat_conv[reversed] and
       nemp = nemp[reversed] and
       suf_free = pref_free[reversed] and
       dec_suf_unique = dec_pref_unique[reversed]

lemma concat_pref_eq: assumes
  "us \<in> lists \<C>" "ws \<in> lists \<C>" and
  "p \<cdot> concat us = concat ws" and "p <p hd ws"
shows "us = ws" and "p = \<epsilon>"
proof (atomize(full))
  show "us = ws \<and> p = \<epsilon>"
  proof (cases "ws = \<epsilon>")
    assume "ws = \<epsilon>"
    hence "p \<cdot> concat us = \<epsilon>"
      unfolding \<open>p \<cdot> concat us = concat ws\<close> by simp
    thus "us = ws \<and> p = \<epsilon>"
      unfolding \<open>ws = \<epsilon>\<close> using \<open>us \<in> lists \<C>\<close>
      using concat_emp[OF emp_not_in \<open>us \<in> lists \<C>\<close>] by blast
  next
    assume "ws \<noteq> \<epsilon>"
    from  concat_suf_eq[reversed, OF assms(1-3)]
    show "us = ws \<and> p = \<epsilon>"
      unfolding hd_map[OF \<open>ws \<noteq> \<epsilon>\<close>] spref_rev_suf_iff[symmetric]
      using \<open>p <p hd ws\<close> by blast
  qed
qed

thm is_code
    code_axioms
    code

end

subsection \<open>Bifix code\<close>

locale bifix_code = prefix_code + suf: suffix_code
begin

lemma joint_interp_triv: assumes
  "us \<in> lists \<C>" "ws \<in> lists \<C>" and
  interp: "p (concat us) s \<sim>\<^sub>\<I> ws" and
  joint: "\<not> p us s \<sim>\<^sub>\<D> ws"
shows "p = \<epsilon>" and "s = \<epsilon>" and "us = ws"
proof-
  from non_disjoint_interpE[OF interp joint]
  obtain ws1 ws2 us1 us2 where "us1 \<cdot> us2 = us" "ws1 \<cdot> ws2 = ws"
    "p \<cdot> concat us1 = concat ws1" "concat us2 \<cdot> s = concat ws2".
  hence "ws1 \<in> lists \<C>" "ws2 \<in> lists \<C>" "us1 \<in> lists \<C>" "us2 \<in> lists \<C>"
    using \<open>us \<in> lists \<C>\<close> \<open>ws \<in> lists \<C>\<close> by inlists
  have eq1: "us2 = ws2 \<and> s = \<epsilon>"
  proof (cases "ws2 = \<epsilon>")
    assume "ws2 = \<epsilon>"
    show ?thesis
      thm emp_concat_emp'
      using \<open>concat us2 \<cdot> s = concat ws2\<close> emp_concat_emp'[OF \<open>us2 \<in> lists \<C>\<close>] unfolding \<open>ws2 = \<epsilon>\<close>
      concat.simps by blast
  next
    assume "ws2 \<noteq> \<epsilon>"
    with concat_suf_eq[OF \<open>us2 \<in> lists \<C>\<close> \<open>ws2 \<in> lists \<C>\<close> \<open>concat us2 \<cdot> s = concat ws2\<close>]
    show ?thesis
      using fac_interpD(2)[OF interp, folded \<open>ws1 \<cdot> ws2 = ws\<close>] by force
  qed
  have eq2: "us1 = ws1 \<and> p = \<epsilon>"
  proof (cases "ws1 = \<epsilon>")
    assume "ws1 = \<epsilon>"
    show ?thesis
      using \<open>p \<cdot> concat us1 = concat ws1\<close> emp_concat_emp'[OF \<open>us1 \<in> lists \<C>\<close>] unfolding \<open>ws1 = \<epsilon>\<close>
      concat.simps by blast
  next
    assume "ws1 \<noteq> \<epsilon>"
    with suf.concat_pref_eq[OF \<open>us1 \<in> lists \<C>\<close> \<open>ws1 \<in> lists \<C>\<close> \<open>p \<cdot> concat us1 = concat ws1\<close>]
    show ?thesis
      using fac_interpD(1)[OF interp, folded \<open>ws1 \<cdot> ws2 = ws\<close>] by force
  qed
  show "p = \<epsilon>" and "s = \<epsilon>" and "us = ws"
    using eq1 eq2 \<open>us1 \<cdot> us2 = us\<close> \<open>ws1 \<cdot> ws2 = ws\<close> by blast+
qed

end

section \<open>Marked code\<close>

locale marked_code =
  fixes \<C>
  assumes
    emp_not_in: "\<epsilon> \<notin> \<C>" and
    marked: "u \<in> \<C> \<Longrightarrow> v \<in> \<C> \<Longrightarrow> hd u = hd v \<Longrightarrow> u = v"

begin

lemma nemp: "u \<in> \<C> \<Longrightarrow> u \<noteq> \<epsilon>"
  using emp_not_in by blast

sublocale prefix_code
  by (unfold_locales) (simp_all add: emp_not_in marked nemp pref_hd_eq)


lemma marked_concat_lcp: "us \<in> lists \<C> \<Longrightarrow> vs \<in> lists \<C> \<Longrightarrow> concat (us \<and>\<^sub>p vs) = (concat us) \<and>\<^sub>p (concat vs)"
proof (induct us vs rule: list_induct2')
  case (4 x xs y ys)
  hence "x \<in> \<C>" and "y \<in> \<C>" and "xs \<in> lists \<C>" and "ys \<in> lists \<C>"
    by simp_all
  show ?case
  proof (cases)
    assume "x = y"
    thus "concat (x # xs \<and>\<^sub>p y # ys) = concat (x # xs) \<and>\<^sub>p concat (y # ys)"
      using "4.hyps"[OF \<open>xs \<in> lists \<C>\<close> \<open>ys \<in> lists \<C>\<close>] by (simp add: lcp_ext_left)
  next
    assume "x \<noteq> y"
    with marked[OF \<open>x \<in> \<C>\<close> \<open>y \<in> \<C>\<close>] have "hd x \<noteq> hd y" by blast
    hence "concat (x # xs) \<and>\<^sub>p concat (y # ys) = \<epsilon>"
      by (simp add: \<open>x \<in> \<C>\<close> \<open>y \<in> \<C>\<close> nemp lcp_distinct_hd)
    moreover have "concat (x # xs \<and>\<^sub>p y # ys) = \<epsilon>"
      using \<open>x \<noteq> y\<close> by simp
    ultimately show ?thesis by presburger
  qed
qed simp_all

lemma hd_concat_hd: assumes "xs \<in> lists \<C>" and "ys \<in> lists \<C>" and "xs \<noteq> \<epsilon>" and "ys \<noteq> \<epsilon>" and
  "hd (concat xs) = hd (concat ys)"
shows "hd xs = hd ys"
proof-
  have "hd (hd xs) = hd (hd ys)"
    using assms  hd_concat[OF \<open>xs \<noteq> \<epsilon>\<close> lists_hd_in_set[THEN nemp]] hd_concat[OF \<open>ys \<noteq> \<epsilon>\<close> lists_hd_in_set[THEN nemp]]
    by presburger

  from marked[OF lists_hd_in_set lists_hd_in_set this] assms(1-4)
  show "hd xs = hd ys"
    by simp
qed

end

section "Non-overlapping code"

locale non_overlapping =
  fixes \<C>
  assumes
    emp_not_in: "\<epsilon> \<notin> \<C>" and
    no_overlap: "u \<in> \<C> \<Longrightarrow> v \<in> \<C> \<Longrightarrow> z \<le>p u \<Longrightarrow> z \<le>s v \<Longrightarrow> z \<noteq> \<epsilon> \<Longrightarrow> u = v" and
    no_fac: "u \<in> \<C> \<Longrightarrow> v \<in> \<C> \<Longrightarrow> u \<le>f v \<Longrightarrow>  u = v"
begin

lemma nemp: "u \<in> \<C>  \<Longrightarrow> u \<noteq> \<epsilon>"
  using emp_not_in by force

sublocale prefix_code
  using nemp non_overlapping.no_fac non_overlapping_axioms prefix_code.intro
   prefix_imp_sublist by metis

lemma rev_non_overlapping: "non_overlapping (rev ` \<C>)"
proof
  show "\<epsilon> \<notin> rev ` \<C>"
    using nemp by force
  show "u \<in> rev ` \<C> \<Longrightarrow> v \<in> rev ` \<C> \<Longrightarrow> z \<le>p u \<Longrightarrow> z \<le>s v \<Longrightarrow> z \<noteq> \<epsilon> \<Longrightarrow> u = v" for u v z
    using no_overlap[reversed] unfolding rev_in_conv..
  show "u \<in> rev ` \<C> \<Longrightarrow> v \<in> rev ` \<C> \<Longrightarrow> u \<le>f v \<Longrightarrow> u = v" for u v
    using no_fac[reversed] unfolding rev_in_conv by presburger
qed

sublocale suf: suffix_code \<C>
proof-
  interpret i: non_overlapping "rev ` \<C>"
    using rev_non_overlapping.
  from i.prefix_code_axioms
  show "suffix_code \<C>"
    by unfold_locales
qed

lemma overlap_concat_last: assumes  "u \<in> \<C>" and "vs \<in> lists \<C>" and "vs \<noteq> \<epsilon>" and
      "r \<noteq> \<epsilon>" and "r \<le>p u" and  "r \<le>s p \<cdot> concat vs"
  shows  "u = last vs"
proof-
  from suffix_same_cases[OF suf_ext[OF concat_last_suf[OF \<open>vs \<noteq> \<epsilon>\<close>]] \<open>r \<le>s p \<cdot> concat vs\<close>]
  show "u = last vs"
  proof (rule disjE)
    assume "r \<le>s last vs"
    from no_overlap[OF \<open>u \<in> \<C>\<close> _ \<open>r \<le>p u\<close> this \<open>r \<noteq> \<epsilon>\<close>]
    show "u = last vs"
      using \<open>vs \<in> lists \<C>\<close> \<open>vs \<noteq> \<epsilon>\<close> by force
  next
    assume "last vs \<le>s r"
    from no_fac[OF _ \<open>u \<in> \<C>\<close> pref_suf_fac, OF _ \<open>r \<le>p u\<close>  this]
    show "u = last vs"
      using \<open>vs \<in> lists \<C>\<close> \<open>vs \<noteq> \<epsilon>\<close> by force
  qed
qed

lemma overlap_concat_hd: assumes  "u \<in> \<C>" and "vs \<in> lists \<C>" and "vs \<noteq> \<epsilon>" and "r \<noteq> \<epsilon>" and "r \<le>s u" and  "r \<le>p concat vs \<cdot> s"
  shows  "u = hd vs"
proof-
  interpret c: non_overlapping "rev ` \<C>" by (simp add: rev_non_overlapping)
  from c.overlap_concat_last[reversed, OF assms]
  show ?thesis.
qed

lemma fac_concat_fac:
  assumes "us \<in> lists \<C>" "vs \<in> lists \<C>"
    and "1 < card (set us)"
    and "concat vs = p \<cdot> concat us \<cdot> s"
  obtains ps ss where  "concat ps = p" and "concat ss = s" and "ps \<cdot> us \<cdot> ss = vs"
proof-
  have "us \<noteq> \<epsilon>"
    using \<open>1 < card (set us)\<close> by fastforce
  let ?a = "hd us"
  define us1 where "us1 = takeWhile (\<lambda> b. b = ?a) us"
  define us2 where "us2 = dropWhile (\<lambda> b. b = ?a) us"
  define k where "k = \<^bold>|us1\<^bold>|"
  have "us1 \<noteq> \<epsilon>"
    unfolding us1_def takeWhile_eq_Nil_iff using \<open>us \<noteq> \<epsilon>\<close> by blast
  note nemp_len[OF this, folded k_def]
  have "us = us1 \<cdot> us2"
    unfolding us1_def us2_def by simp
  have "set us1 = {?a}"
    using set_sing_nemp_eq takeWhile_subset \<open>us1 \<noteq> \<epsilon>\<close> unfolding us1_def by metis
  hence "us1 = [?a]\<^sup>@k"
    using sing_pow_exp unfolding k_def by fastforce
  have "last us1 = ?a"
    unfolding \<open>us1 = [?a]\<^sup>@k\<close>[unfolded  pow_pos2[OF \<open>0 < k\<close>]] using last_snoc.
  have "us2 \<noteq> \<epsilon>"
    using \<open>1 < card (set us)\<close>[unfolded \<open>us = us1 \<cdot> us2\<close>] \<open>set us1 = {?a}\<close> by force
  have "hd us2 \<noteq> ?a"
    using  hd_dropWhile[OF \<open>us2 \<noteq> \<epsilon>\<close>[unfolded us2_def]] unfolding us2_def.
  note this[symmetric, folded \<open>last us1 = ?a\<close>]

  have "us2 \<in> lists \<C>" "us1 \<in> lists \<C>"
    using \<open>us = us1 \<cdot> us2\<close> \<open>us \<in> lists \<C>\<close> by simp_all
  hence "concat us2 \<noteq> \<epsilon>"
    using \<open>us2 \<noteq> \<epsilon>\<close> nemp by force
  hence "p \<cdot> concat us1 <p concat vs"
    using \<open>us = us1 \<cdot> us2\<close> unfolding \<open>concat vs = p \<cdot> concat us \<cdot> s\<close> by simp
  from pref_mod_list'[OF this]
  obtain j r where "j < \<^bold>|vs\<^bold>|" "r <p vs ! j" "concat (take j vs) \<cdot> r = p \<cdot> concat us1".
  have "r = \<epsilon>"
  proof (rule ccontr)
    assume "r \<noteq> \<epsilon>"
    from spref_exE[OF \<open>r <p vs ! j\<close>]
    obtain z where "r \<cdot> z = vs ! j" "z \<noteq> \<epsilon>".
    from overlap_concat_last[OF _ \<open>us1 \<in> lists \<C>\<close> \<open>us1 \<noteq> \<epsilon>\<close> \<open>r \<noteq> \<epsilon>\<close> sprefD1[OF \<open>r <p vs ! j\<close>] sufI[OF \<open>concat (take j vs) \<cdot> r = p \<cdot> concat us1\<close>]]
    have "vs ! j = last us1"
      using  nth_in_lists[OF \<open>j < \<^bold>|vs\<^bold>|\<close> \<open>vs \<in> lists \<C>\<close>].

    have concat_vs: "concat vs = concat (take j vs) \<cdot> vs!j \<cdot> concat (drop (Suc j) vs)"
      unfolding lassoc concat_take_Suc[OF \<open>j < \<^bold>|vs\<^bold>|\<close>] concat_morph[symmetric] by force
    from this[folded \<open>r \<cdot> z = vs ! j\<close>]
    have "z \<cdot> concat (drop (Suc j) vs) = concat us2 \<cdot> s"
      unfolding \<open>concat vs = p \<cdot> concat us \<cdot> s\<close> lassoc \<open>concat (take j vs) \<cdot> r = p \<cdot> concat us1\<close>            \<open>us = us1 \<cdot> us2\<close> concat_morph
      unfolding rassoc cancel by simp
    from overlap_concat_hd[OF _ \<open>us2 \<in> lists \<C>\<close> \<open>us2 \<noteq> \<epsilon>\<close> \<open>z \<noteq> \<epsilon>\<close> sufI[OF \<open>r \<cdot> z = vs ! j\<close>]  prefI[OF this]]
    have "vs ! j = hd us2"
      using  nth_in_lists[OF \<open>j < \<^bold>|vs\<^bold>|\<close> \<open>vs \<in> lists \<C>\<close>].

    thus False
      unfolding \<open>vs ! j = last us1\<close> using  \<open>last us1 \<noteq> hd us2\<close> by contradiction
  qed

  have "drop j vs \<in> lists \<C>" and "take j vs \<in> lists \<C>"
    using \<open>vs \<in> lists \<C>\<close> by inlists
  have "concat us2 \<cdot> s = concat (drop j vs)"
    using arg_cong[OF takedrop[of j vs], of concat] \<open>concat (take j vs) \<cdot> r = p \<cdot> concat us1\<close>
    unfolding \<open>concat vs = p \<cdot> concat us \<cdot> s\<close> concat_morph \<open>r = \<epsilon>\<close> emp_simps \<open>us = us1 \<cdot> us2\<close> by auto
  from prefI[OF this]
  have "us2 \<le>p drop j vs"
    using concat_pref_concat_conv[OF \<open>us2 \<in> lists \<C>\<close> \<open>drop j vs \<in> lists \<C>\<close>] by blast
  hence s: "concat (us2\<inverse>\<^sup>>drop j vs) = s"
    using \<open>concat us2 \<cdot> s = concat (drop j vs)\<close> concat_morph_lq lqI by blast

  from \<open>concat (take j vs) \<cdot> r = p \<cdot> concat us1\<close>[unfolded \<open>r = \<epsilon>\<close> emp_simps]
  have "concat us1 \<le>s concat (take j vs)"
    by fastforce
  hence "us1 \<le>s take j vs"
    using suf.concat_pref_concat_conv[reversed, OF \<open>us1 \<in> lists \<C>\<close> \<open>take j vs \<in> lists \<C>\<close>] by blast
  from arg_cong[OF rq_suf[OF this], of concat, unfolded concat_morph]
  have p: "concat (take j vs\<^sup><\<inverse>us1 ) = p"
    using rqI[OF \<open>concat (take j vs) = p \<cdot> concat us1\<close>[symmetric]]
    rq_triv by metis

  have "take j vs\<^sup><\<inverse>us1  \<cdot> us \<cdot> us2\<inverse>\<^sup>>drop j vs = vs"
    unfolding \<open>us = us1 \<cdot> us2\<close> rassoc lq_pref[OF \<open>us2 \<le>p drop j vs\<close>]
    unfolding lassoc  rq_suf[OF \<open>us1 \<le>s take j vs\<close>] by simp

  from that[OF p s this]
  show thesis.
qed

theorem prim_morph:
  assumes "ws \<in> lists \<C>"
    and "\<^bold>|ws\<^bold>| \<noteq> 1"
    and "primitive ws"
  shows "primitive (concat ws)"
proof (rule ccontr)
  have "ws \<in> lists \<C>" and "ws \<cdot> ws \<in> lists \<C>"
    using \<open>ws \<in> lists \<C>\<close> by simp_all
  moreover have "1 < card (set ws)" using \<open>primitive ws\<close> \<open>\<^bold>|ws\<^bold>| \<noteq> 1\<close>
    by (rule prim_card_set)
  moreover assume "\<not> primitive (concat ws)"
  then obtain k z where "2 \<le> k" and "z \<^sup>@ k = concat ws"
    by (elim not_prim_primroot_expE)
  have "concat (ws \<cdot> ws) = z \<cdot> concat ws \<cdot> z\<^sup>@(k-1)"
    unfolding concat_morph \<open>z \<^sup>@ k = concat ws\<close>[symmetric] pow_add[symmetric] pow_Suc[symmetric]
    using \<open>2 \<le> k\<close> by simp
  ultimately obtain ps ss where "concat ps = z" and "concat ss = z\<^sup>@(k-1)" and  "ps \<cdot> ws \<cdot> ss = ws \<cdot> ws"
    by (rule fac_concat_fac)
  have "ps \<^sup>@ k \<in> lists \<C>"
    using \<open>ps \<cdot> ws \<cdot> ss = ws \<cdot> ws\<close> \<open>ws \<cdot> ws \<in> lists \<C>\<close> by inlists
  moreover have "concat (ps \<^sup>@ k) = concat ws"
    unfolding concat_pow_list \<open>concat ps = z\<close> \<open>z \<^sup>@ k = concat ws\<close>..
  ultimately have "ps \<^sup>@ k = ws" using \<open>ws \<in> lists \<C>\<close> by (intro is_code)
  show False
    using prim_exp_one[OF \<open>primitive ws\<close> \<open>ps \<^sup>@ k = ws\<close>] \<open>2 \<le> k\<close> by presburger
qed

lemma prim_concat_conv:
  assumes "ws \<in> lists \<C>"
    and "\<^bold>|ws\<^bold>| \<noteq> 1"
  shows "primitive (concat ws) \<longleftrightarrow> primitive ws"
  using prim_concat_prim prim_morph[OF assms]..

end

section \<open>Binary code\<close>

text\<open>We pay a special attention to two element codes.
In particular, we show that two words form a code if and only if they do not commute. This means that two
words either commute, or do not satisfy any nontrivial relation.
\<close>

definition  bin_lcp  where "bin_lcp x y  =  x\<cdot>y \<and>\<^sub>p y\<cdot>x"
definition  bin_lcs  where "bin_lcs x y  =  x\<cdot>y \<and>\<^sub>s y\<cdot>x"

definition  bin_mismatch where "bin_mismatch x y =  (x\<cdot>y)!\<^bold>|bin_lcp x y\<^bold>|"
definition  bin_mismatch_suf where " bin_mismatch_suf x y = bin_mismatch (rev y) (rev x)"

value[nbe] "[0::nat,1,0]!3"

lemma bin_lcs_rev: "bin_lcs x y = rev (bin_lcp (rev x) (rev y))"
  unfolding bin_lcp_def bin_lcs_def  longest_common_suffix_def rev_append using lcp_sym by fastforce

lemma bin_lcp_sym: "bin_lcp x y = bin_lcp y x"
  unfolding bin_lcp_def using lcp_sym.

lemma bin_mismatch_comm: "(bin_mismatch x y = bin_mismatch y x) \<longleftrightarrow> (x \<cdot> y = y \<cdot> x)"
  unfolding bin_mismatch_def bin_lcp_def lcp_sym[of "y \<cdot> x"]
  using  lcp_mismatch'[of "x \<cdot> y" "y \<cdot> x", unfolded comm_comp_eq_conv[of x y]]   by fastforce

lemma bin_lcp_pref_fst_snd: "bin_lcp x y \<le>p x \<cdot> y"
  unfolding bin_lcp_def using lcp_pref.

lemma bin_lcp_pref_snd_fst: "bin_lcp x y \<le>p y \<cdot> x"
  using bin_lcp_pref_fst_snd[of y x, unfolded bin_lcp_sym[of y x]].

lemma bin_lcp_bin_lcs [reversal_rule]:  "bin_lcp (rev x) (rev y) = rev (bin_lcs x y)"
  unfolding bin_lcp_def bin_lcs_def rev_append[symmetric] lcs_lcp
    lcs_sym[of "x \<cdot> y"]..

lemmas bin_lcs_sym = bin_lcp_sym[reversed]

lemma bin_lcp_len: "x \<cdot> y \<noteq> y \<cdot> x \<Longrightarrow> \<^bold>|bin_lcp x y\<^bold>| < \<^bold>|x \<cdot> y\<^bold>|"
  unfolding bin_lcp_def
  using lcp_len' pref_comm_eq by blast

lemmas bin_lcs_len = bin_lcp_len[reversed]

lemma bin_mismatch_pref_suf'[reversal_rule]:
  "bin_mismatch (rev y) (rev x) =  bin_mismatch_suf x y"
  unfolding bin_mismatch_suf_def..

subsection \<open>Binary code locale\<close>

locale binary_code =
  fixes u\<^sub>0 u\<^sub>1
  assumes non_comm: "u\<^sub>0 \<cdot> u\<^sub>1 \<noteq> u\<^sub>1 \<cdot> u\<^sub>0"

begin

text\<open>A crucial property of two element codes is the constant decoding delay given by the word $\alpha$,
which is a prefix of any generating word (sufficiently long), while the letter
immediately after this common prefix indicates the first element of the decomposition.
\<close>

definition uu where "uu a = (if a then u\<^sub>0 else u\<^sub>1)"

lemma bin_code_set_bool: "{uu a,uu (\<not> a)} = {u\<^sub>0,u\<^sub>1}"
  by (induct a, unfold uu_def, simp_all add: insert_commute)

lemma bin_code_set_bool': "{uu a,uu (\<not> a)} = {u\<^sub>1,u\<^sub>0}"
  by (induct a, unfold uu_def, simp_all add: insert_commute)

lemma bin_code_swap: "binary_code u\<^sub>1 u\<^sub>0"
  using binary_code.intro[OF non_comm[symmetric]].

lemma bin_code_bool: "binary_code (uu a)  (uu (\<not> a))"
  unfolding uu_def by (induct a, simp_all add: bin_code_swap binary_code_axioms)

lemma bin_code_neq: "u\<^sub>0 \<noteq> u\<^sub>1"
  using non_comm by auto

lemma bin_code_neq_bool: "uu a \<noteq> uu (\<not> a)"
  unfolding uu_def by (induct a) (use bin_code_neq in fastforce)+

lemma bin_fst_nemp: "u\<^sub>0 \<noteq> \<epsilon>" and bin_snd_nemp: "u\<^sub>1 \<noteq> \<epsilon>" and bin_nemp_bool: "uu a \<noteq> \<epsilon>"
  using non_comm uu_def by auto

lemma bin_not_comp: "\<not> u\<^sub>0 \<cdot> u\<^sub>1 \<bowtie> u\<^sub>1 \<cdot> u\<^sub>0"
  using comm_comp_eq_conv non_comm by blast

lemma bin_not_comp_bool: "\<not> (uu a \<cdot> uu (\<not> a) \<bowtie> uu (\<not> a) \<cdot> uu a)"
  unfolding uu_def by (induct a, use bin_not_comp pref_comp_sym in auto)

lemma bin_not_comp_suf: "\<not> u\<^sub>0 \<cdot> u\<^sub>1 \<bowtie>\<^sub>s u\<^sub>1 \<cdot> u\<^sub>0"
  using comm_comp_eq_conv_suf non_comm[reversed] by blast

lemma bin_not_comp_suf_bool: "\<not> (uu a \<cdot> uu (\<not> a) \<bowtie>\<^sub>s uu (\<not> a) \<cdot> uu a)"
  unfolding uu_def by (induct a, use bin_not_comp_suf suf_comp_sym in auto)

lemma bin_mismatch_neq: "bin_mismatch u\<^sub>0 u\<^sub>1 \<noteq> bin_mismatch u\<^sub>1 u\<^sub>0"
  using non_comm[folded bin_mismatch_comm].

abbreviation bin_code_lcp ("\<alpha>") where  "bin_code_lcp \<equiv> bin_lcp u\<^sub>0 u\<^sub>1"
abbreviation bin_code_lcs where "bin_code_lcs \<equiv> bin_lcs u\<^sub>0 u\<^sub>1"
abbreviation bin_code_mismatch_fst ("c\<^sub>0") where "bin_code_mismatch_fst \<equiv> bin_mismatch u\<^sub>0 u\<^sub>1"
abbreviation bin_code_mismatch_snd ("c\<^sub>1") where "bin_code_mismatch_snd \<equiv> bin_mismatch u\<^sub>1 u\<^sub>0"

definition cc where "cc a = (if a then c\<^sub>0 else c\<^sub>1)"


lemmas bin_lcp_swap = bin_lcp_sym[of u\<^sub>0 u\<^sub>1, symmetric] and
       bin_lcp_pref = bin_lcp_pref_fst_snd[of u\<^sub>0 u\<^sub>1] and
       bin_lcp_pref' = bin_lcp_pref_snd_fst[of u\<^sub>0 u\<^sub>1] and
       bin_lcp_short = bin_lcp_len[OF non_comm, unfolded lenmorph]

lemmas bin_code_simps = cc_def uu_def if_True if_False bool_simps

lemma bin_lcp_bool: "bin_lcp (uu a) (uu (\<not> a)) = bin_code_lcp"
  unfolding uu_def by (induct a, simp_all add: bin_lcp_swap)

lemma bin_lcp_spref: "\<alpha> <p u\<^sub>0 \<cdot> u\<^sub>1"
  using bin_lcp_pref bin_lcp_pref' bin_not_comp by fastforce

lemma bin_lcp_spref': "\<alpha> <p u\<^sub>1 \<cdot> u\<^sub>0"
  using bin_lcp_pref bin_lcp_pref' bin_not_comp by fastforce

lemma bin_lcp_spref_bool: "\<alpha> <p uu a \<cdot> uu (\<not> a)"
  unfolding uu_def by (induct a, use  bin_lcp_spref bin_lcp_spref' in auto)

lemma bin_mismatch_bool': "\<alpha> \<cdot> [cc a] \<le>p uu a \<cdot> uu (\<not> a)"
  using add_nth_pref[OF bin_lcp_spref_bool, of a]
  unfolding uu_def cc_def bin_mismatch_def bin_lcp_bool bin_lcp_swap
  by (induct a) simp_all

lemma bin_mismatch_bool: "\<alpha> \<cdot> [cc a] \<le>p uu a \<cdot> \<alpha>"
proof-
  from pref_prolong[OF bin_mismatch_bool', OF triv_pref]
  have "\<alpha> \<cdot> [cc a] \<le>p uu a \<cdot> (uu (\<not> a) \<cdot> uu a)"
    by blast
  from pref_prod_pref_short[OF this bin_lcp_pref_snd_fst, unfolded bin_lcp_bool lenmorph sing_len]
  show ?thesis
    using nemp_len[OF bin_nemp_bool, of a] by linarith
qed

lemmas bin_fst_mismatch = bin_mismatch_bool[of True, unfolded bin_code_simps] and
       bin_fst_mismatch' = bin_mismatch_bool'[of True, unfolded bin_code_simps] and
       bin_snd_mismatch = bin_mismatch_bool[of False, unfolded bin_code_simps] and
       bin_snd_mismatch' = bin_mismatch_bool'[of False, unfolded bin_code_simps]

lemma bin_lcp_pref_all: "xs \<in> lists {u\<^sub>0,u\<^sub>1} \<Longrightarrow> \<alpha> \<le>p concat xs \<cdot> \<alpha>"
proof (induct xs)
  case (Cons a xs)
  have "a \<in> {u\<^sub>0,u\<^sub>1}" and "xs \<in> lists {u\<^sub>0, u\<^sub>1}"
    using \<open>a # xs \<in> lists {u\<^sub>0, u\<^sub>1}\<close> by simp_all
  show ?case
  proof (rule two_elemP[OF \<open>a \<in> {u\<^sub>0,u\<^sub>1}\<close>], simp_all)
    show "\<alpha> \<le>p u\<^sub>0 \<cdot> concat xs \<cdot> \<alpha>"
      using pref_extD[OF bin_fst_mismatch] Cons.hyps[OF \<open>xs \<in> lists {u\<^sub>0, u\<^sub>1}\<close>] pref_prolong by blast
  next
    show "\<alpha> \<le>p u\<^sub>1 \<cdot> concat xs \<cdot> \<alpha>"
      using pref_extD[OF bin_snd_mismatch] Cons.hyps[OF \<open>xs \<in> lists {u\<^sub>0, u\<^sub>1}\<close>] pref_prolong by blast
  qed
qed simp

lemma bin_lcp_pref_all_hull: "w \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle> \<Longrightarrow> \<alpha> \<le>p w \<cdot> \<alpha>"
  using bin_lcp_pref_all using hull_concat_listsE by metis

lemma bin_lcp_mismatch_pref_all_bool: assumes "q \<le>p w" and "w \<in> \<langle>{uu b,uu (\<not> b)}\<rangle>" and "\<^bold>|\<alpha>\<^bold>| < \<^bold>|uu a \<cdot> q\<^bold>|"
  shows "\<alpha> \<cdot> [cc a] \<le>p uu a \<cdot> q"
proof-
  have aux: "uu a \<cdot> w \<cdot> \<alpha> = (uu a \<cdot> q) \<cdot> (q\<inverse>\<^sup>>w \<cdot> \<alpha>)" "{uu b,uu (\<not> b)} = {u\<^sub>0,u\<^sub>1}"
    using lq_pref[OF \<open>q \<le>p w\<close>] bin_code_set_bool by force+
  have "\<^bold>|\<alpha> \<cdot> [cc a]\<^bold>| \<le> \<^bold>|uu a \<cdot> q\<^bold>|"
    using \<open>\<^bold>|\<alpha>\<^bold>| < \<^bold>|uu a \<cdot> q\<^bold>|\<close> by auto
  thus ?thesis
    using pref_prolong[OF bin_mismatch_bool  bin_lcp_pref_all_hull[OF \<open>w \<in> \<langle>{uu b,uu (\<not> b)}\<rangle>\<close>[unfolded aux]], of a]
    unfolding aux by blast
qed

lemmas bin_lcp_mismatch_pref_all_fst = bin_lcp_mismatch_pref_all_bool[of _ _ True True, unfolded bin_code_simps] and
       bin_lcp_mismatch_pref_all_snd = bin_lcp_mismatch_pref_all_bool[of _ _ True False, unfolded bin_code_simps]

lemma bin_lcp_pref_all_len: assumes "q \<le>p w" and "w \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>" and "\<^bold>|\<alpha>\<^bold>| \<le> \<^bold>|q\<^bold>|"
  shows "\<alpha> \<le>p q"
  using bin_lcp_pref_all_hull[OF \<open>w \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close>] pref_ext[OF \<open>q \<le>p w\<close>] prefix_length_prefix[OF _ _ \<open>\<^bold>|bin_code_lcp\<^bold>| \<le> \<^bold>|q\<^bold>|\<close>] by blast

lemma bin_mismatch_all_bool: assumes "xs \<in> lists {uu b, uu (\<not> b)}" shows "\<alpha> \<cdot> [cc a] \<le>p (uu a) \<cdot> concat xs \<cdot> \<alpha>"
  using pref_prolong[OF bin_mismatch_bool bin_lcp_pref_all, of xs a] assms unfolding  bin_code_set_bool[of b].

lemmas bin_fst_mismatch_all = bin_mismatch_all_bool[of _ True True, unfolded bin_code_simps] and
       bin_snd_mismatch_all = bin_mismatch_all_bool[of _ True False, unfolded bin_code_simps]

lemma bin_mismatch_all_hull_bool: assumes "w \<in> \<langle>{uu b,uu (\<not> b)}\<rangle>" shows "\<alpha> \<cdot> [cc a] \<le>p uu a \<cdot> w \<cdot> \<alpha>"
  using bin_mismatch_all_bool hull_concat_listsE[OF assms] by metis

lemmas bin_fst_mismatch_all_hull = bin_mismatch_all_hull_bool[of _ True True, unfolded bin_code_simps] and
       bin_snd_mismatch_all_hull = bin_mismatch_all_hull_bool[of _ True False, unfolded bin_code_simps]

lemma bin_mismatch_all_len_bool: assumes "q \<le>p uu a \<cdot> w" and "w \<in> \<langle>{uu b,uu (\<not> b)}\<rangle>" and "\<^bold>|\<alpha>\<^bold>| < \<^bold>|q\<^bold>|"
  shows "\<alpha> \<cdot> [cc a] \<le>p q"
proof-
  have "\<^bold>|\<alpha> \<cdot> [cc a]\<^bold>| \<le> \<^bold>|uu a \<cdot> w\<^bold>|" "\<^bold>|\<alpha> \<cdot> [cc a]\<^bold>| \<le> \<^bold>|q\<^bold>|"
    using less_le_trans[OF \<open>\<^bold>|\<alpha>\<^bold>| < \<^bold>|q\<^bold>|\<close> pref_len[OF \<open>q \<le>p uu a \<cdot> w\<close>]] \<open>\<^bold>|\<alpha>\<^bold>| < \<^bold>|q\<^bold>|\<close> by force+
  from pref_prod_le[OF bin_mismatch_all_hull_bool[OF assms(2), unfolded lassoc], OF this(1)]
  show ?thesis
    by (rule prefix_length_prefix) fact+
qed

lemmas bin_fst_mismatch_all_len = bin_mismatch_all_len_bool[of _ True _ True, unfolded bin_code_simps] and
       bin_snd_mismatch_all_len = bin_mismatch_all_len_bool[of _ False _ True, unfolded bin_code_simps]

lemma bin_code_delay: assumes "\<^bold>|\<alpha>\<^bold>| \<le> \<^bold>|q\<^sub>0\<^bold>|" and "\<^bold>|\<alpha>\<^bold>| \<le> \<^bold>|q\<^sub>1\<^bold>|" and
      "q\<^sub>0 \<le>p u\<^sub>0 \<cdot> w\<^sub>0" and "q\<^sub>1 \<le>p u\<^sub>1 \<cdot> w\<^sub>1" and
      "w\<^sub>0 \<in> \<langle>{u\<^sub>0, u\<^sub>1}\<rangle>" and "w\<^sub>1 \<in> \<langle>{u\<^sub>0, u\<^sub>1}\<rangle>"
  shows "q\<^sub>0 \<and>\<^sub>p q\<^sub>1 = \<alpha>"
proof-
  have p1: "\<alpha> \<cdot> [c\<^sub>0] \<le>p u\<^sub>0 \<cdot> w\<^sub>0 \<cdot> \<alpha>"
    using assms(5) using bin_fst_mismatch_all_hull by auto
  have p2: "\<alpha> \<cdot> [c\<^sub>1] \<le>p u\<^sub>1 \<cdot> w\<^sub>1 \<cdot> \<alpha>"
    using assms(6) using bin_snd_mismatch_all_hull by auto
  have lcp: "u\<^sub>0 \<cdot> w\<^sub>0 \<cdot> \<alpha> \<and>\<^sub>p u\<^sub>1 \<cdot> w\<^sub>1 \<cdot> \<alpha> = \<alpha>"
    using lcp_first_mismatch_pref[OF p1 p2 bin_mismatch_neq].
  from lcp_extend_eq[of "q\<^sub>0" "u\<^sub>0 \<cdot> w\<^sub>0 \<cdot> \<alpha>" "q\<^sub>1" "u\<^sub>1 \<cdot> w\<^sub>1 \<cdot> \<alpha>",
       unfolded this,OF _ _ assms(1-2)]
  show ?thesis
    using pref_ext[OF \<open>q\<^sub>0 \<le>p u\<^sub>0 \<cdot> w\<^sub>0\<close>] pref_ext[OF \<open>q\<^sub>1 \<le>p u\<^sub>1 \<cdot> w\<^sub>1\<close>] by force
qed

lemma hd_lq_mismatch_fst: "hd (\<alpha>\<inverse>\<^sup>>(u\<^sub>0 \<cdot> \<alpha>)) = c\<^sub>0"
  using hd_lq_conv_nth[OF prefix_snocD[OF bin_fst_mismatch]] bin_fst_mismatch
  by (auto simp add: prefix_def)

lemma hd_lq_mismatch_snd: "hd (\<alpha>\<inverse>\<^sup>>(u\<^sub>1 \<cdot> \<alpha>)) = c\<^sub>1"
  using hd_lq_conv_nth[OF prefix_snocD[OF bin_snd_mismatch]] bin_snd_mismatch
  by (auto simp add: prefix_def)

lemma hds_bin_mismatch_neq: "hd (\<alpha>\<inverse>\<^sup>>(u\<^sub>0 \<cdot> \<alpha>)) \<noteq> hd (\<alpha>\<inverse>\<^sup>>(u\<^sub>1 \<cdot> \<alpha>))"
  unfolding hd_lq_mismatch_fst hd_lq_mismatch_snd
  using bin_mismatch_neq.

lemma bin_lcp_fst_pow_pref: assumes "0 < k" shows "\<alpha> \<cdot> [c\<^sub>0] \<le>p u\<^sub>0\<^sup>@k \<cdot> u\<^sub>1 \<cdot> z"
  using assms
proof (induct k rule: nat_induct_non_zero)
  case 1
  then show ?case
    unfolding pow_list_1 using  pref_prolong[OF bin_fst_mismatch' triv_pref].
next
  case (Suc n)
  show ?case
    unfolding pow_Suc rassoc
    by (rule pref_prolong[OF bin_fst_mismatch])
    (use append_prefixD[OF Suc.hyps(2)] in blast)
qed

lemmas bin_lcp_snd_pow_pref = binary_code.bin_lcp_fst_pow_pref[OF bin_code_swap, unfolded bin_lcp_swap]

lemma bin_lcp_fst_lcp: "\<alpha> \<le>p u\<^sub>0 \<cdot> \<alpha>" and bin_lcp_snd_lcp: "\<alpha> \<le>p u\<^sub>1 \<cdot> \<alpha>"
  using pref_extD[OF bin_fst_mismatch]  pref_extD[OF bin_snd_mismatch].

lemma bin_lcp_pref_all_set: assumes "set ws = {u\<^sub>0,u\<^sub>1}"
  shows "\<alpha> \<le>p concat ws"
proof-
  have "ws \<in> lists {u\<^sub>0, u\<^sub>1}"
    using assms by blast
  have "\<^bold>|u\<^sub>0\<^bold>| + \<^bold>|u\<^sub>1\<^bold>| \<le> \<^bold>|concat ws\<^bold>|"
    using assms two_in_set_concat_len[OF bin_code_neq] by simp
  with pref_prod_le[OF bin_lcp_pref_all[OF \<open>ws \<in> lists {u\<^sub>0, u\<^sub>1}\<close>]] bin_lcp_short
  show ?thesis
    by simp
qed

lemma bin_lcp_conjug_morph:
  assumes "u \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>" and "v \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>"
  shows "\<alpha>\<inverse>\<^sup>>(u \<cdot> \<alpha>) \<cdot> \<alpha>\<inverse>\<^sup>>(v \<cdot> \<alpha>) = \<alpha>\<inverse>\<^sup>>((u \<cdot> v) \<cdot> \<alpha>)"
  unfolding lq_reassoc[OF bin_lcp_pref_all_hull[OF \<open>u \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close>]] rassoc
    lq_pref[OF bin_lcp_pref_all_hull[OF \<open>v \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close>]]..

lemma lcp_bin_conjug_prim_iff:
  "set ws = {u\<^sub>0,u\<^sub>1} \<Longrightarrow> primitive (\<alpha>\<inverse>\<^sup>>(concat ws) \<cdot> \<alpha>) \<longleftrightarrow> primitive (concat ws)"
  using conjug_prim_iff[OF root_conjug[OF pref_ext[OF bin_lcp_pref_all_set]], symmetric]
  unfolding lq_reassoc[OF bin_lcp_pref_all_set] by simp

lemma bin_lcp_conjug_inj_on: "inj_on (\<lambda>u. \<alpha>\<inverse>\<^sup>>(u \<cdot> \<alpha>)) \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>"
  unfolding inj_on_def using bin_lcp_pref_all_hull cancel_right lq_pref
  by metis

lemma bin_code_lcp_marked: assumes "us \<in> lists {u\<^sub>0,u\<^sub>1}" and "vs \<in> lists {u\<^sub>0,u\<^sub>1}" and "hd us \<noteq> hd vs"
  shows "concat us \<cdot> \<alpha> \<and>\<^sub>p concat vs \<cdot> \<alpha> = \<alpha>"
proof (cases "us = \<epsilon> \<or> vs = \<epsilon>")
  assume "us = \<epsilon> \<or> vs = \<epsilon>"
  thus ?thesis
    using append_self_conv2 assms(1) assms(2) bin_lcp_pref_all concat.simps(1) lcp_pref_conv lcp_sym by metis
next
  assume "\<not> (us = \<epsilon> \<or> vs = \<epsilon>)" hence "us \<noteq> \<epsilon>" and "vs \<noteq> \<epsilon>" by blast+
  have spec_case: "concat us \<cdot> \<alpha> \<and>\<^sub>p concat vs \<cdot> \<alpha> = \<alpha>" if "us \<in> lists {u\<^sub>0,u\<^sub>1}" and "vs \<in> lists {u\<^sub>0,u\<^sub>1}" and "hd us = u\<^sub>0" and "hd vs = u\<^sub>1" and "us \<noteq> \<epsilon>" and "vs \<noteq> \<epsilon>" for us vs
  proof-
    have "concat us = u\<^sub>0 \<cdot> concat (tl us)"
      unfolding hd_concat_tl[OF \<open>us \<noteq> \<epsilon>\<close>, symmetric] \<open>hd us = u\<^sub>0\<close>..
    from bin_fst_mismatch_all[OF tl_in_lists[OF \<open>us \<in> lists {u\<^sub>0,u\<^sub>1}\<close>], folded rassoc this]
    have pref1: "\<alpha> \<cdot> [c\<^sub>0] \<le>p concat us \<cdot> \<alpha>".
    have "concat vs = u\<^sub>1 \<cdot> concat (tl vs)"
      unfolding hd_concat_tl[OF \<open>vs \<noteq> \<epsilon>\<close>, symmetric] \<open>hd vs = u\<^sub>1\<close>..
    from bin_snd_mismatch_all[OF tl_in_lists[OF \<open>vs \<in> lists {u\<^sub>0,u\<^sub>1}\<close>], folded rassoc this]
    have pref2: "\<alpha> \<cdot> [c\<^sub>1] \<le>p concat vs \<cdot> \<alpha>".
    show ?thesis
      using  lcp_first_mismatch_pref[OF pref1 pref2 bin_mismatch_neq].
  qed
  have "hd us \<in>  {u\<^sub>0,u\<^sub>1}" and "hd vs \<in>  {u\<^sub>0,u\<^sub>1}" using
      lists_hd_in_set[OF \<open>us \<noteq> \<epsilon>\<close> \<open>us \<in> lists {u\<^sub>0, u\<^sub>1}\<close>] lists_hd_in_set[OF \<open>vs \<noteq> \<epsilon>\<close> \<open>vs \<in> lists {u\<^sub>0, u\<^sub>1}\<close>].
  then consider "hd us = u\<^sub>0 \<and> hd vs = u\<^sub>1" | "hd us = u\<^sub>1 \<and> hd vs = u\<^sub>0"
    using \<open>hd us \<noteq> hd vs\<close> by fastforce
  then show ?thesis
    using spec_case[rule_format] \<open>us \<noteq> \<epsilon>\<close> \<open>vs \<noteq> \<epsilon>\<close> assms lcp_sym by metis
qed

\<comment> \<open>ALT PROOF\<close>
lemma  assumes "us \<in> lists {u\<^sub>0,u\<^sub>1}" and "vs \<in> lists {u\<^sub>0,u\<^sub>1}" and "hd us \<noteq> hd vs"
  shows "concat us \<cdot> \<alpha> \<and>\<^sub>p concat vs \<cdot> \<alpha> = \<alpha>"
  using assms
proof (induct us vs rule: list_induct2')
  case (2 x xs)
  show ?case
    using bin_lcp_pref_all[OF \<open>x # xs \<in> lists {u\<^sub>0, u\<^sub>1}\<close>, folded lcp_pref_conv, unfolded lcp_sym[of \<alpha>]] by simp
next
  case (3 y ys)
  show ?case
    using bin_lcp_pref_all[OF \<open>y # ys \<in> lists {u\<^sub>0, u\<^sub>1}\<close>, folded lcp_pref_conv] by simp
next
  case (4 x xs y ys)
  interpret i: binary_code x y
    using "4.prems"(1) "4.prems"(2) "4.prems"(3) non_comm binary_code.intro by auto
  have alph: "{u\<^sub>0,u\<^sub>1} = {x,y}"
    using "4.prems"(1) "4.prems"(2) "4.prems"(3) by auto
  from disjE[OF this[unfolded doubleton_eq_iff]]
  have "i.bin_code_lcp = \<alpha>"
    using i.bin_lcp_swap[symmetric] by blast
  have c0: "i.bin_code_lcp \<cdot> [i.bin_code_mismatch_fst] \<le>p x \<cdot> concat xs \<cdot> i.bin_code_lcp"
    using  i.bin_lcp_pref_all[of xs] \<open>x # xs \<in> lists {u\<^sub>0, u\<^sub>1}\<close>[unfolded Cons_in_lists_iff alph]
      pref_prolong[OF i.bin_fst_mismatch] by blast
  have c1: "i.bin_code_lcp \<cdot> [i.bin_code_mismatch_snd] \<le>p y \<cdot> concat ys \<cdot> i.bin_code_lcp"
    using pref_prolong[OF conjunct2[OF \<open>y # ys \<in> lists {u\<^sub>0, u\<^sub>1}\<close>[unfolded      Cons_in_lists_iff alph],
          THEN i.bin_snd_mismatch_all[of ys]], OF self_pref].
  have "i.bin_code_lcp\<cdot>[i.bin_code_mismatch_fst] \<and>\<^sub>p i.bin_code_lcp\<cdot>[i.bin_code_mismatch_snd] = i.bin_code_lcp"
    by (simp add: i.bin_mismatch_neq lcp_first_mismatch')
  from lcp_rulers[OF c0 c1, unfolded this, unfolded bin_lcp_swap]
  show ?case
    unfolding concat.simps(2) rassoc using i.bin_mismatch_neq
      \<open>i.bin_code_lcp = \<alpha>\<close> by force
qed simp

lemma bin_code_lcp_concat: assumes "us \<in> lists {u\<^sub>0,u\<^sub>1}" and "vs \<in> lists {u\<^sub>0,u\<^sub>1}" and "\<not> us \<bowtie> vs"
  shows "concat us \<cdot> \<alpha> \<and>\<^sub>p concat vs \<cdot> \<alpha> = concat (us \<and>\<^sub>p vs) \<cdot> \<alpha>"
proof-
  obtain us' vs' where us: "(us \<and>\<^sub>p vs) \<cdot> us' = us" and vs: "(us \<and>\<^sub>p vs) \<cdot> vs' = vs" and "us' \<noteq> \<epsilon>" and "vs' \<noteq> \<epsilon>" and "hd us' \<noteq> hd vs'"
    using lcp_mismatchE[OF \<open>\<not> us \<bowtie> vs\<close>].
  have cu: "concat us \<cdot> \<alpha> = concat (us \<and>\<^sub>p vs) \<cdot> concat us' \<cdot> \<alpha>"
    unfolding lassoc concat_morph[symmetric] us..
  have cv: "concat vs \<cdot> \<alpha> = concat (us \<and>\<^sub>p vs) \<cdot> concat vs' \<cdot> \<alpha>"
    unfolding lassoc concat_morph[symmetric] vs..
  have "us' \<in> lists {u\<^sub>0,u\<^sub>1}"
    using \<open>us \<in> lists {u\<^sub>0,u\<^sub>1}\<close> us by inlists
  have "vs' \<in> lists {u\<^sub>0,u\<^sub>1}"
    using \<open>vs \<in> lists {u\<^sub>0,u\<^sub>1}\<close> vs by inlists
  show "concat us \<cdot> \<alpha> \<and>\<^sub>p concat vs \<cdot> \<alpha> = concat (us \<and>\<^sub>p vs) \<cdot> \<alpha>"
    unfolding cu cv
    using bin_code_lcp_marked[OF \<open>us' \<in> lists {u\<^sub>0,u\<^sub>1}\<close> \<open>vs' \<in> lists {u\<^sub>0,u\<^sub>1}\<close> \<open>hd us' \<noteq> hd vs'\<close>]
    unfolding lcp_ext_left by fast
qed

lemma bin_code_lcp_concat': assumes "us \<in> lists {u\<^sub>0,u\<^sub>1}" and "vs \<in> lists {u\<^sub>0,u\<^sub>1}" and "\<not> concat us \<bowtie> concat vs"
  shows "concat us \<and>\<^sub>p concat vs = concat (us \<and>\<^sub>p vs) \<cdot> \<alpha>"
  using bin_code_lcp_concat[OF assms(1-2)] assms(3) lcp_ext_right_conv pref_concat_pref prefix_comparable_def by metis

lemma bin_lcp_pows:  "0 < k \<Longrightarrow> 0 < l \<Longrightarrow> u\<^sub>0\<^sup>@k \<cdot> u\<^sub>1 \<cdot> z \<and>\<^sub>p u\<^sub>1\<^sup>@l \<cdot> u\<^sub>0 \<cdot> z' = \<alpha>"
  using lcp_first_mismatch_pref[OF bin_lcp_fst_pow_pref bin_lcp_snd_pow_pref bin_mismatch_neq].

theorem bin_code: assumes "us \<in> lists {u\<^sub>0,u\<^sub>1}" and "vs \<in> lists {u\<^sub>0,u\<^sub>1}" and "concat us = concat vs"
  shows "us = vs"
  using assms
proof (induct us vs rule: list_induct2')
  case (4 x xs y ys)
  then show ?case
  proof-
    have "x =y"
      using bin_code_lcp_marked[OF \<open>x # xs \<in> lists {u\<^sub>0, u\<^sub>1}\<close> \<open>y # ys \<in> lists {u\<^sub>0, u\<^sub>1}\<close>] \<open>y # ys \<in> lists {u\<^sub>0, u\<^sub>1}\<close> non_comm
      unfolding \<open>concat (x # xs) = concat (y # ys)\<close> unfolding concat.simps(2) lcp_self list.sel(1)
      by auto
     thus "x # xs = y # ys"
      using "4.hyps" \<open>concat (x # xs) = concat (y # ys)\<close>[unfolded  concat.simps(2) \<open>x = y\<close>, unfolded cancel]
        \<open>y # ys \<in> lists {u\<^sub>0, u\<^sub>1}\<close>[unfolded Cons_in_lists_iff] \<open>x # xs \<in> lists {u\<^sub>0, u\<^sub>1}\<close>[unfolded Cons_in_lists_iff]
      by simp
  qed
qed (auto simp: bin_fst_nemp bin_snd_nemp)

lemma code_bin_roots: "binary_code (\<rho> u\<^sub>0) (\<rho> u\<^sub>1)"
  using non_comm comm_primroot_conv'  by (unfold_locales) blast

sublocale code "{u\<^sub>0,u\<^sub>1}"
  using bin_code by unfold_locales

lemma primroot_dec: "(Dec {\<rho> u\<^sub>0, \<rho> u\<^sub>1} u\<^sub>0) = [\<rho> u\<^sub>0]\<^sup>@e\<^sub>\<rho> u\<^sub>0" "(Dec {\<rho> u\<^sub>0, \<rho> u\<^sub>1} u\<^sub>1) = [\<rho> u\<^sub>1]\<^sup>@e\<^sub>\<rho> u\<^sub>1"
proof-
  interpret rs: binary_code "\<rho> u\<^sub>0" "\<rho> u\<^sub>1"
    by (simp add: code_bin_roots)
  from primroot_exp_eq
  have "concat ([\<rho> u\<^sub>0]\<^sup>@e\<^sub>\<rho> u\<^sub>0) = u\<^sub>0" "concat ([\<rho> u\<^sub>1]\<^sup>@e\<^sub>\<rho> u\<^sub>1) = u\<^sub>1"
    by force+
  from rs.code_unique_dec[OF _ this(1)] rs.code_unique_dec[OF _ this(2)]
  show "(Dec {\<rho> u\<^sub>0, \<rho> u\<^sub>1} u\<^sub>0) = [\<rho> u\<^sub>0]\<^sup>@e\<^sub>\<rho> u\<^sub>0" "(Dec {\<rho> u\<^sub>0, \<rho> u\<^sub>1} u\<^sub>1) = [\<rho> u\<^sub>1]\<^sup>@e\<^sub>\<rho> u\<^sub>1"
    by (simp_all add: sing_pow_lists)
qed

lemma bin_code_prefs: assumes "w \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>" and "p \<le>p w" "w' \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>" and "\<^bold>|u\<^sub>1\<^bold>| \<le> \<^bold>|p\<^bold>|"
  shows " \<not> u\<^sub>0 \<cdot>  p \<le>p u\<^sub>1 \<cdot> w'"
proof
  assume contr: "u\<^sub>0 \<cdot> p  \<le>p u\<^sub>1 \<cdot> w'"
  have "\<^bold>|\<alpha>\<^bold>| < \<^bold>|u\<^sub>0 \<cdot> p\<^bold>|"
    using \<open>\<^bold>|u\<^sub>1\<^bold>| \<le> \<^bold>|p\<^bold>|\<close> bin_lcp_short by auto
  hence "\<alpha> \<cdot> [c\<^sub>0] \<le>p u\<^sub>0 \<cdot> p"
    using \<open>p \<le>p w\<close>  \<open>w \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close> bin_lcp_mismatch_pref_all_fst by auto
  from pref_ext[OF pref_trans[OF this contr], unfolded rassoc]
  have "\<alpha> \<cdot> [c\<^sub>0] \<le>p u\<^sub>1 \<cdot> w' \<cdot> \<alpha>".
  from bin_mismatch_neq same_sing_pref[OF bin_snd_mismatch_all_hull[OF \<open>w' \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close>] this]
  show False
    by simp
qed

lemma bin_code_rev: "binary_code (rev u\<^sub>0) (rev u\<^sub>1)"
  by (unfold_locales, unfold comm_rev_iff, simp add: non_comm)


lemma bin_lcp_pows_lcp:  "0 < k \<Longrightarrow> 0 < l \<Longrightarrow> u\<^sub>0\<^sup>@k \<cdot> u\<^sub>1\<^sup>@l \<and>\<^sub>p u\<^sub>1\<^sup>@l \<cdot> u\<^sub>0\<^sup>@k = u\<^sub>0 \<cdot> u\<^sub>1 \<and>\<^sub>p u\<^sub>1 \<cdot> u\<^sub>0"
   using bin_lcp_pows unfolding bin_lcp_def using pow_pos by metis

lemma bin_mismatch: "u\<^sub>0 \<cdot> \<alpha> \<and>\<^sub>p u\<^sub>1 \<cdot> \<alpha> = \<alpha>"
  using lcp_first_mismatch_pref[OF bin_fst_mismatch bin_snd_mismatch bin_mismatch_neq].

lemma not_comp_bin_fst_snd: "\<not> u\<^sub>0 \<cdot> \<alpha> \<bowtie> u\<^sub>1 \<cdot>  \<alpha>"
  using ruler_comp[OF bin_fst_mismatch bin_snd_mismatch] bin_mismatch_neq
  unfolding prefix_comparable_def pref_cancel_conv by force

theorem bin_bounded_delay: assumes "z \<le>p u\<^sub>0 \<cdot> w\<^sub>0" and "z \<le>p u\<^sub>1 \<cdot> w\<^sub>1"
  and "w\<^sub>0 \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>" and "w\<^sub>1 \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>"
shows "\<^bold>|z\<^bold>| \<le> \<^bold>|\<alpha>\<^bold>|"
proof (rule leI, rule notI)
  assume "\<^bold>|\<alpha>\<^bold>| < \<^bold>|z\<^bold>|"
  hence "\<^bold>|\<alpha> \<cdot> [a]\<^bold>| \<le> \<^bold>|z\<^bold>|" for a
    unfolding lenmorph sing_len by simp
  have "z \<le>p u\<^sub>0 \<cdot> w\<^sub>0 \<cdot> \<alpha>" and "z \<le>p u\<^sub>1 \<cdot> w\<^sub>1 \<cdot> \<alpha>"
    using  pref_prolong[OF \<open>z \<le>p u\<^sub>0 \<cdot> w\<^sub>0\<close> triv_pref] pref_prolong[OF \<open>z \<le>p u\<^sub>1 \<cdot> w\<^sub>1\<close> triv_pref].
  have "\<alpha> \<cdot> [c\<^sub>0] \<le>p u\<^sub>0 \<cdot> w\<^sub>0 \<cdot> \<alpha>" and "\<alpha> \<cdot> [c\<^sub>1] \<le>p u\<^sub>1 \<cdot> w\<^sub>1 \<cdot> \<alpha>"
    using bin_fst_mismatch_all_hull[OF \<open>w\<^sub>0 \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close>] bin_snd_mismatch_all_hull[OF \<open>w\<^sub>1 \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close>].
  from \<open>z \<le>p u\<^sub>0 \<cdot> w\<^sub>0 \<cdot> \<alpha>\<close> \<open>\<alpha> \<cdot> [c\<^sub>0] \<le>p u\<^sub>0 \<cdot> w\<^sub>0 \<cdot> \<alpha>\<close> \<open>\<^bold>|\<alpha> \<cdot> [c\<^sub>0]\<^bold>| \<le> \<^bold>|z\<^bold>|\<close>
  have "\<alpha> \<cdot> [c\<^sub>0] \<le>p z"
    using prefix_length_prefix by blast
  from \<open>z \<le>p u\<^sub>1 \<cdot> w\<^sub>1 \<cdot> \<alpha>\<close> \<open>\<alpha> \<cdot> [c\<^sub>1] \<le>p u\<^sub>1 \<cdot> w\<^sub>1 \<cdot> \<alpha>\<close> \<open>\<^bold>|\<alpha> \<cdot> [c\<^sub>1]\<^bold>| \<le> \<^bold>|z\<^bold>|\<close>
  have "\<alpha> \<cdot> [c\<^sub>1] \<le>p z"
    using prefix_length_prefix by blast
  from \<open>\<alpha> \<cdot> [c\<^sub>1] \<le>p z\<close> \<open>\<alpha> \<cdot> [c\<^sub>0] \<le>p z\<close> bin_mismatch_neq
  show False
    unfolding prefix_def by force
qed

thm binary_code.bin_lcp_pows_lcp

lemma prim_roots_lcp: "bin_lcp  (\<rho> u\<^sub>0)  (\<rho> u\<^sub>1)  = \<alpha>"
proof-
  obtain k where "\<rho> u\<^sub>0\<^sup>@k = u\<^sub>0" "0 < k"
    using primroot_expE.
  obtain m where "\<rho> u\<^sub>1\<^sup>@m = u\<^sub>1" "0 < m"
    using primroot_expE.
  have "\<rho> u\<^sub>0 \<cdot> \<rho> u\<^sub>1 \<noteq> \<rho> u\<^sub>1 \<cdot> \<rho> u\<^sub>0"
    using non_comm[unfolded comm_primroot_conv'[of u\<^sub>0]].
  then interpret r: binary_code "\<rho> u\<^sub>0" "\<rho> u\<^sub>1" by unfold_locales
  from r.bin_lcp_pows_lcp[OF \<open>0 < k\<close> \<open>0 < m\<close>, unfolded \<open>\<rho> u\<^sub>1\<^sup>@m = u\<^sub>1\<close> \<open>\<rho> u\<^sub>0\<^sup>@k = u\<^sub>0\<close>, symmetric]
  show ?thesis
     unfolding bin_lcp_def.
qed

lemma bin_roots_decompose:
        "Dec {\<rho> u\<^sub>0, u\<^sub>1} u\<^sub>0 = [\<rho> u\<^sub>0]\<^sup>@e\<^sub>\<rho> u\<^sub>0"
        "Dec {\<rho> u\<^sub>0, u\<^sub>1} u\<^sub>1 = [u\<^sub>1]"
        "Dec {u\<^sub>0, \<rho> u\<^sub>1} u\<^sub>1 = [\<rho> u\<^sub>1]\<^sup>@e\<^sub>\<rho> u\<^sub>1"
        "Dec {u\<^sub>0, \<rho> u\<^sub>1} u\<^sub>0 = [u\<^sub>0]"
        "Dec {u\<^sub>0,u\<^sub>1} u\<^sub>0 = [u\<^sub>0]"
        "Dec {u\<^sub>0,u\<^sub>1} u\<^sub>1 = [u\<^sub>1]"
proof-
  show "Dec {u\<^sub>0,u\<^sub>1} u\<^sub>0 = [u\<^sub>0]" "Dec {u\<^sub>0,u\<^sub>1} u\<^sub>1 = [u\<^sub>1]"
    using code_el_dec by simp_all
  interpret r: binary_code "\<rho> u\<^sub>0" u\<^sub>1
    using non_comm unfolding binary_code_def using comm_primroot_conv[of u\<^sub>1 u\<^sub>0]
    by presburger
  interpret r': binary_code u\<^sub>0 "\<rho> u\<^sub>1"
    using non_comm unfolding binary_code_def using comm_primroot_conv[of u\<^sub>0 u\<^sub>1]
    by presburger
  from r.code_el_dec
  show "Dec {\<rho> u\<^sub>0, u\<^sub>1} u\<^sub>1 = [u\<^sub>1]"
    by force
  from r'.code_el_dec
  show "Dec {u\<^sub>0, \<rho> u\<^sub>1} u\<^sub>0 = [u\<^sub>0]"
    by force
  show "Dec {\<rho> u\<^sub>0, u\<^sub>1} u\<^sub>0 = [\<rho> u\<^sub>0]\<^sup>@e\<^sub>\<rho> u\<^sub>0"
    using r.code_unique_dec'[OF sing_pow_lists[of "\<rho> u\<^sub>0" "{\<rho> u\<^sub>0, u\<^sub>1}" "e\<^sub>\<rho> u\<^sub>0"]]
    unfolding concat_pow_list concat_sing' primroot_exp_eq by simp
  show "Dec {u\<^sub>0, \<rho> u\<^sub>1} u\<^sub>1 = [\<rho> u\<^sub>1] \<^sup>@ e\<^sub>\<rho> u\<^sub>1"
    using r'.code_unique_dec'[OF sing_pow_lists[of "\<rho> u\<^sub>1" "{u\<^sub>0, \<rho> u\<^sub>1}" "e\<^sub>\<rho> u\<^sub>1"]]
    unfolding concat_pow_list concat_sing' primroot_exp_eq by simp
qed

lemma ref_fst_sq: "Ref {\<rho> u\<^sub>0, u\<^sub>1}[u\<^sub>0,u\<^sub>0] = [\<rho> u\<^sub>0]\<^sup>@(e\<^sub>\<rho> u\<^sub>0 * 2)"
  unfolding refine_def pow_mult pow_list_2
  using bin_roots_decompose by simp

lemma ref_fst_pow: "Ref {\<rho> u\<^sub>0, u\<^sub>1}[u\<^sub>0]\<^sup>@k = [\<rho> u\<^sub>0]\<^sup>@(e\<^sub>\<rho> u\<^sub>0 * k)"
  unfolding refine_def pow_mult pow_list_2
  using bin_roots_decompose by simp

lemma bin_code_concat_len: assumes "ws \<in> lists {u\<^sub>0,u\<^sub>1}"
  shows "\<^bold>|concat ws\<^bold>| = count_list ws u\<^sub>0 * \<^bold>|u\<^sub>0\<^bold>| + count_list ws u\<^sub>1 * \<^bold>|u\<^sub>1\<^bold>|"
  using bin_len_concat[OF bin_code_neq assms].

subsubsection \<open>Maximal r-prefixes\<close>

lemma bin_lcp_per_root_max_pref_short:  assumes "\<alpha> <p u\<^sub>0 \<cdot> u\<^sub>1 \<and>\<^sub>p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1" and "r \<noteq> \<epsilon>" and "q \<le>p w" and "w \<in> \<langle>{u\<^sub>0, u\<^sub>1}\<rangle>"
  shows "u\<^sub>1 \<cdot> q \<and>\<^sub>p r \<cdot> u\<^sub>1 \<cdot> q = take \<^bold>|u\<^sub>1 \<cdot> q\<^bold>| \<alpha>"
proof-
  have "q \<bowtie> \<alpha>"
    using bin_lcp_pref_all_hull[OF \<open>w \<in> \<langle>{u\<^sub>0, u\<^sub>1}\<rangle>\<close>] ruler_comp[OF \<open>q \<le>p w\<close>, of \<alpha> "w \<cdot> \<alpha>"] by blast
  hence comp1: "u\<^sub>1 \<cdot> q \<bowtie> \<alpha> \<cdot> [c\<^sub>1]"
    using  ruler_comp[OF self_pref bin_snd_mismatch, of "u\<^sub>1 \<cdot> q"] unfolding comp_cancel by blast

  from add_nth_pref[OF assms(1), THEN pref_lcp_pref] bin_fst_mismatch'
  have "(u\<^sub>0 \<cdot> u\<^sub>1 \<and>\<^sub>p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1) ! \<^bold>|\<alpha>\<^bold>| = c\<^sub>0"
    using same_sing_pref by fast

  from add_nth_pref[OF assms(1), unfolded this]
  have "\<alpha> \<cdot> [c\<^sub>0] \<le>p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1"
    by force

  have len: "\<^bold>|\<alpha> \<cdot> [c\<^sub>0]\<^bold>| \<le> \<^bold>|r \<cdot> \<alpha>\<^bold>|"
    using nemp_len[OF \<open>r \<noteq> \<epsilon>\<close>] unfolding lenmorph sing_len by linarith

  have comp2: "r \<cdot> u\<^sub>1 \<cdot> q \<bowtie> \<alpha> \<cdot> [c\<^sub>0]"
  proof(rule ruler_comp[OF _ _ comp_refl])
    show "r \<cdot> u\<^sub>1 \<cdot> q \<le>p r \<cdot> u\<^sub>1 \<cdot> w \<cdot> \<alpha>"
      using \<open>q \<le>p w\<close> by fastforce
    show "\<alpha> \<cdot> [c\<^sub>0] \<le>p r \<cdot> u\<^sub>1 \<cdot> w \<cdot> \<alpha>"
    proof(rule pref_prolong)
      show "\<alpha> \<cdot> [c\<^sub>0] \<le>p r \<cdot> \<alpha>"
        using \<open>\<alpha> \<cdot> [c\<^sub>0] \<le>p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1\<close> bin_lcp_pref len pref_prod_pref_short by blast
      show "\<alpha> \<le>p u\<^sub>1 \<cdot> w \<cdot> \<alpha>"
        using \<open>w \<in> \<langle>{u\<^sub>0, u\<^sub>1}\<rangle>\<close> bin_lcp_pref_all_hull[of "u\<^sub>1 \<cdot> w"] by auto
    qed
  qed

  have min: "(min \<^bold>|u\<^sub>1 \<cdot> q\<^bold>| \<^bold>|r \<cdot> u\<^sub>1 \<cdot> q\<^bold>|) = \<^bold>|u\<^sub>1 \<cdot> q\<^bold>|"
    unfolding lenmorph by simp

  show ?thesis
    using  bin_mismatch_neq double_ruler[OF comp1 comp2,unfolded min]
    by (simp add: lcp_mismatch_eq_len mismatch_incopm)
qed

lemma bin_per_root_max_pref_short:  assumes "(u\<^sub>0 \<cdot> u\<^sub>1) <p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1" and "q \<le>p w" and "w \<in> \<langle>{u\<^sub>0, u\<^sub>1}\<rangle>"
  shows "u\<^sub>1 \<cdot> q \<and>\<^sub>p r \<cdot> u\<^sub>1 \<cdot> q = take \<^bold>|u\<^sub>1 \<cdot> q\<^bold>| \<alpha>"
proof (rule bin_lcp_per_root_max_pref_short[OF _ _ assms(2-3)])
  show "\<alpha> <p u\<^sub>0 \<cdot> u\<^sub>1 \<and>\<^sub>p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1"
    unfolding lcp.absorb3[OF assms(1)] using bin_fst_mismatch'[THEN prefix_snocD].
qed (use assms(1) in blast)

lemma bin_root_max_pref_long:  assumes "r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1 =  u\<^sub>0 \<cdot> u\<^sub>1 \<cdot> r" and "q \<le>p w" and "w \<in> \<langle>{u\<^sub>0, u\<^sub>1}\<rangle>" and "\<^bold>|\<alpha>\<^bold>| \<le> \<^bold>|q\<^bold>|"
  shows "u\<^sub>0 \<cdot> \<alpha>  \<le>p u\<^sub>0 \<cdot> q \<and>\<^sub>p r \<cdot> u\<^sub>0 \<cdot> q"
proof (rule pref_pref_lcp)
  have len: " \<^bold>|u\<^sub>0 \<cdot> \<alpha>\<^bold>| \<le> \<^bold>|r \<cdot> u\<^sub>0 \<cdot> \<alpha>\<^bold>|"
    by simp
  from bin_lcp_pref_all_len[OF assms(2-4)]
  show "u\<^sub>0 \<cdot> \<alpha> \<le>p u\<^sub>0 \<cdot> q"
    unfolding pref_cancel_conv.
  have "u\<^sub>0 \<cdot> \<alpha> \<le>p r \<cdot> u\<^sub>0 \<cdot> \<alpha>"
  proof(rule ruler_le[OF _ _ len])
    show "u\<^sub>0 \<cdot> \<alpha> \<le>p (r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1) \<cdot> u\<^sub>0 \<cdot> u\<^sub>1"
      unfolding assms(1) unfolding rassoc pref_cancel_conv assms(1)
      using pref_ext[OF pref_ext[OF bin_lcp_pref'], unfolded rassoc].
    show "r \<cdot> u\<^sub>0 \<cdot> \<alpha> \<le>p (r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1) \<cdot> u\<^sub>0 \<cdot> u\<^sub>1"
      unfolding rassoc pref_cancel_conv using pref_ext[OF bin_lcp_pref', unfolded rassoc].
  qed
  from pref_prolong[OF this[unfolded lassoc], OF \<open>\<alpha> \<le>p q\<close>, unfolded rassoc]
  show "u\<^sub>0 \<cdot> \<alpha> \<le>p r \<cdot> u\<^sub>0 \<cdot> q".
qed

lemma per_root_lcp_per_root: "u\<^sub>0 \<cdot> u\<^sub>1 <p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1 \<Longrightarrow> \<alpha> \<cdot> [c\<^sub>0] \<le>p r \<cdot> \<alpha>"
  using per_root_pref_sing[OF _ bin_fst_mismatch'].

lemma per_root_bin_fst_snd_lcp:  assumes "u\<^sub>0 \<cdot> u\<^sub>1 <p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1" and
                   "q \<le>p w" and "w \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>" and "\<^bold>|\<alpha>\<^bold>| < \<^bold>|u\<^sub>1 \<cdot> q\<^bold>|"
                   "q' \<le>p w'" and "w' \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>" and "\<^bold>|\<alpha>\<^bold>| \<le> \<^bold>|q'\<^bold>|"
  shows "u\<^sub>1 \<cdot> q \<and>\<^sub>p r \<cdot> q' = \<alpha>"
proof-
  have pref1: "\<alpha> \<cdot> [c\<^sub>1] \<le>p u\<^sub>1 \<cdot> q"
    using \<open>\<^bold>|\<alpha>\<^bold>| < \<^bold>|u\<^sub>1 \<cdot> q\<^bold>|\<close> \<open>q \<le>p w\<close> bin_snd_mismatch_all_len[of "u\<^sub>1 \<cdot> q", OF _ \<open>w \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close>]
    unfolding pref_cancel_conv by blast

  have "\<alpha> \<le>p q'"
    using \<open>\<^bold>|\<alpha>\<^bold>| \<le> \<^bold>|q'\<^bold>|\<close> \<open>q' \<le>p w'\<close> \<open>w' \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close> bin_lcp_pref_all_len by blast
  have pref2: "\<alpha> \<cdot> [c\<^sub>0] \<le>p r \<cdot> \<alpha>"
    using assms(1) per_root_lcp_per_root by auto
  hence pref2: "\<alpha> \<cdot> [c\<^sub>0] \<le>p r \<cdot> q'"
    using \<open>\<alpha> \<le>p q'\<close> pref_prolong by blast

  show ?thesis
    using lcp_first_mismatch_pref[OF pref1 pref2 bin_mismatch_neq[symmetric]].

qed






end
text \<open>lemmas allowing to translate properties of binary code to its roots\<close>

named_theorems bin_code_primroots
lemma bin_lcp_eq_primroots [bin_code_primroots]: assumes "x \<cdot> y \<noteq> y \<cdot> x"
  shows "bin_lcp (\<rho> x) (\<rho> y) = bin_lcp x y"
using binary_code.prim_roots_lcp[OF binary_code.intro[OF assms]].

lemma bin_lcs_eq_primroots [bin_code_primroots]: assumes "x \<cdot> y \<noteq> y \<cdot> x"
  shows "bin_lcs (\<rho> x) (\<rho> y) = bin_lcs x y"
  using bin_lcp_eq_primroots[reversed, OF assms[symmetric]].

lemma bin_mismatch_fst_eq_primroots [bin_code_primroots]: assumes "x \<cdot> y \<noteq> y \<cdot> x"
  shows "bin_mismatch (\<rho> x) (\<rho> y) = bin_mismatch x y"
proof-
  interpret binary_code x y
    using assms by unfold_locales
  interpret r: binary_code "\<rho> x" "\<rho> y"
    using assms  by unfold_locales (simp add: comm_primroot_conv'[of x])
  have "r.bin_code_lcp = bin_code_lcp"
    using prim_roots_lcp.
  have "bin_lcp (\<rho> x) (\<rho> y) \<cdot> [bin_mismatch (\<rho> x) (\<rho> y)] = bin_lcp x y \<cdot> [bin_mismatch x y]"
  proof (rule ruler_eq_len[OF _ bin_fst_mismatch], unfold prim_roots_lcp[symmetric])
    show " r.bin_code_lcp \<cdot> [r.bin_code_mismatch_fst] \<le>p x \<cdot> r.bin_code_lcp"
      by (subst (3) pop_primroot[of x], unfold rassoc, rule r.bin_fst_mismatch_all_hull, blast)
  qed force
  then show ?thesis
  unfolding bin_lcp_eq_primroots[OF assms] cancel by blast
qed

lemmas bin_mismatch_snd_eq_primroots[bin_code_primroots] = bin_mismatch_fst_eq_primroots[OF not_sym] and
       bin_mismatch_suf_fst_eq_primroots[bin_code_primroots] = bin_mismatch_fst_eq_primroots[reversed] and
       bin_mismatch_suf_snd_primroots[bin_code_primroots] = bin_mismatch_fst_eq_primroots[reversed, OF not_sym]

lemma bin_lcp_eq_primroots' [bin_code_primroots]: "x \<cdot> y \<noteq> y \<cdot> x \<Longrightarrow> \<rho> x \<cdot> \<rho> y \<and>\<^sub>p \<rho> y \<cdot> \<rho> x = x \<cdot> y \<and>\<^sub>p y \<cdot> x"
  using bin_lcp_eq_primroots unfolding bin_lcp_def.

lemmas no_comm_bin_code = binary_code.bin_code[unfolded binary_code_def]

theorem bin_code_code[intro]: assumes "u \<cdot> v \<noteq> v \<cdot> u" shows "code {u, v}"
  unfolding code_def using no_comm_bin_code[OF assms] by blast

lemma code_bin_code: "u \<noteq> v \<Longrightarrow> code {u,v} \<Longrightarrow> u \<cdot> v \<noteq> v \<cdot> u"
  using code.code_comm_eq[of "{u,v}" u v] by blast

lemma lcp_roots_lcp: "x \<cdot> y \<noteq> y \<cdot> x \<Longrightarrow> x \<cdot> y \<and>\<^sub>p y \<cdot> x = \<rho> x \<cdot> \<rho> y \<and>\<^sub>p \<rho> y \<cdot> \<rho> x"
  using binary_code.prim_roots_lcp[unfolded binary_code_def bin_lcp_def, symmetric].

lemma sing_gen_primroot [simp]: "u \<in> \<langle>{\<rho> u}\<rangle>"
  unfolding sing_gen_pow_conv by simp

lemma sing_gen_pref_cancel [elim]: " u \<cdot> v \<in> \<langle>{r}\<rangle> \<Longrightarrow>  u \<in> \<langle>{r}\<rangle> \<Longrightarrow> v \<in> \<langle>{r}\<rangle>"
  using exp_pref_cancel[of _ r v] unfolding sing_gen_pow_ex_conv by blast

lemma sing_gen_suf_cancel [elim]: " u \<cdot> v \<in> \<langle>{r}\<rangle> \<Longrightarrow>  v \<in> \<langle>{r}\<rangle> \<Longrightarrow> u \<in> \<langle>{r}\<rangle>"
  using exp_suf_cancel[of u _ r] unfolding sing_gen_pow_ex_conv by blast

lemma prim_comm_root[elim]: assumes "primitive r" and "u \<cdot> r = r \<cdot> u" shows "u \<in> \<langle>{r}\<rangle>"
  using \<open>u\<cdot>r = r\<cdot>u\<close>[unfolded comm] prim_exp_eq[OF \<open>primitive r\<close>] pow_sing_gen  by metis

lemma prim_root_drop_exp[elim]: assumes "u\<^sup>@k \<in> \<langle>{r}\<rangle>" and "0 < k" and  "primitive r"
  shows "u \<in> \<langle>{r}\<rangle>"
  using pow_list_comm_comm[of k u _ r, OF \<open>0 < k\<close>, THEN prim_comm_root[OF \<open>primitive r\<close>]]
    \<open>u\<^sup>@k \<in> \<langle>{r}\<rangle>\<close> sing_gen_power by blast

lemma per_root_trans[intro]: assumes "w <p u \<cdot> w" and "u \<in> \<langle>{t}\<rangle>" shows "w <p t \<cdot> w"
   by (rule  sing_genE[OF \<open>u \<in> \<langle>{t}\<rangle>\<close> per_root_drop_exp]) (use \<open>w <p u \<cdot> w\<close> in blast)

lemma per_root_trans'[intro]: "w \<le>p u \<cdot> w \<Longrightarrow> u \<in> \<langle>{r}\<rangle> \<Longrightarrow> u \<noteq> \<epsilon> \<Longrightarrow> w \<le>p r \<cdot> w"
  using per_root_trans sprefD1 per_rootI by metis

lemmas per_root_trans_suf'[intro] = per_root_trans'[reversed]

lemma per_root_pref: "w \<noteq> \<epsilon> \<Longrightarrow> w \<in> \<langle>{r}\<rangle> \<Longrightarrow> r \<le>p w"
  by (rule hull.cases) blast+

lemmas per_root_suf = per_root_pref[reversed]

lemma comm_primrootE: assumes  "x \<cdot> y = y \<cdot> x"
  obtains t where "x \<in> \<langle>{t}\<rangle>" and "y \<in> \<langle>{t}\<rangle>" and "primitive t"
  using comm_primroots assms prim_sing  primroot_prim
   emp_in sing_gen_primroot by metis

lemma root_trans[trans]: "\<lbrakk>v \<in> \<langle>{u}\<rangle>; u \<in>\<langle>{t}\<rangle>\<rbrakk> \<Longrightarrow> v \<in> \<langle>{t}\<rangle>"
  by (metis sing_gen_pow_ex_conv pow_mult)


lemma root_rev_iff[reversal_rule]: "((rev u) \<in> \<langle>{rev t}\<rangle>) \<longleftrightarrow> (u \<in> \<langle>{t}\<rangle>)"
  unfolding sing_gen_pow_ex_conv rev_pow[symmetric] by blast

section \<open>Two words hull (not necessarily a code)\<close>

lemma bin_lists_len_count: assumes "x \<noteq> y" and "ws \<in> lists {x,y}" shows
  "count_list ws x + count_list ws y = \<^bold>|ws\<^bold>|"
proof-
  have "finite {x,y}" by simp
  have "set ws \<subseteq> {x,y}" using \<open>ws \<in> lists{x,y}\<close> by blast
  show ?thesis
    using sum_count_set[OF \<open>set ws \<subseteq> {x,y}\<close> \<open>finite {x,y}\<close>] \<open>x \<noteq> y\<close> by simp
qed

lemma two_elem_first_block: assumes "w \<in> \<langle>{u,v}\<rangle>"
  obtains m where "u\<^sup>@m \<cdot> v \<le>p w \<or> w = u\<^sup>@m"
  using assms
proof (induction rule: hull.induct)
  case (prod_cl w1 w2)
  show ?case
  proof (rule two_elem_cases[OF \<open>w1 \<in> {u,v}\<close>])
    assume "w1 = u"
    from  prod_cl.IH[OF prod_cl.prems, of Suc, unfolded this pow_Suc]
    show thesis
      by simp
  next
    assume "w1 = v"
    from  prod_cl.IH[OF prod_cl.prems, of "\<lambda> _.0", unfolded pow_zero this]
    show thesis
      by simp
  qed
qed force

lemmas two_elem_last_block = two_elem_first_block[reversed]

lemma two_elem_pref: assumes  "v \<le>p u \<cdot> p" and "p \<in> \<langle>{u,v}\<rangle>"
  shows "v \<le>p u \<cdot> v"
proof (rule two_elem_first_block[OF \<open>p \<in> \<langle>{u,v}\<rangle>\<close>], erule disjE)
  show "v \<le>p u \<cdot> v" if "u \<^sup>@ m \<cdot> v \<le>p p" for m
  proof-
    have "u\<^sup>@(Suc m) \<cdot> v \<le>p u \<cdot> p"
     unfolding pow_Suc rassoc  pref_cancel_conv by fact
  from ruler_le[OF \<open>v \<le>p u \<cdot> p\<close> this]
  have "v \<le>p u \<^sup>@ Suc m \<cdot> v"
    unfolding lenmorph by linarith
  from per_drop_exp[OF zero_less_Suc this]
  show "v \<le>p u \<cdot> v".
   qed
  show "v \<le>p u \<cdot> v" if "p = u \<^sup>@ m" for m
    using \<open>v \<le>p u \<cdot> p\<close>[unfolded that, folded pow_Suc]
    by (rule pref_pow_root)
qed

lemmas two_elem_suf = two_elem_pref[reversed]

lemma gen_drop_exp: assumes "p \<in> \<langle>{u,v\<^sup>@(Suc k)}\<rangle>" shows "p \<in> \<langle>{u,v}\<rangle>"
  by (rule hull.induct[OF assms], simp, blast)

lemma gen_drop_exp_pos: assumes "p \<in> \<langle>{u,v\<^sup>@k}\<rangle>" "0 < k" shows "p \<in> \<langle>{u,v}\<rangle>"
  using gen_drop_exp[of _ _ "k-1", unfolded Suc_minus_pos[OF \<open>0 < k\<close>], OF \<open>p \<in> \<langle>{u,v\<^sup>@k}\<rangle>\<close>].

lemma gen_prim: "p \<in> \<langle>{u,v}\<rangle> \<Longrightarrow> p \<in> \<langle>{u,\<rho> v}\<rangle>"
  using gen_drop_exp_pos primroot_expE by metis

lemma roots_hull: assumes "w \<in> \<langle>{u\<^sup>@k,v\<^sup>@m}\<rangle>" shows "w \<in> \<langle>{u,v}\<rangle>"
proof-
  have "u\<^sup>@k \<in> \<langle>{u,v}\<rangle>" and "v\<^sup>@m \<in> \<langle>{u,v}\<rangle>"
    by (simp_all add: gen_in power_in)
  hence "{u\<^sup>@k,v\<^sup>@m} \<subseteq> \<langle>{u,v}\<rangle>"
    by blast
  from hull_mono'[OF this]
  show "w \<in> \<langle>{u,v}\<rangle>"
    using \<open>w \<in> \<langle>{u\<^sup>@k,v\<^sup>@m}\<rangle>\<close> by blast
qed

lemma in_hull_primroots: "w \<in> \<langle>{x, y}\<rangle> \<Longrightarrow> w \<in> \<langle>{\<rho> x, \<rho> y}\<rangle>"
  using roots_hull[of w "e\<^sub>\<rho> x" "\<rho> x"  "e\<^sub>\<rho> y" "\<rho> y"] unfolding primroot_exp_eq.

lemma roots_hull_sub: "\<langle>{u\<^sup>@k,v\<^sup>@m}\<rangle> \<subseteq> \<langle>{u,v}\<rangle>"
  by (rule subsetI; rule roots_hull)

lemma primroot_gen[simp,intro]: "v \<in> \<langle>{u, \<rho> v}\<rangle>"
  using power_in[of "\<rho> v" "{u,\<rho> v}"]
  by (cases "v = \<epsilon>", simp) (metis primroot_expE gen_in insert_iff)

lemma primroot_gen'[simp, intro]: "u \<in> \<langle>{\<rho> u, v}\<rangle>"
  using primroot_gen insert_commute by metis

lemma lists_lists_gen_primroots [intro]: "u \<in> lists {x, y} \<Longrightarrow> u \<in> lists \<langle>{\<rho> x, \<rho> y}\<rangle>"
  using lists_mono[OF genset_sub] by fast

lemma dec_primroot_bin_sing [intro]: assumes "a \<in> {x,y}" "x \<cdot> y \<noteq> y \<cdot> x"
  shows "Dec {\<rho> x, \<rho> y} a = [\<rho> a]\<^sup>@e\<^sub>\<rho> a"
  using assms by (auto intro: binary_code.primroot_dec[OF binary_code.intro[OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close>]])

lemma ref_primroot_bin_sing: assumes "a \<in> {x,y}" "x \<cdot> y \<noteq> y \<cdot> x"
  shows "Ref {\<rho> x, \<rho> y} [a] = [\<rho> a]\<^sup>@e\<^sub>\<rho> a"
  by (rule code.code_unique_ref[of "{\<rho> x, \<rho> y}" "[a]", unfolded dec_primroot_bin_sing[OF assms] concat_simps])
  (use assms in blast, use assms in force)

lemma lists_sub_mono_gen: "ws \<in> lists S \<Longrightarrow> S \<subseteq> \<langle>G\<rangle> \<Longrightarrow> ws \<in> lists \<langle>G\<rangle>"
  using sub_lists_mono.

subsection \<open>Binary Mismatch tools\<close>

definition bin_mismatch_hard :: "'a list \<Rightarrow> 'a list \<Rightarrow> 'a list  \<Rightarrow> bool" where
  "bin_mismatch_hard x y w \<equiv> \<exists> k. \<rho> x \<cdot> (\<rho> x)\<^sup>@k \<cdot> \<rho> y \<le>p w"

lemma bin_mismatch_hard_def':  assumes "x \<cdot> y \<noteq> y \<cdot> x" "bin_mismatch_hard x y w"
  shows "bin_lcp x y \<cdot> [bin_mismatch x y] \<le>p w"
proof-
  from \<open>x \<cdot> y \<noteq> y \<cdot> x\<close>
  interpret binary_code "\<rho> x" "\<rho> y"
    by unfold_locales blast
  find_theorems "bin_lcp ?x ?y \<cdot> [bin_mismatch ?x ?y]"
  from bin_lcp_fst_pow_pref[OF zero_less_Suc, of _ \<epsilon>, unfolded emp_simps] \<open>bin_mismatch_hard x y w\<close>[unfolded bin_mismatch_hard_def lassoc pow_Suc[symmetric]]
  show ?thesis
    unfolding bin_code_primroots[OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close>] using pref_trans by blast
qed

definition bin_mismatch_pref :: "'a list \<Rightarrow> 'a list \<Rightarrow> 'a list  \<Rightarrow> bool" where
  "bin_mismatch_pref x y w \<equiv> \<exists> k.  \<rho> x\<^sup>@k \<cdot> \<rho> y \<le>p w"

lemma bm_pref_letter:  assumes "x \<cdot> y \<noteq> y \<cdot> x" and "bin_mismatch_pref x y (w \<cdot> \<rho> y)"
  shows "bin_lcp x y \<cdot> [bin_mismatch x y] \<le>p x \<cdot> w \<cdot> bin_lcp x y" (is "?\<alpha> \<cdot> [?c] \<le>p x \<cdot> w \<cdot> ?\<alpha>")
proof-
  from binary_code.code_bin_roots[OF binary_code.intro, OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close>]
  interpret binary_code "\<rho> x" "\<rho> y".
  write bin_code_lcp ("\<alpha>") and bin_code_mismatch_fst ("c\<^sub>x") and bin_code_mismatch_snd ("c\<^sub>y")
  have "x \<noteq> \<epsilon>"
    using \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> by blast
  from exE[OF assms(2)[unfolded bin_mismatch_pref_def]]
  obtain k where pref_x: "\<rho> x \<^sup>@ (e\<^sub>\<rho> x + k) \<cdot> \<rho> y \<le>p x \<cdot> w \<cdot> \<rho> y"
    unfolding pow_add primroot_exp_eq rassoc pref_cancel_conv.
  have mismatch_x: "\<alpha> \<cdot> [c\<^sub>x] \<le>p \<rho> x \<^sup>@ (e\<^sub>\<rho> x + k) \<cdot> \<rho> y"
    using bin_lcp_fst_pow_pref[of "e\<^sub>\<rho> x + k" \<epsilon>, unfolded emp_simps]
    \<open>x \<noteq> \<epsilon>\<close>[folded primroot_exp_zero_conv] by blast
  have "\<alpha> \<cdot> [c\<^sub>x] \<le>p x \<cdot> w \<cdot> \<alpha>"
    by (rule ruler_le[OF pref_trans[OF mismatch_x pref_ext[OF pref_x, unfolded rassoc]]],
       unfold pref_cancel_conv, rule bin_lcp_pref')
    (unfold pref_cancel_conv lenmorph sing_len, use nemp_len[OF \<open>x \<noteq> \<epsilon>\<close>] in linarith)
  then show ?thesis
    unfolding bin_code_primroots[OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close>].
qed

\<comment> \<open>Binary mismatch elims\<close>
named_theorems bm_elims

lemma bm_eq1 [bm_elims]: assumes "x \<cdot> w1 = y \<cdot> w2" and  "bin_mismatch_pref x y (w1 \<cdot> \<rho> y)" and "bin_mismatch_pref y x (w2 \<cdot> \<rho> x)"
  shows "x \<cdot> y = y \<cdot> x"
proof(rule nemp_comm, rule ccontr)
  assume a: "x \<cdot> y \<noteq> y \<cdot> x"
  from binary_code.code_bin_roots[OF binary_code.intro, OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close>]
  interpret binary_code "\<rho> x" "\<rho> y".
  from bm_pref_letter[OF a assms(2), unfolded lassoc  \<open>x \<cdot> w1 = y \<cdot> w2\<close>]
    bm_pref_letter[OF a[symmetric] assms(3), unfolded lassoc]
  show False
    unfolding bin_lcp_sym[of y x] bin_code_primroots[OF a, symmetric]
    using  same_sing_pref bin_mismatch_neq by fast
qed

lemma bm_eq2 [bm_elims]: assumes "\<rho> x \<cdot> w1 = y \<cdot> w2" and  "bin_mismatch_pref x y (w1 \<cdot> \<rho> y)" and "bin_mismatch_pref y x (w2 \<cdot> \<rho> x)"
  shows "x \<cdot> y = y \<cdot> x"
  using bm_eq1[OF \<open>\<rho> x \<cdot> w1 = y \<cdot> w2\<close>] assms(2-3) unfolding bin_mismatch_pref_def primroot_idemp
  comm_primroot_conv[of "\<rho> x" y] using comm_primroot_conv'[of x y, symmetric] by argo

lemma bm_eq3 [bm_elims]: assumes "x \<cdot> w1 = \<rho> y \<cdot> w2" and  "bin_mismatch_pref x y (w1 \<cdot>  \<rho> y)" and "bin_mismatch_pref y x (w2 \<cdot> \<rho> x)"
  shows "x \<cdot> y = y \<cdot> x"
  using bm_eq1[OF \<open>x \<cdot> w1 = \<rho> y \<cdot> w2\<close>] assms(2-3) unfolding bin_mismatch_pref_def primroot_idemp
  comm_primroot_conv[of "\<rho> x" y] using comm_primroot_conv[of x y, symmetric] by blast

lemma bm_eq4 [bm_elims]: assumes "\<rho> x \<cdot> w1 = \<rho> y \<cdot> w2" and  "bin_mismatch_pref x y (w1 \<cdot> \<rho> y) " and "bin_mismatch_pref y x (w2 \<cdot> \<rho> x)"
  shows "x \<cdot> y = y \<cdot> x"
  using bm_eq3[OF \<open>\<rho> x \<cdot> w1 = \<rho> y \<cdot> w2\<close>, symmetric] assms(2-3) unfolding bin_mismatch_pref_def primroot_idemp comm_primroot_conv'[of x y, symmetric] using comm_primroot_conv[of y x] by presburger

lemma bm_hard_lcp [bm_elims]: assumes "x \<cdot> y \<noteq> y \<cdot> x" and "bin_mismatch_hard x y w1" and "bin_mismatch_hard y x w2"
  shows "w1 \<and>\<^sub>p w2 = x \<cdot> y \<and>\<^sub>p y \<cdot> x"
proof-
  interpret binary_code "\<rho> x" "\<rho> y"
    using \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> by unfold_locales blast
  show ?thesis
    using  bin_mismatch_hard_def'[OF assms(1-2)] bin_mismatch_hard_def'[OF assms(1)[symmetric] assms(3)]
    unfolding bin_code_primroots[OF assms(1), symmetric] bin_lcp_def[symmetric] bin_lcp_sym[of y]
    prefix_def rassoc bin_mismatch_hard_def
    using lcp_first_mismatch[OF bin_mismatch_neq] by blast
qed

lemma bin_mismatch_pref_ext: "bin_mismatch_pref x y w1 \<Longrightarrow> bin_mismatch_pref x y (w1 \<cdot> z)"
  unfolding bin_mismatch_pref_def using pref_ext by blast

lemma bm_pref1 [bm_elims]: assumes "x \<cdot> w1 \<le>p y \<cdot> w2" and  "bin_mismatch_pref x y w1"
  and "bin_mismatch_pref  y x (w2 \<cdot> \<rho> x)"
shows "x \<cdot> y = y \<cdot> x"
  using bm_eq1[OF lq_pref[OF \<open>x \<cdot> w1 \<le>p y \<cdot> w2\<close>, unfolded rassoc]]
  bin_mismatch_pref_ext[OF assms(2)]  assms(3) unfolding rassoc  by presburger

lemma bm_pref2 [bm_elims]: assumes "\<rho> x \<cdot> w1 \<le>p  y \<cdot> w2" and "bin_mismatch_pref x y w1"
  and "bin_mismatch_pref y x (w2 \<cdot> \<rho> x)"
shows "x \<cdot> y = y \<cdot> x"
  using bm_eq2[OF lq_pref[OF \<open>\<rho> x \<cdot> w1 \<le>p y \<cdot> w2\<close>, unfolded rassoc]] bin_mismatch_pref_ext[OF assms(2)]
  assms(3) unfolding rassoc  by presburger

lemma bm_pref3 [bm_elims]: assumes "x \<cdot> w1 \<le>p  \<rho> y \<cdot> w2" and "bin_mismatch_pref x y w1"
  and "bin_mismatch_pref y x (w2 \<cdot> \<rho> x)"
shows "x \<cdot> y = y \<cdot> x"
  using bm_eq3[OF lq_pref[OF \<open>x \<cdot> w1 \<le>p \<rho> y \<cdot> w2\<close>, unfolded rassoc]] bin_mismatch_pref_ext[OF assms(2)]
    assms(3) unfolding rassoc  by presburger

lemma bm_pref4 [bm_elims]: assumes "\<rho> x \<cdot> w1 \<le>p  \<rho> y \<cdot> w2" and "bin_mismatch_pref x y w1"
  and "bin_mismatch_pref y x (w2 \<cdot> \<rho> x)"
shows "x \<cdot> y = y \<cdot> x"
  using bm_eq4[OF lq_pref[OF \<open>\<rho> x \<cdot> w1 \<le>p \<rho> y \<cdot> w2\<close>, unfolded rassoc]] bin_mismatch_pref_ext[OF assms(2)]  assms(3) unfolding rassoc by presburger

lemmas [bm_elims] = bm_elims[symmetric]

\<comment> \<open>Binary mismatch predicate evaluation\<close>

named_theorems bm_simps
lemma bm_mismatch_rhoI1 [bm_simps]: assumes "bin_mismatch_pref x y w" shows  "bin_mismatch_hard x y (\<rho> x \<cdot> w)"
  using assms  unfolding bin_mismatch_pref_def bin_mismatch_hard_def
  by blast

lemma bm_mismatchI1 [bm_simps]: assumes "bin_mismatch_pref x y w" shows  "bin_mismatch_hard x y (x \<cdot> w)"
proof-
  from assms[unfolded bin_mismatch_pref_def]
  obtain k where "\<rho> x \<^sup>@ k \<cdot> \<rho> y \<le>p w"
    by blast
  have "\<rho> x \<cdot> \<rho> x \<^sup>@ (e\<^sub>\<rho> x - 1 + k) \<cdot> \<rho> y \<le>p x \<cdot> w"
    by (subst (4) pop_primroot[of x], unfold pow_add rassoc pref_cancel_conv) fact
  then show ?thesis
    unfolding bin_mismatch_hard_def by blast
qed

lemma bm_mismatch_rhoI2' [bm_simps]: assumes "bin_mismatch_pref x y w" shows  "bin_mismatch_pref x y (\<rho> x\<^sup>@k \<cdot> w)"
 unfolding bin_mismatch_pref_def
proof (rule exE[OF assms[unfolded bin_mismatch_pref_def]])
  fix m
  assume "\<rho> x \<^sup>@ m \<cdot> \<rho> y \<le>p w"
  show "\<exists>ka. \<rho> x \<^sup>@ ka \<cdot> \<rho> y \<le>p \<rho> x \<^sup>@ k \<cdot> w"
    by (rule exI[of _ "k+m"], unfold pow_add rassoc pref_cancel_conv) fact
qed

lemma bm_mismatch_rhoI2 [bm_simps]: assumes "bin_mismatch_pref x y w" shows  "bin_mismatch_pref x y (\<rho> x \<cdot> w)"
  using bm_mismatch_rhoI2'[OF assms, of 1, unfolded pow_list_1].

lemma bm_mismatchI2 [bm_simps]: assumes "bin_mismatch_pref x y w" shows  "bin_mismatch_pref x y (x \<cdot> w)"
  using bm_mismatch_rhoI2'[OF assms, of "e\<^sub>\<rho> x", unfolded primroot_exp_eq].

lemma bm_mismatchI2' [bm_simps]: assumes "bin_mismatch_pref x y w" shows  "bin_mismatch_pref x y (x\<^sup>@k \<cdot> w)"
  using bm_mismatch_rhoI2'[OF assms, of "e\<^sub>\<rho> x * k", unfolded pow_mult primroot_exp_eq].

lemma [bm_simps]: " bin_mismatch_pref x y (y \<cdot> v)"
  unfolding bin_mismatch_pref_def using append_Nil[of "\<rho> y", folded pow_zero[of "\<rho>  x"]]
   pref_ext[OF primroot_pref[of y]] by metis

lemma [bm_simps]: " bin_mismatch_pref x y y"
  unfolding bin_mismatch_pref_def using  append_Nil[of "\<rho> y", folded pow_zero[of "\<rho>  x"]] primroot_pref[of y] by metis

lemma [bm_simps]:  assumes  "w1 \<in> \<langle>{x,y}\<rangle>" "bin_mismatch_pref x y w"
  shows "bin_mismatch_pref x y (w1 \<cdot> w)"
proof-
  obtain k where "\<rho> x\<^sup>@k \<cdot> \<rho> y \<le>p w "
    using \<open>bin_mismatch_pref x y w\<close>[unfolded bin_mismatch_pref_def] by blast
  obtain m where or: "x \<^sup>@ m \<cdot> y \<le>p w1 \<or> w1 = x \<^sup>@ m"
    using two_elem_first_block[OF \<open>w1 \<in> \<langle>{x,y}\<rangle>\<close>] by blast
  show "bin_mismatch_pref x y (w1 \<cdot> w)"
    unfolding bin_mismatch_pref_def
  proof (rule disjE[OF or])
    assume "x \<^sup>@ m \<cdot> y \<le>p w1"
    from pref_shorten[OF primroot_pref this]
    have "\<rho> x\<^sup>@(e\<^sub>\<rho> x * m) \<cdot> \<rho>  y \<le>p w1"
      unfolding pow_mult primroot_exp_eq.
    from pref_ext[OF this]
    show "\<exists>k. \<rho> x \<^sup>@ k \<cdot> \<rho> y \<le>p w1 \<cdot> w"
      by blast
  next
    assume "w1 = x \<^sup>@ m"
    have "\<rho> x\<^sup>@(e\<^sub>\<rho> x * m + k) \<cdot> \<rho> y \<le>p w1 \<cdot> w"
      unfolding pow_add pow_mult primroot_exp_eq \<open>w1 = x \<^sup>@ m\<close> pref_cancel_conv
      using \<open>\<rho> x\<^sup>@k \<cdot> \<rho> y \<le>p w\<close> by fastforce
    then show "\<exists>k. \<rho> x \<^sup>@ k \<cdot> \<rho> y \<le>p w1 \<cdot> w"
      by blast
  qed
qed

lemma bm_pref_triv_rho [bm_simps]: "bin_mismatch_pref x y (\<rho> y \<cdot> w)"
  unfolding bin_mismatch_pref_def using  append_Nil[of "\<rho> y",folded pow_zero[of "\<rho> x"]] by blast

lemma bm_pref_fst_rho [bm_simps]: "bin_mismatch_pref x y (\<rho> y)"
  unfolding bin_mismatch_pref_def using  append_Nil[of "\<rho> y",folded pow_zero[of "\<rho> x"]] self_pref[of "\<rho> y"] by metis

\<comment> \<open>Binary hull membership evaluation\<close>

lemma[bm_simps]: "x \<in> \<langle>{x,y}\<rangle>"
  by blast
lemma[bm_simps]: "y \<in> \<langle>{x,y}\<rangle>"
  by blast
lemma[bm_simps]: "w \<in> \<langle>{x,y}\<rangle> \<longleftrightarrow> w \<in> \<langle>{y,x}\<rangle>"
  by (simp add: insert_commute)

lemmas[bm_simps] = hull_closed power_in
lemmas [bm_simps] = lcp_ext_left

\<comment> \<open>the method setup\<close>

method mismatch0 =
  (insert method_facts, use nothing in
    \<open>(
      ((simp  only: shifts bm_simps)?,
      (elim bm_elims);(simp_all only: shifts bm_simps))
      )\<close>
  )

lemmas bm_simps_rev =  bm_simps[reversed]
lemmas bm_elims_rev = bm_elims[reversed]

method mismatch_rev =
  (insert method_facts, use nothing in
    \<open>(
      ((simp  only: shifts_rev bm_simps_rev)?,
      (elim bm_elims_rev);(simp_all only: shifts_rev bm_simps_rev))
      )\<close>
  )

method mismatch =
  (insert method_facts, use nothing in
    \<open>(mismatch0;fail|mismatch_rev)\<close>
  )

find_theorems name: bm_elims "?x \<and>\<^sub>p ?y"

subsubsection "Mismatch method demonstrations"

\<comment> \<open>equalitites\<close>

lemma assumes "x \<cdot> y \<cdot> z = y \<cdot> y \<cdot> x \<cdot> v" shows "x \<cdot> y = y \<cdot> x"
  using assms by mismatch

lemma assumes "y \<cdot> x \<cdot> x \<cdot> y \<cdot> z = (y \<cdot> x \<cdot> y) \<cdot> y \<cdot> x \<cdot> v" shows "x \<cdot> y = y \<cdot> x" \<comment> \<open>cancel\<close>
  using assms by mismatch

lemma "y \<cdot> y \<cdot> x \<cdot> v = x \<cdot> x \<cdot> y \<cdot> z \<Longrightarrow> x \<cdot> y = y \<cdot> x" \<comment> \<open>swap\<close>
  by mismatch

lemma "\<rho> x \<cdot> \<rho> x\<^sup>@k \<cdot> y = y \<cdot> w \<cdot> \<rho> x \<Longrightarrow> w \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> x \<cdot> y = y \<cdot> x" \<comment> \<open>primitive root and hull\<close>
  by mismatch

lemma "(r \<cdot> q) \<cdot> (r \<cdot> q) \<^sup>@ t \<cdot> r \<cdot> (q \<cdot> r \<cdot> r \<cdot> q) \<^sup>@ k = ((r \<cdot> q) \<^sup>@ t \<cdot> r \<cdot> (q \<cdot> r \<cdot> r \<cdot> q) \<^sup>@ k) \<cdot> r \<cdot> q \<Longrightarrow>
    0 < k \<Longrightarrow> r \<cdot> q = q \<cdot> r" \<comment> \<open>power\<close>
  by mismatch

lemma "((u \<cdot> v) \<^sup>@ m \<cdot> u) \<cdot> v \<cdot> (u \<cdot> v) \<^sup>@ k =  (v \<cdot> (u \<cdot> v) \<^sup>@ k) \<cdot> (u \<cdot> v) \<^sup>@ m \<cdot> u \<Longrightarrow> u \<cdot> v = v \<cdot> u"
  by mismatch

lemma "w1 \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> y \<cdot> x \<cdot> w2 \<cdot> z = x \<cdot> w1 \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma "x \<cdot> y \<cdot> u \<cdot> x \<cdot> y = y \<cdot> v \<cdot> y \<cdot> x \<Longrightarrow> x \<cdot> y = y \<cdot> x" \<comment> \<open>reverse mismatch\<close>
  apply mismatch_rev.

lemma "z \<cdot> x \<cdot> y \<cdot> x \<cdot> x  = v \<cdot> x \<cdot> y \<cdot> y \<Longrightarrow> y \<cdot> x = x \<cdot> y" \<comment> \<open>reverse mismatch\<close>
  by mismatch

lemma "k \<noteq> 0 \<Longrightarrow> j \<noteq> 0 \<Longrightarrow> (x \<^sup>@ j \<cdot> y \<^sup>@ k) \<cdot> y = y\<^sup>@k \<cdot> x \<^sup>@ j \<cdot> y \<^sup>@ (k - 1) \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch


\<comment> \<open>prefixes\<close>
lemma "\<rho> x \<cdot> \<rho> x\<^sup>@k \<cdot> y \<le>p y \<cdot> w \<cdot> \<rho> x \<Longrightarrow> w \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma "y \<cdot> x \<le>p x\<^sup>@k \<cdot> x \<cdot> y \<cdot> w \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma "0 < k \<Longrightarrow> y \<cdot> x \<le>p x\<^sup>@k \<cdot> y \<cdot> w \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma "0 < k \<Longrightarrow> 0 < l \<Longrightarrow> y\<^sup>@l \<cdot> x \<le>p x\<^sup>@k \<cdot> y \<cdot> w \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma "0 < k \<Longrightarrow> 1 < l \<Longrightarrow> y\<^sup>@l \<cdot> x \<le>p y \<cdot> x\<^sup>@k \<cdot> y \<cdot> w \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma "0 < k \<Longrightarrow> m < l \<Longrightarrow> y\<^sup>@l \<cdot> x \<le>p y\<^sup>@m \<cdot> x\<^sup>@k \<cdot> y \<cdot> w \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma assumes "0 < k" "m < l" "y\<^sup>@l \<cdot> x \<le>p y\<^sup>@m \<cdot> x\<^sup>@k \<cdot> y \<cdot> w" shows "x \<cdot> y = y \<cdot> x"
  using assms  by mismatch

lemma "w1 \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> w2 \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> x \<cdot> w2 \<cdot> y \<cdot> z = y \<cdot> w1 \<cdot> x \<cdot> v \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

\<comment> \<open>suffixes\<close>

lemma assumes "x \<cdot> x \<cdot> y \<cdot> y \<cdot> y \<cdot> y \<le>s z\<cdot> y \<cdot> y \<cdot> x \<cdot> x" shows "x \<cdot> y = y \<cdot> x"
  using assms by mismatch

lemma  "u \<cdot> v \<^sup>@ k \<cdot> v \<le>s v \<cdot> w \<cdot> u \<Longrightarrow> w \<in> \<langle>{u, v}\<rangle> \<Longrightarrow> u \<cdot> v = v \<cdot> u"
  by mismatch

lemma  "u \<cdot> \<rho> v \<^sup>@ k \<cdot> \<rho> v \<le>s \<rho> v \<cdot> w \<cdot> u \<Longrightarrow> w \<in> \<langle>{u, v}\<rangle> \<Longrightarrow> u \<cdot> v = v \<cdot> u"
  by mismatch

lemma  "u \<cdot> v \<^sup>@ k \<cdot> \<rho> v \<le>s \<rho> v \<cdot> w \<cdot> u \<Longrightarrow> w \<in> \<langle>{u, v}\<rangle> \<Longrightarrow> u \<cdot> v = v \<cdot> u"
  by mismatch

lemma "2 \<le> j \<Longrightarrow> q \<cdot> p \<cdot> q \<le>s (q \<cdot> p) \<^sup>@ et' \<cdot> q \<^sup>@ j \<Longrightarrow> p \<cdot> q = q \<cdot> p"
  by mismatch

lemma "w1 \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> w2 \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> x \<cdot> y \<cdot> w2 \<cdot> x \<le>s x \<cdot> w1 \<cdot> y \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma "w \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> w' \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> bin_mismatch_pref x  y (w \<cdot> w \<cdot> y)"
  by (simp add: bm_simps)

\<comment> \<open>Known issue: In he next example, the method fails because the assumption simplifies the goal;
   on the other hand, avoiding simplification would fail to use the positivity of \<open>k\<close>\<close>
lemma "x \<cdot> y = y\<^sup>@k \<Longrightarrow> 0 < k \<Longrightarrow> x \<cdot> y = y \<cdot> x"
      oops

\<comment> \<open>least common prefix\<close>

lemma assumes "x \<cdot> y \<noteq> y \<cdot> x"
  shows "x \<cdot> x \<cdot> y \<and>\<^sub>p y \<cdot> y \<cdot> x = (x \<cdot> y \<and>\<^sub>p y \<cdot> x)"
  using assms by mismatch

lemma assumes "x \<cdot> y \<noteq> y \<cdot> x"
  shows "w \<cdot> z \<cdot> x \<cdot> x \<cdot> y \<and>\<^sub>p w \<cdot> z \<cdot> y \<cdot> y \<cdot> x = (w \<cdot> z) \<cdot> (x \<cdot> y \<and>\<^sub>p y \<cdot> x)"
  using assms by mismatch

lemma assumes "x \<cdot> y \<noteq> y \<cdot> x"
  shows "y \<cdot> y \<cdot> x \<and>\<^sub>p x \<cdot> x \<cdot> y = (x \<cdot> y \<and>\<^sub>p y \<cdot> x)"
  using assms[symmetric] unfolding lcp_sym[of "x \<cdot> y"] \<comment> \<open>lcp mismatch is order sensitive\<close>
  by mismatch

\<comment> \<open>least common suffix\<close>

lemma assumes "x \<cdot> y \<noteq> y \<cdot> x"
  shows "x \<cdot> x \<cdot> y \<and>\<^sub>s y \<cdot> y \<cdot> x = (x \<cdot> y \<and>\<^sub>s y \<cdot> x)"
  using assms  by mismatch

lemma assumes "x \<cdot> y \<noteq> y \<cdot> x"
  shows "x \<cdot> x \<cdot> y \<cdot> z \<cdot> y \<and>\<^sub>s y \<cdot> y \<cdot> x \<cdot> z \<cdot> y = (x \<cdot> y \<and>\<^sub>s y \<cdot> x) \<cdot> (z \<cdot> y)"
  using assms by mismatch

subsection \<open>Applied mismatch\<close>

 \<comment> \<open>An alternative proof of @{thm pow_list_comm_comm}. Note the artificial introduction of the "mismatch"\<close>
lemma assumes "x\<^sup>@j = y\<^sup>@k" "0 < j" shows "x \<cdot> y = y \<cdot> x"
  using arg_cong[OF assms(1), of "\<lambda> z. z \<cdot> y"] \<open>0 < j\<close>
  by mismatch

lemma pows_comm_comm:  assumes  "u\<^sup>@k \<cdot> v\<^sup>@m = u\<^sup>@l \<cdot> v\<^sup>@n" "k \<noteq> l" shows "u \<cdot> v = v \<cdot> u"
proof-
  have aux: "u\<^sup>@k \<cdot> v\<^sup>@m \<cdot> v \<cdot> u = u\<^sup>@l \<cdot> v\<^sup>@n \<cdot> v \<cdot> u \<Longrightarrow> k \<noteq> l \<Longrightarrow> u \<cdot> v = v \<cdot> u"
  proof (induct k l rule: diff_induct) qed (mismatch, mismatch, fastforce)
  \<comment> \<open>the third goal has nothing to do with mismatch,
  therefore trying to make mismatch+ succeed is misleading\<close>
  from this[unfolded lassoc cancel_right, OF assms]
  show "u \<cdot> v = v \<cdot> u".
qed

\<comment> \<open>The following is needed in binary interpretations\<close>
lemma sq_not_pref_mesosome_cover: assumes eq:  "x \<cdot> x \<cdot> concat ws = p \<cdot> x \<cdot> y\<^sup>@k \<cdot> x \<cdot> s"  and "ws \<in> lists {x,y}" "p <p x" "p \<noteq> \<epsilon>" \<open>1 < k\<close>
  shows "x \<cdot> y = y  \<cdot> x"
proof-
  define z where "z = (p \<cdot> x) \<inverse>\<^sup>> (x \<cdot> x)"
  have "p \<cdot> x <p x \<cdot> x"
    by (rule ruler_less[of _ \<open>x \<cdot> x \<cdot> concat ws\<close>], simp add: eq, simp_all add: spref_len[OF \<open>p <p x\<close>])
  from mid_sq_primroot[OF this, folded z_def]
  have "(\<rho> x)\<^sup>@e\<^sub>\<rho> z  \<cdot> concat ws = y\<^sup>@k \<cdot> x \<cdot> s"
    using eq unfolding lassoc \<open>x \<cdot> x = p \<cdot> x \<cdot> \<rho> x \<^sup>@ e\<^sub>\<rho> z\<close>  unfolding rassoc cancel by blast
  thus "x \<cdot> y = y  \<cdot> x"
    using concat_in_hull'[OF \<open>ws \<in> lists {x,y}\<close>]
    unfolding pow_pos[OF \<open>0 < e\<^sub>\<rho> z\<close>] pop_pow_2[OF \<open>1 < k\<close>, symmetric] rassoc
    by mismatch
qed

lemma (in binary_code) bin_mismatch_pows: "u\<^sub>0\<^sup>@Suc k \<cdot> u\<^sub>1 \<cdot> z \<noteq> u\<^sub>1\<^sup>@Suc l \<cdot> u\<^sub>0 \<cdot> z'"
proof
  assume "u\<^sub>0 \<^sup>@ Suc k \<cdot> u\<^sub>1 \<cdot> z = u\<^sub>1 \<^sup>@ Suc l \<cdot> u\<^sub>0 \<cdot> z'"
  then have "u\<^sub>0 \<cdot> u\<^sub>1 = u\<^sub>1 \<cdot> u\<^sub>0"
    by mismatch
  then show False
    using non_comm by contradiction
qed

section \<open>Free hull\<close>

text\<open>While not every set $G$ of generators is a code, there is a unique minimal
 free monoid containing it, called the \emph{free hull} of $G$.
It can be defined inductively using the property known as the \emph{stability condition}.
\<close>

inductive_set free_hull :: "'a list set \<Rightarrow> 'a list set" ("\<langle>_\<rangle>\<^sub>F")
  for G where
    "\<epsilon> \<in> \<langle>G\<rangle>\<^sub>F"
  | free_gen_in: "w \<in> G \<Longrightarrow> w \<in> \<langle>G\<rangle>\<^sub>F"
  | "w1 \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> w2 \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> w1 \<cdot> w2 \<in> \<langle>G\<rangle>\<^sub>F"
  | stability: "p \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> q \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> p \<cdot> w \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> w \<cdot> q \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> w \<in> \<langle>G\<rangle>\<^sub>F" \<comment> \<open>the stability condition\<close>

lemmas [intro] = free_hull.intros

text\<open>The defined set indeed is a hull.\<close>

lemma free_hull_hull[simp]: "\<langle>\<langle>G\<rangle>\<^sub>F\<rangle> = \<langle>G\<rangle>\<^sub>F"
  by (intro antisym subsetI, rule hull.induct) blast+

text\<open>The free hull is always (non-strictly) larger than the hull.\<close>

lemma hull_sub_free_hull: "\<langle>G\<rangle> \<subseteq> \<langle>G\<rangle>\<^sub>F"
proof
  fix x assume "x \<in> \<langle>G\<rangle>"
  then show "x \<in> \<langle>G\<rangle>\<^sub>F"
    using free_hull.intros(3)
      gen_monoid_induct[of x G "\<lambda> x. x \<in> \<langle>G\<rangle>\<^sub>F", OF \<open>x \<in> \<langle>G\<rangle>\<close> free_hull.intros(1)[of G] free_hull.intros(2)]
    by auto
qed

text\<open>On the other hand, it can be proved that the \emph{free basis}, defined as the basis of the free hull,  has a (non-strictly) smaller cardinality than the ordinary basis. (See the AFP theory Combinatorics-Words-Graph-Lemma.Graph-Lemma)\<close>

definition free_basis ::  "'a list set \<Rightarrow> 'a list set" ("\<BB>\<^sub>F _" [54] 55)
  where  "free_basis G \<equiv> \<BB> \<langle>G\<rangle>\<^sub>F"

lemma basis_gen_hull_free: "\<langle>\<BB>\<^sub>F G\<rangle> = \<langle>G\<rangle>\<^sub>F"
  unfolding free_basis_def  basis_gen_hull free_hull_hull..

lemma genset_sub_free: "G \<subseteq> \<langle>G\<rangle>\<^sub>F"
  by (simp add: free_hull.free_gen_in subsetI)

text
  \<open>We have developed two points of view on freeness:
\<^item> inductive point of view: to satisfy the stability condition;
\<^item> being generated by a code.
\<close>

text\<open>We now show their equivalence\<close>

text\<open>First, basis of a free hull is a code.\<close>

lemma free_basis_code[simp]: "code (\<BB>\<^sub>F G)"
proof (rule code.intro)
  fix xs ys
  show "xs \<in> lists (\<BB>\<^sub>F G) \<Longrightarrow> ys \<in> lists (\<BB>\<^sub>F G) \<Longrightarrow> concat xs = concat ys \<Longrightarrow> xs = ys"
  proof(induction xs ys rule: list_induct2')
    case (2 x xs)
    have "x = \<epsilon>"
      using \<open>concat (x # xs) = concat \<epsilon>\<close> by force
    have "x \<in> \<BB> \<langle>G\<rangle>\<^sub>F"
      using  \<open>x # xs \<in> lists (\<BB>\<^sub>F G)\<close> unfolding free_basis_def by force
    from emp_not_basis[OF this]
    have "x \<noteq> \<epsilon>".
    then show ?case
      using \<open>x = \<epsilon>\<close> by contradiction
  next
    case (3 y ys)
    have "y = \<epsilon>"
      using \<open>concat \<epsilon> = concat (y # ys)\<close> by force
    have "y \<in> \<BB> \<langle>G\<rangle>\<^sub>F"
      using  \<open>y # ys \<in> lists (\<BB>\<^sub>F G)\<close> unfolding free_basis_def by force
    from emp_not_basis[OF this]
    have "y \<noteq> \<epsilon>".
    then show ?case
      using \<open>y = \<epsilon>\<close> by contradiction
  next
    case (4 x xs y ys)
    show ?case
    proof (unfold list.inject, rule conjI)
      have in_free_hull:  "x \<in> \<langle>G\<rangle>\<^sub>F" "y \<in> \<langle>G\<rangle>\<^sub>F" "concat xs \<in> \<langle>G\<rangle>\<^sub>F" "concat ys \<in> \<langle>G\<rangle>\<^sub>F" and
           simple: "x \<in>U \<langle>G\<rangle>\<^sub>F" "y \<in>U \<langle>G\<rangle>\<^sub>F" and
           in_hull: "x \<in> \<langle>\<langle>G\<rangle>\<^sub>F\<rangle>" "y \<in> \<langle>\<langle>G\<rangle>\<^sub>F\<rangle>" and
           nemp: "x \<noteq> \<epsilon>" "y \<noteq> \<epsilon>"
        using \<open>x # xs \<in> lists (\<BB>\<^sub>F G)\<close> \<open>y # ys \<in> lists (\<BB>\<^sub>F G)\<close> free_basis_def
        emp_not_basis unfolding basis_def basis_gen_hull_free[symmetric] by auto
      have "x \<cdot> concat xs = y \<cdot> concat ys"
        using \<open>concat (x # xs) = concat (y # ys)\<close> by simp
      from eqd_or[OF this]
       obtain t where or: "x \<cdot> t = y \<and> t \<cdot> concat ys = concat xs \<or> y \<cdot> t = x \<and> t \<cdot> concat xs = concat ys"
        by blast
      have "t \<in> \<langle>G\<rangle>\<^sub>F"
        by (rule disjE[OF or])
        (use stability[of x G "concat ys" t] stability[of y G "concat xs" t] in_free_hull
        in fastforce)+
      hence "t = \<epsilon>"
        using or simple free_hull_hull in_hull nemp ungen_dec_triv by metis
      thus "x = y"
        using or by blast
      then show "xs = ys"
      using "4.IH" \<open>x # xs \<in> lists (\<BB>\<^sub>F G)\<close> \<open>y # ys \<in> lists (\<BB>\<^sub>F G)\<close> \<open>concat (x # xs) = concat (y # ys)\<close>
      by auto
  qed
qed simp
qed

lemma gen_in_free_hull: "x \<in> G \<Longrightarrow> x \<in> \<langle>\<BB>\<^sub>F G\<rangle>"
  using free_hull.free_gen_in[folded basis_gen_hull_free].

text\<open>Second, a code generates its free hull.\<close>

lemma (in code) code_gen_free_hull: "\<langle>\<C>\<rangle>\<^sub>F = \<langle>\<C>\<rangle>"
proof
  show "\<langle>\<C>\<rangle> \<subseteq> \<langle>\<C>\<rangle>\<^sub>F"
    using hull_mono[of \<C> "\<langle>\<C>\<rangle>\<^sub>F"] free_gen_in[of _ \<C>]  subsetI[of \<C> "\<langle>\<C>\<rangle>\<^sub>F"]
    unfolding free_hull_hull by fast
  show "\<langle>\<C>\<rangle>\<^sub>F \<subseteq> \<langle>\<C>\<rangle>"
  proof
    fix x assume "x \<in> \<langle>\<C>\<rangle>\<^sub>F"
    thm emp_in
    show "x \<in> \<langle>\<C>\<rangle>"
    proof(rule free_hull.induct[OF \<open>x \<in> \<langle>\<C>\<rangle>\<^sub>F\<close>])
      show "w \<in> \<langle>\<C>\<rangle>" if elems: "p \<in> \<langle>\<C>\<rangle>" "q \<in> \<langle>\<C>\<rangle>" "p \<cdot> w \<in> \<langle>\<C>\<rangle>" "w \<cdot> q \<in> \<langle>\<C>\<rangle>" for p q w
      proof-
        have eq: "(Dec \<C> p) \<cdot> (Dec \<C> w \<cdot> q) = (Dec \<C> p \<cdot> w) \<cdot> (Dec \<C> q)"
          using code_dec_morph[OF \<open>p \<in> \<langle>\<C>\<rangle>\<close> \<open>w \<cdot> q \<in> \<langle>\<C>\<rangle>\<close>, unfolded lassoc]
          unfolding code_dec_morph[OF \<open>p \<cdot> w \<in> \<langle>\<C>\<rangle>\<close> \<open>q \<in> \<langle>\<C>\<rangle>\<close>, symmetric].
        from eqd_or[OF this]
        obtain ts where or': "(Dec \<C> p) \<cdot> ts = Dec \<C> p \<cdot> w \<or> (Dec \<C> p \<cdot> w) \<cdot> ts = Dec \<C> p "
          by blast
        hence "concat ((Dec \<C> p) \<cdot> ts)  = concat (Dec \<C> p \<cdot> w) \<or>
              concat((Dec \<C> p \<cdot> w) \<cdot> ts) = concat (Dec \<C> p)"
          by presburger
        hence "concat ts = w"
          unfolding concat_dec[OF \<open>p \<cdot> w \<in> \<langle>\<C>\<rangle>\<close>] concat_dec[OF \<open>p \<in> \<langle>\<C>\<rangle>\<close>] concat_morph
          rassoc by blast
        have "ts \<in> lists \<C>"
          using dec_in_lists[OF \<open>p \<cdot> w \<in> \<langle>\<C>\<rangle>\<close>] dec_in_lists[OF \<open>p \<in> \<langle>\<C>\<rangle>\<close>]
          or' append_in_lists_conv[of _ ts \<C>] by metis
        thus "w \<in> \<langle>\<C>\<rangle>"
          using \<open>concat ts = w\<close> by blast
      qed
    qed (simp_all add: \<open>x \<in> \<langle>\<C>\<rangle>\<^sub>F\<close> hull_closed gen_in)
  qed
qed


text\<open>That is, a code is its own free basis\<close>

lemma (in code) code_free_basis: "\<C> = \<BB>\<^sub>F \<C>"
  using basis_of_hull[of \<C>, unfolded code_gen_free_hull[symmetric]
      code_is_basis, symmetric] unfolding free_basis_def.

text\<open>This allows to use the introduction rules of the free hull to prove one of the basic characterizations
 of the code, called the stability condition\<close>

lemma (in code) stability: "p \<in> \<langle>\<C>\<rangle> \<Longrightarrow> q \<in> \<langle>\<C>\<rangle> \<Longrightarrow> p \<cdot> w \<in> \<langle>\<C>\<rangle> \<Longrightarrow> w \<cdot> q \<in> \<langle>\<C>\<rangle> \<Longrightarrow> w \<in> \<langle>\<C>\<rangle>"
  unfolding code_gen_free_hull[symmetric] using free_hull.intros(4) by auto

text\<open>Moreover, the free hull of G is the smallest code-generated hull containing G.
In other words, the term free hull is appropriate.\<close>



text\<open>First, several intuitive monotonicity and closure results.\<close>

lemma free_hull_mono: assumes "G \<subseteq> H" shows "\<langle>G\<rangle>\<^sub>F \<subseteq> \<langle>H\<rangle>\<^sub>F"
proof
  fix x assume "x \<in> \<langle>G\<rangle>\<^sub>F"
  have el: "\<And> w. w \<in> G \<Longrightarrow> w \<in> \<langle>H\<rangle>\<^sub>F"
    using \<open>G \<subseteq> H\<close> free_hull.free_gen_in by auto
  show "x \<in> \<langle>H\<rangle>\<^sub>F"
    by (rule free_hull.induct[of x G]) (auto simp add: \<open>x \<in> \<langle>G\<rangle>\<^sub>F\<close> el)
qed

lemma free_hull_idem: "\<langle>\<langle>G\<rangle>\<^sub>F\<rangle>\<^sub>F = \<langle>G\<rangle>\<^sub>F"
proof
  show "\<langle>\<langle>G\<rangle>\<^sub>F\<rangle>\<^sub>F \<subseteq> \<langle>G\<rangle>\<^sub>F"
  proof
    fix x assume "x \<in> \<langle>\<langle>G\<rangle>\<^sub>F\<rangle>\<^sub>F"
    show "x \<in> \<langle>G\<rangle>\<^sub>F"
    proof (rule free_hull.induct[of x "\<langle>G\<rangle>\<^sub>F"])
      show "\<And>p q w. p \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow>  q \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow>  p \<cdot> w \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow>  w \<cdot> q \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> w \<in> \<langle>G\<rangle>\<^sub>F"
        using free_hull.intros(4) by auto
    qed (simp_all add: \<open>x \<in> \<langle>\<langle>G\<rangle>\<^sub>F\<rangle>\<^sub>F\<close> free_hull.intros(1), simp add: free_hull.intros(2), simp add: free_hull.intros(3))
  qed
next
  show "\<langle>G\<rangle>\<^sub>F \<subseteq> \<langle>\<langle>G\<rangle>\<^sub>F\<rangle>\<^sub>F"
    using free_hull_hull hull_sub_free_hull by auto
qed

lemma hull_gen_free_hull: "\<langle>\<langle>G\<rangle>\<rangle>\<^sub>F = \<langle>G\<rangle>\<^sub>F"
proof
  show " \<langle>\<langle>G\<rangle>\<rangle>\<^sub>F \<subseteq> \<langle>G\<rangle>\<^sub>F"
    using free_hull_idem free_hull_mono hull_sub_free_hull by metis
next
  show "\<langle>G\<rangle>\<^sub>F \<subseteq> \<langle>\<langle>G\<rangle>\<rangle>\<^sub>F"
    by (simp add: free_hull_mono)
qed

text \<open>Code is also the free basis of its hull.\<close>

lemma (in code) code_free_basis_hull: "\<C> = \<BB>\<^sub>F \<langle>\<C>\<rangle>"
  unfolding free_basis_def using code_free_basis[unfolded free_basis_def]
  unfolding  hull_gen_free_hull.

text\<open>The minimality of the free hull easily follows.\<close>

theorem (in code) free_hull_min: assumes "G \<subseteq> \<langle>\<C>\<rangle>" shows "\<langle>G\<rangle>\<^sub>F \<subseteq> \<langle>\<C>\<rangle>"
  using free_hull_mono[OF \<open>G \<subseteq> \<langle>\<C>\<rangle>\<close>] unfolding hull_gen_free_hull
  unfolding code_gen_free_hull.

theorem free_hull_inter: "\<langle>G\<rangle>\<^sub>F = \<Inter> {M. G \<subseteq> M \<and> M = \<langle>M\<rangle>\<^sub>F}"
proof
  have "X \<in> {M. G \<subseteq> M \<and> M = \<langle>M\<rangle>\<^sub>F} \<Longrightarrow> \<langle>G\<rangle>\<^sub>F \<subseteq> X" for X
    unfolding mem_Collect_eq[of _ "\<lambda> M. G \<subseteq> M \<and> M = \<langle>M\<rangle>\<^sub>F"]
    using free_hull_mono[of G X] by simp
  from Inter_greatest[of "{M. G \<subseteq> M \<and> M = \<langle>M\<rangle>\<^sub>F}", OF this]
  show "\<langle>G\<rangle>\<^sub>F \<subseteq> \<Inter> {M. G \<subseteq> M \<and> M = \<langle>M\<rangle>\<^sub>F}"
    by blast
next
  show " \<Inter> {M. G \<subseteq> M \<and> M = \<langle>M\<rangle>\<^sub>F} \<subseteq> \<langle>G\<rangle>\<^sub>F"
    by (simp add: Inter_lower free_hull_idem genset_sub_free)
qed

text\<open>Decomposition into the free basis is a morphism.\<close>

lemma free_basis_dec_morph: "u \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> v \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow>
    Dec (\<BB>\<^sub>F G) (u \<cdot> v) = (Dec (\<BB>\<^sub>F G) u) \<cdot> (Dec (\<BB>\<^sub>F G) v)"
  using code.code_dec_morph[OF free_basis_code, of u G v, symmetric,
      unfolded  basis_gen_hull_free[of G]].

section \<open>Reversing hulls and decompositions\<close>

lemma basis_rev_commute[reversal_rule]: "\<BB> (rev ` G) = rev ` (\<BB> G)"
proof
  have "\<langle>rev ` \<BB> G\<rangle> = \<langle>rev ` G\<rangle>" and *: "\<langle>rev ` \<BB> (rev ` G)\<rangle> = \<langle>rev ` rev `G\<rangle>"
    unfolding rev_hull[symmetric] basis_gen_hull by blast+
  from gen_basis_sub[OF this(1)]
  show "\<BB> (rev ` G) \<subseteq> rev ` \<BB> G".
  from image_mono[OF gen_basis_sub[OF *], of rev]
  show "rev ` (\<BB> G) \<subseteq> \<BB> (rev ` G)"
    unfolding rev_rev_image_eq.
qed

lemma rev_free_hull_comm: "\<langle>rev ` X\<rangle>\<^sub>F = rev ` \<langle>X\<rangle>\<^sub>F"
proof-
  have "rev ` \<langle>X\<rangle>\<^sub>F \<subseteq> \<langle>rev ` X\<rangle>\<^sub>F" for X :: "'a list set"
  proof
    fix x assume "x \<in> rev ` \<langle>X\<rangle>\<^sub>F"
    hence  "rev x \<in> \<langle>X\<rangle>\<^sub>F"
      by (simp add: rev_in_conv)
    have "rev x \<in> rev ` \<langle>rev ` X\<rangle>\<^sub>F"
      by (induct rule: free_hull.induct[OF \<open>rev x \<in> \<langle>X\<rangle>\<^sub>F\<close>])
        (auto simp add: rev_in_conv[symmetric])
    then show "x \<in> \<langle>rev ` X\<rangle>\<^sub>F"
      by blast
  qed
  from this
    image_mono[OF this[of "rev ` X", unfolded rev_rev_image_eq], of rev, unfolded rev_rev_image_eq]
  show  "\<langle>rev ` X\<rangle>\<^sub>F = rev ` \<langle>X\<rangle>\<^sub>F"
    by blast
qed

lemma free_basis_rev_commute [reversal_rule]: "\<BB>\<^sub>F rev ` X =  rev ` (\<BB>\<^sub>F X)"
  unfolding free_basis_def basis_rev_commute free_basis_def rev_free_hull_comm..

lemma rev_dec[reversal_rule]: assumes "x \<in> \<langle>X\<rangle>\<^sub>F" shows "Dec rev ` (\<BB>\<^sub>F X) (rev x) = map rev (rev (Dec (\<BB>\<^sub>F X) x))"
proof-
  have "x \<in> \<langle>\<BB>\<^sub>F X\<rangle>"
    using \<open>x \<in> \<langle>X\<rangle>\<^sub>F\<close> by (simp add: basis_gen_hull_free)
  from concat_dec[OF this]
  have "concat (map rev (rev (Dec \<BB>\<^sub>F X x))) = rev x"
    unfolding rev_concat[symmetric] by blast
  from  rev_image_eqI[OF rev_in_lists[OF dec_in_lists[OF \<open>x \<in> \<langle>\<BB>\<^sub>F X\<rangle>\<close>]], of _ "map rev"]
  have "map rev (rev (Dec \<BB>\<^sub>F X x)) \<in> lists (rev ` (\<BB>\<^sub>F X))"
    unfolding lists_image by blast
  from code.code_unique_dec'[OF code.code_rev_code[OF free_basis_code] this]
  show ?thesis
    unfolding \<open>concat (map rev (rev (Dec \<BB>\<^sub>F X x))) = rev x\<close>.
qed

lemma rev_hd_dec_last_eq[reversal_rule]: assumes "x \<in> X" and  "x \<noteq> \<epsilon>" shows
  "rev (hd (Dec (rev ` (\<BB>\<^sub>F X)) (rev x))) = last (Dec \<BB>\<^sub>F X x)"
proof-
  have "rev (Dec \<BB>\<^sub>F X x) \<noteq> \<epsilon>"
    using \<open>x \<in> X\<close> basis_gen_hull_free dec_nemp'[OF \<open>x \<noteq> \<epsilon>\<close>] by blast
  show ?thesis
    unfolding hd_rev rev_dec[OF free_gen_in[OF \<open>x \<in> X\<close>]] hd_map[OF \<open>rev (Dec \<BB>\<^sub>F X x) \<noteq> \<epsilon>\<close>]
    by simp
qed

lemma rev_hd_dec_last_eq'[reversal_rule]: assumes "x \<in> X" and  "x \<noteq> \<epsilon>" shows
  "(hd (Dec (rev ` (\<BB>\<^sub>F X)) (rev x))) = rev (last (Dec \<BB>\<^sub>F X x))"
  using assms(1) assms(2) rev_hd_dec_last_eq rev_swap by blast

section \<open>Lists as the free hull of singletons\<close>

text\<open>A crucial property of free monoids of words is that they can be seen as lists over the free basis,
instead as lists over the original alphabet.\<close>

abbreviation sings where "sings B \<equiv> {[b] | b. b \<in> B}"

term "Set.filter P A"

lemma sings_image: "sings B =  (\<lambda> x. [x]) ` B"
  using Setcompr_eq_image.

lemma lists_sing_map_concat_ident: "xs \<in> lists (sings B) \<Longrightarrow> xs = map (\<lambda> x. [x]) (concat xs)"
  by (induct xs, simp, auto)

lemma code_sings: "code (sings B)"
proof
  fix xs ys assume xs: "xs \<in> lists (sings B)" and ys: "ys \<in> lists (sings B)"
    and eq: "concat xs = concat ys"
  from lists_sing_map_concat_ident[OF xs, unfolded eq]
  show "xs = ys" unfolding  lists_sing_map_concat_ident[OF ys, symmetric].
qed

lemma sings_gen_lists: "\<langle>sings B\<rangle> = lists B"
  unfolding hull_concat_lists
proof(intro equalityI subsetI, standard)
  fix xs
  show "xs \<in> concat ` lists (sings B) \<Longrightarrow> \<forall>x\<in>set xs. x \<in> B"
    by force
  assume "xs \<in> lists B"
  hence "map (\<lambda>x. x # \<epsilon>) xs \<in> lists (sings B)"
    by force
  from imageI[OF this, of concat]
  show "xs \<in> concat ` lists (sings B)"
    unfolding concat_map_sing_ident[of xs].
qed

lemma sing_gen_lists: "lists {x} = \<langle>{[x]}\<rangle>"
  using sings_gen_lists[of "{x}"] by simp

lemma bin_gen_lists: "lists {x, y} = \<langle>{[x],[y]}\<rangle>"
  using sings_gen_lists[of "{x,y}"] unfolding Setcompr_eq_image by simp

lemma "sings B = \<BB>\<^sub>F (lists B)"
  using code.code_free_basis_hull[OF code_sings, of B, unfolded sings_gen_lists].

lemma map_sings: "xs \<in> lists B \<Longrightarrow> map (\<lambda>x. x # \<epsilon>) xs \<in> lists (sings B)"
  by (induct xs) auto

lemma dec_sings: "xs \<in> lists B \<Longrightarrow> Dec (sings B) xs = map (\<lambda> x. [x]) xs"
  using code.code_unique_dec'[OF code_sings, of "map (\<lambda> x. [x]) xs" B, OF map_sings]
  unfolding concat_map_sing_ident.

lemma sing_lists_exp: assumes "ws \<in> lists {a}"
  obtains k where "ws = [a]\<^sup>@k"
  using  sing_set_wordE[OF assms[folded in_lists_conv_set_subset]].

lemma sing_lists_exp_len: "ws \<in> lists {a} \<longleftrightarrow> [a]\<^sup>@\<^bold>|ws\<^bold>| = ws"
  by  (induct ws, auto)

lemma sing_lists_exp_count: "ws \<in> lists {a} \<longleftrightarrow> [a]\<^sup>@(count_list ws a) = ws"
  by  (standard, induct ws, force, simp)
  (use sing_pow_lists[OF singletonI, of "count_list ws a" a] in argo)

lemma sing_set_pow_count_list: "set ws \<subseteq> {a} \<longleftrightarrow> [a]\<^sup>@(count_list ws a) = ws"
  unfolding in_lists_conv_set_subset using  sing_lists_exp_count.

lemma count_le_length_iff: "set ws \<subseteq> {a} \<longleftrightarrow> count_list ws a = \<^bold>|ws\<^bold>|"
proof
  show "set ws \<subseteq> {a} \<Longrightarrow> count_list ws a = \<^bold>|ws\<^bold>|"
  unfolding sing_set_pow_count_list using
  lenarg[of "[a] \<^sup>@ count_list ws a" ws, unfolded pow_len sing_len] by force
next
  assume "count_list ws a = \<^bold>|ws\<^bold>|"
  then show "set ws \<subseteq> {a}"
  proof (induct ws, force)
    case (Cons b ws)
    have "b = a"
      using Cons.prems count_le_length[of ws a] impossible_Cons[of "b#ws"]
      unfolding count_list.simps(2) by metis
    have "count_list ws a = \<^bold>|ws\<^bold>|"
      using Cons.prems unfolding \<open>b = a\<close> by force
    from Cons.hyps[OF this]
    show "set (b # ws) \<subseteq> {a}"
      unfolding \<open>b = a\<close> by simp
  qed
qed

lemma sing_set_pow: "set ws \<subseteq> {a} \<longleftrightarrow> [a]\<^sup>@\<^bold>|ws\<^bold>| = ws"
  using sing_lists_exp_len[unfolded sing_lists_exp_count]
  using sing_set_pow_count_list[unfolded count_le_length_iff]
  unfolding count_le_length_iff  by fast

lemma count_sing_exp[simp]: "count_list ([a]\<^sup>@k) a = k"
  by (induct k, simp_all)

lemma count_sing_exp'[simp]: "count_list ([a]) a = 1"
  by simp

lemma count_sing_distinct[simp]: "a \<noteq> b \<Longrightarrow> count_list ([a]\<^sup>@k) b = 0"
  by (induct k, simp, auto)

lemma count_sing_distinct'[simp]: "a \<noteq> b \<Longrightarrow> count_list ([a]) b = 0"
  by simp

lemma sing_letter_imp_prim: assumes "count_list w a = 1" shows "primitive w"
proof
  fix r k
  assume "r \<^sup>@ k = w"
  have "count_list w a = k * count_list r a"
    by (simp only: count_list_pow_list flip: \<open>r \<^sup>@ k = w\<close>)
  then show "k = 1"
    unfolding \<open>count_list w a = 1\<close> by simp
qed

lemma prim_abk: "a \<noteq> b \<Longrightarrow> primitive ([a] \<cdot> [b] \<^sup>@ k)"
  by (intro sing_letter_imp_prim[of _ a]) simp

lemma sing_code: "x \<noteq> \<epsilon> \<Longrightarrow> code {x}"
proof (rule code.intro)
  fix xs ys
  assume "x \<noteq> \<epsilon>" "xs \<in> lists {x}" "ys \<in> lists {x}" "concat xs = concat ys"
  from \<open>xs \<in> lists {x}\<close>[unfolded sing_lists_exp_len, symmetric]
       \<open>ys \<in> lists {x}\<close>[unfolded sing_lists_exp_len, symmetric]
  have "\<^bold>|xs\<^bold>| = \<^bold>|ys\<^bold>|"
    using \<open>concat xs = concat ys\<close> concat_pow_list_single eq_pow_exp[OF \<open>x \<noteq> \<epsilon>\<close>] by metis
  then show "xs = ys"
    using \<open>xs = [x] \<^sup>@ \<^bold>|xs\<^bold>|\<close> \<open>ys = [x] \<^sup>@ \<^bold>|ys\<^bold>|\<close> by argo
qed

lemma sings_card: "card A = card (sings A)"
  by(rule bij_betw_same_card, rule bij_betwI'[of _ "\<lambda>x. [x]"], auto)

lemma sings_finite: "finite A = finite (sings A)"
  by(rule bij_betw_finite, rule bij_betwI'[of _ "\<lambda>x. [x]"], auto)

lemma sings_conv: "A = B \<longleftrightarrow> sings A = sings B"
proof(standard, simp)
  have "\<And>x A B. sings A = sings B \<Longrightarrow> x \<in> A \<Longrightarrow> x \<in> B"
  proof-
    fix x :: "'b" and A B
    assume "sings A = sings B" "x \<in> A"
    hence "[x] \<in> sings B"
      using \<open>sings A = sings B\<close> by blast
    thus "x \<in> B"
      by blast
  qed
  from this[of A B] this[of B A, OF sym]
  show "sings A = sings B \<Longrightarrow> A = B"
    by blast
qed

section \<open>Various additional lemmas\<close>

subsection \<open>Roots of binary set\<close>

lemma two_roots_code: assumes "x \<noteq> \<epsilon>" and  "y \<noteq> \<epsilon>" shows "code {\<rho> x, \<rho> y}"
  using assms
proof (cases "\<rho> x = \<rho> y")
  assume "\<rho> x = \<rho> y"
  thus "code {\<rho> x, \<rho> y}" using sing_code[OF primroot_nemp[OF \<open>x \<noteq> \<epsilon>\<close>]] by simp
next
  assume "\<rho> x \<noteq> \<rho> y"
  hence "\<rho> x \<cdot> \<rho> y \<noteq> \<rho> y \<cdot> \<rho> x"
    using comm_prim[OF primroot_prim[OF \<open>x \<noteq> \<epsilon>\<close>] primroot_prim[OF \<open>y \<noteq> \<epsilon>\<close>]] by blast
  thus "code {\<rho> x, \<rho> y}"
    by (simp add: bin_code_code)
qed

lemma primroot_in_set_dec: assumes "x \<noteq> \<epsilon>" and  "y \<noteq> \<epsilon>" shows "\<rho> x \<in> set (Dec {\<rho> x, \<rho> y} x)"
proof-
  obtain k where "concat ([\<rho> x]\<^sup>@k) = x" "0 < k"
    using primroot_expE
      concat_pow_list_single[symmetric, of _ "\<rho> x"] by metis
  from code.code_unique_dec'[OF two_roots_code[OF assms], of "[\<rho> x]\<^sup>@k", unfolded \<open>concat ([\<rho> x]\<^sup>@k) = x\<close>]
  have "Dec {\<rho> x, \<rho> y} x = [\<rho> x]\<^sup>@k"
    using insertI1 sing_pow_lists by metis
  show ?thesis
    unfolding \<open>Dec {\<rho> x, \<rho> y} x = [\<rho> x]\<^sup>@k\<close> using \<open>0 < k\<close> by simp
qed

lemma primroot_dec: assumes "x \<cdot> y \<noteq> y \<cdot> x"
  shows "(Dec {\<rho> x, \<rho> y} x) = [\<rho> x]\<^sup>@e\<^sub>\<rho> x" "(Dec {\<rho> x, \<rho> y} y) = [\<rho> y]\<^sup>@e\<^sub>\<rho> y"
  by (simp_all add: binary_code.intro[OF assms] binary_code.primroot_dec)


subsection Other

lemma bin_count_one_decompose: assumes "ws \<in> lists {x,y}" and "x \<noteq> y" and  "count_list ws y = 1"
  obtains k m where "[x]\<^sup>@k \<cdot> [y] \<cdot> [x]\<^sup>@m = ws"
proof-
  have "\<not> set ws \<subseteq> {x}"
    using count_sing_distinct[OF \<open>x \<noteq> y\<close>] \<open>count_list ws y = 1\<close> by auto
  from distinct_letter_in[OF this]
  obtain ws' k  b where "[x]\<^sup>@k \<cdot> [b] \<cdot> ws' = ws" and "b \<noteq> x" by blast
  hence "b = y"
    using \<open>ws \<in> lists {x,y}\<close>  by force
  have "ws' \<in> lists {x,y}"
    using \<open>ws \<in> lists {x,y}\<close>[folded \<open>[x]\<^sup>@k \<cdot> [b] \<cdot> ws' = ws\<close>] by simp
  have "count_list ws' y = 0"
    using arg_cong[OF \<open>[x]\<^sup>@k \<cdot> [b] \<cdot> ws' = ws\<close>, of "\<lambda> x. count_list x y"]
    unfolding count_list_append \<open>count_list ws y = 1\<close> \<open>b = y\<close>  by force
     hence "set ws' \<subseteq> {x}"
     unfolding count_list_0_iff using  \<open>ws' \<in> lists {x,y}\<close> by blast
  then obtain m where "ws' = [x]\<^sup>@m"
     by blast
  from that[OF \<open>[x]\<^sup>@k \<cdot> [b] \<cdot> ws' = ws\<close>[unfolded this \<open>b = y\<close>]]
  show thesis.
qed

lemma bin_count_one_conjug: assumes "ws \<in> lists {x,y}" and "x \<noteq> y" and "count_list ws y = 1"
  shows "ws \<sim> [x]\<^sup>@(count_list ws x) \<cdot> [y]"
proof-
  obtain e1 e2 where "[x]\<^sup>@e1 \<cdot> [y] \<cdot> [x]\<^sup>@e2 = ws"
    using bin_count_one_decompose[OF assms].
  from conjugI'[of "[x] \<^sup>@ e1 \<cdot> [y]" "[x]\<^sup>@e2", unfolded rassoc this]
  have "ws \<sim> [x]\<^sup>@(e2 + e1) \<cdot> [y]"
    unfolding pow_add rassoc.
  moreover have "count_list ([x]\<^sup>@(e2 + e1) \<cdot> [y]) x = e2 + e1"
    using \<open>x \<noteq> y\<close> by simp
  ultimately show ?thesis
    by (simp add: count_list_conjug)
qed

lemma bin_prim_long_set: assumes "ws \<in> lists {x,y}" and "primitive ws" and  "2 \<le> \<^bold>|ws\<^bold>|"
  shows "set ws = {x,y}"
proof-
  have "\<not> set ws \<subseteq> {c}" for c
    using \<open>primitive ws\<close> pow_nemp_imprim \<open>2 \<le> \<^bold>|ws\<^bold>|\<close>
      sing_lists_exp_len[folded in_lists_conv_set_subset] by metis
  then show "set ws = {x,y}"
    unfolding subset_singleton_iff using \<open>ws \<in> lists {x,y}\<close>[folded in_lists_conv_set_subset] doubleton_subset_cases by metis
qed

lemma bin_prim_long_pref: assumes "ws \<in> lists {x,y}" and "primitive ws" and  "2 \<le> \<^bold>|ws\<^bold>|"
  obtains ws' where "ws \<sim> ws'" and "[x,y] \<le>p ws'"
proof-
  from pow_nemp_imprim[OF \<open>2 \<le> \<^bold>|ws\<^bold>|\<close>, of "[x]"] sing_lists_exp_len[of ws x]
  have "\<not> ws \<in> lists {x}"
    using \<open>primitive ws\<close> \<open>2 \<le> \<^bold>|ws\<^bold>|\<close> by fastforce
  hence "x \<noteq> y"
    using \<open>ws \<in> lists {x,y}\<close> by fastforce
  from switch_fac[OF \<open>x \<noteq> y\<close> bin_prim_long_set[OF assms]]
  show thesis
    using \<open>2 \<le> \<^bold>|ws\<^bold>|\<close> rotate_into_pos_sq[of \<epsilon> "[x,y]" ws thesis, unfolded emp_simps, OF \<open>[x, y] \<le>f ws \<cdot> ws\<close> _ _ that, of id]
    by force
qed

end
