(*  Author:     Ata Keskin, TU München 
*)

theory Stochastic_Process
imports Filtered_Measure Measure_Space_Supplement "HOL-Probability.Independent_Family"
begin      

section \<open>Stochastic Processes\<close>

subsection \<open>Stochastic Process\<close>

text \<open>A stochastic process is a collection of random variables, indexed by a type \<^typ>\<open>'b\<close>.\<close>

locale stochastic_process =
  fixes M t\<^sub>0 and X :: "'b :: {second_countable_topology, order_topology, t2_space} \<Rightarrow> 'a \<Rightarrow> 'c :: {second_countable_topology, banach}"
  assumes random_variable[measurable]: "\<And>i. t\<^sub>0 \<le> i \<Longrightarrow> X i \<in> borel_measurable M"
begin

definition left_continuous where "left_continuous = (AE \<xi> in M. \<forall>t. continuous (at_left t) (\<lambda>i. X i \<xi>))"
definition right_continuous where "right_continuous = (AE \<xi> in M. \<forall>t. continuous (at_right t) (\<lambda>i. X i \<xi>))"

end

lemma stochastic_process_const_fun:
  assumes "f \<in> borel_measurable M"
  shows "stochastic_process M t\<^sub>0 (\<lambda>_. f)" using assms by (unfold_locales)

lemma stochastic_process_const:
  shows "stochastic_process M t\<^sub>0 (\<lambda>i _. c i)" by (unfold_locales) simp

text \<open>In the following segment, we cover basic operations on stochastic processes.\<close>

context stochastic_process
begin

lemma compose_stochastic:
  assumes "\<And>i. t\<^sub>0 \<le> i \<Longrightarrow> f i \<in> borel_measurable borel"
  shows "stochastic_process M t\<^sub>0 (\<lambda>i \<xi>. (f i) (X i \<xi>))"
  by (unfold_locales) (intro measurable_compose[OF random_variable assms]) 

lemma norm_stochastic: "stochastic_process M t\<^sub>0 (\<lambda>i \<xi>. norm (X i \<xi>))" by (fastforce intro: compose_stochastic)

lemma scaleR_right_stochastic:
  assumes "stochastic_process M t\<^sub>0 Y"
  shows "stochastic_process M t\<^sub>0 (\<lambda>i \<xi>. (Y i \<xi>) *\<^sub>R (X i \<xi>))"
  using stochastic_process.random_variable[OF assms] random_variable by (unfold_locales) simp

lemma scaleR_right_const_fun_stochastic: 
  assumes "f \<in> borel_measurable M" 
  shows "stochastic_process M t\<^sub>0 (\<lambda>i \<xi>. f \<xi> *\<^sub>R (X i \<xi>))" 
  by (unfold_locales) (intro borel_measurable_scaleR assms random_variable)

lemma scaleR_right_const_stochastic: "stochastic_process M t\<^sub>0 (\<lambda>i \<xi>. c i *\<^sub>R (X i \<xi>))"
  by (unfold_locales) simp

lemma add_stochastic:
  assumes "stochastic_process M t\<^sub>0 Y"
  shows "stochastic_process M t\<^sub>0 (\<lambda>i \<xi>. X i \<xi> + Y i \<xi>)"
  using stochastic_process.random_variable[OF assms] random_variable by (unfold_locales) simp

lemma diff_stochastic:
  assumes "stochastic_process M t\<^sub>0 Y"
  shows "stochastic_process M t\<^sub>0 (\<lambda>i \<xi>. X i \<xi> - Y i \<xi>)"
  using stochastic_process.random_variable[OF assms] random_variable by (unfold_locales) simp

lemma uminus_stochastic: "stochastic_process M t\<^sub>0 (-X)" using scaleR_right_const_stochastic[of "\<lambda>_. -1"] by (simp add: fun_Compl_def)

lemma partial_sum_stochastic: "stochastic_process M t\<^sub>0 (\<lambda>n \<xi>. \<Sum>i\<in>{t\<^sub>0..n}. X i \<xi>)" by (unfold_locales) simp

lemma partial_sum'_stochastic: "stochastic_process M t\<^sub>0 (\<lambda>n \<xi>. \<Sum>i\<in>{t\<^sub>0..<n}. X i \<xi>)" by (unfold_locales) simp

end

lemma stochastic_process_sum:
  assumes "\<And>i. i \<in> I \<Longrightarrow> stochastic_process M t\<^sub>0 (X i)"
  shows "stochastic_process M t\<^sub>0 (\<lambda>k \<xi>. \<Sum>i \<in> I. X i k \<xi>)" using assms[THEN stochastic_process.random_variable] by (unfold_locales, auto)

subsubsection \<open>Natural Filtration\<close>

text \<open>The natural filtration induced by a stochastic process \<^term>\<open>X\<close> is the filtration generated by all events involving the process up to the time index \<^term>\<open>t\<close>, i.e. \<open>F t = \<sigma>({X s | s. s \<le> t})\<close>.\<close>

definition natural_filtration :: "'a measure \<Rightarrow> 'b \<Rightarrow> ('b \<Rightarrow> 'a \<Rightarrow> 'c :: topological_space) \<Rightarrow> 'b :: {second_countable_topology, order_topology} \<Rightarrow> 'a measure" where
  "natural_filtration M t\<^sub>0 Y = (\<lambda>t. family_vimage_algebra (space M) {Y i | i. i \<in> {t\<^sub>0..t}} borel)"

abbreviation "nat_natural_filtration \<equiv> \<lambda>M. natural_filtration M (0 :: nat)"
abbreviation "real_natural_filtration \<equiv> \<lambda>M. natural_filtration M (0 :: real)"

lemma space_natural_filtration[simp]: "space (natural_filtration M t\<^sub>0 X t) = space M" unfolding natural_filtration_def space_family_vimage_algebra ..

lemma sets_natural_filtration: "sets (natural_filtration M t\<^sub>0 X t) = sigma_sets (space M) (\<Union>i\<in>{t\<^sub>0..t}. {X i -` A \<inter> space M | A. A \<in> borel})"
  unfolding natural_filtration_def sets_family_vimage_algebra by (intro sigma_sets_eqI) blast+

lemma sets_natural_filtration': 
  assumes "borel = sigma UNIV S"
  shows "sets (natural_filtration M t\<^sub>0 X t) = sigma_sets (space M) (\<Union>i\<in>{t\<^sub>0..t}. {X i -` A \<inter> space M | A. A \<in> S})"
proof (subst sets_natural_filtration, intro sigma_sets_eqI, clarify)
  fix i and A :: "'a set" assume asm: "i \<in> {t\<^sub>0..t}" "A \<in> sets borel"
  hence "A \<in> sigma_sets UNIV S" unfolding assms by simp
  thus "X i -` A \<inter> space M \<in> sigma_sets (space M) (\<Union>i\<in>{t\<^sub>0..t}. {X i -` A \<inter> space M |A. A \<in> S})"
  proof (induction)
    case (Compl a)
    have "X i -` (UNIV - a) \<inter> space M = space M - (X i -` a \<inter> space M)" by blast
    then show ?case using Compl(2)[THEN sigma_sets.Compl] by presburger
  next
    case (Union a)
    have "X i -` \<Union> (range a) \<inter> space M = \<Union> (range (\<lambda>j. X i -` a j \<inter> space M))" by blast
    then show ?case using Union(2)[THEN sigma_sets.Union] by presburger
  qed (auto intro: asm sigma_sets.Empty)
qed (intro sigma_sets.Basic, force simp add: assms)

lemma sets_natural_filtration_open: 
  "sets (natural_filtration M t\<^sub>0 X t) = sigma_sets (space M) (\<Union>i\<in>{t\<^sub>0..t}. {X i -` A \<inter> space M | A. open A})"
  using sets_natural_filtration' by (force simp only: borel_def mem_Collect_eq)

lemma sets_natural_filtration_oi: 
  "sets (natural_filtration M t\<^sub>0 X t) = sigma_sets (space M) (\<Union>i\<in>{t\<^sub>0..t}. {X i -` A \<inter> space M | A :: _ :: {linorder_topology, second_countable_topology} set. A \<in> range greaterThan})" 
  by (rule sets_natural_filtration'[OF borel_Ioi])

lemma sets_natural_filtration_io:
  "sets (natural_filtration M t\<^sub>0 X t) = sigma_sets (space M) (\<Union>i\<in>{t\<^sub>0..t}. {X i -` A \<inter> space M | A :: _ :: {linorder_topology, second_countable_topology} set. A \<in> range lessThan})" 
  by (rule sets_natural_filtration'[OF borel_Iio])

lemma sets_natural_filtration_ci:
  "sets (natural_filtration M t\<^sub>0 X t) = sigma_sets (space M) (\<Union>i\<in>{t\<^sub>0..t}. {X i -` A \<inter> space M | A :: real set. A \<in> range atLeast})" 
  by (rule sets_natural_filtration'[OF borel_Ici])

context stochastic_process
begin

lemma subalgebra_natural_filtration: 
  shows "subalgebra M (natural_filtration M t\<^sub>0 X i)" 
  unfolding subalgebra_def using measurable_family_iff_sets by (force simp add: natural_filtration_def)

lemma filtered_measure_natural_filtration: 
  shows "filtered_measure M (natural_filtration M t\<^sub>0 X) t\<^sub>0"
    by (unfold_locales) (intro subalgebra_natural_filtration, simp only: sets_natural_filtration, intro sigma_sets_subseteq, force) 

text \<open>In order to show that the natural filtration constitutes a filtered \<open>\<sigma>\<close>-finite measure, we need to provide a countable exhausting set in the preimage of \<^term>\<open>X t\<^sub>0\<close>.\<close>

lemma sigma_finite_filtered_measure_natural_filtration:
  assumes exhausting_set: "countable A" "(\<Union>A) = space M" "\<And>a. a \<in> A \<Longrightarrow> emeasure M a \<noteq> \<infinity>" "\<And>a. a \<in> A \<Longrightarrow> \<exists>b \<in> borel. a = X t\<^sub>0 -` b \<inter> space M"
    shows "sigma_finite_filtered_measure M (natural_filtration M t\<^sub>0 X) t\<^sub>0"
proof (unfold_locales)
  have "A \<subseteq> sets (restr_to_subalg M (natural_filtration M t\<^sub>0 X t\<^sub>0))" using exhausting_set by (simp add: sets_restr_to_subalg[OF subalgebra_natural_filtration] sets_natural_filtration) fast
  moreover have "\<Union> A = space (restr_to_subalg M (natural_filtration M t\<^sub>0 X t\<^sub>0))" unfolding space_restr_to_subalg using exhausting_set by simp
  moreover have "\<forall>a\<in>A. emeasure (restr_to_subalg M (natural_filtration M t\<^sub>0 X t\<^sub>0)) a \<noteq> \<infinity>" using calculation(1) exhausting_set(3) 
    by (auto simp add: sets_restr_to_subalg[OF subalgebra_natural_filtration] emeasure_restr_to_subalg[OF subalgebra_natural_filtration])
  ultimately show "\<exists>A. countable A \<and> A \<subseteq> sets (restr_to_subalg M (natural_filtration M t\<^sub>0 X t\<^sub>0)) \<and> \<Union> A = space (restr_to_subalg M (natural_filtration M t\<^sub>0 X t\<^sub>0)) \<and> (\<forall>a\<in>A. emeasure (restr_to_subalg M (natural_filtration M t\<^sub>0 X t\<^sub>0)) a \<noteq> \<infinity>)" using exhausting_set by blast
  show "\<And>i j. \<lbrakk>t\<^sub>0 \<le> i; i \<le> j\<rbrakk> \<Longrightarrow> sets (natural_filtration M t\<^sub>0 X i) \<subseteq> sets (natural_filtration M t\<^sub>0 X j)" using filtered_measure.subalgebra_F[OF filtered_measure_natural_filtration] by (simp add: subalgebra_def)
qed (auto intro: subalgebra_natural_filtration)

lemma finite_filtered_measure_natural_filtration:
  assumes "finite_measure M"
  shows "finite_filtered_measure M (natural_filtration M t\<^sub>0 X) t\<^sub>0" 
  using finite_measure.axioms[OF assms] filtered_measure_natural_filtration by intro_locales

end

text \<open>Filtration generated by independent variables.\<close>

lemma (in prob_space) indep_set_natural_filtration:
  assumes "t\<^sub>0 \<le> s" "s < t" "indep_vars (\<lambda>_. borel) X {t\<^sub>0..}"
  shows "indep_set (natural_filtration M t\<^sub>0 X s) (vimage_algebra (space M) (X t) borel)"
proof -
  have "indep_sets (\<lambda>i. {X i -` A \<inter> space M |A. A \<in> sets borel}) (\<Union>(range (case_bool {t\<^sub>0..s} {t})))" 
    using assms 
    by (intro assms(3)[unfolded indep_vars_def, THEN conjunct2, THEN indep_sets_mono]) (auto simp add: case_bool_if)
  thus ?thesis unfolding indep_set_def using assms
    by (intro indep_sets_cong[THEN iffD1, OF refl _ indep_sets_collect_sigma[of "\<lambda>i. {X i -` A \<inter> space M | A. A \<in> borel}" "case_bool {t\<^sub>0..s} {t}"]])
       (simp add: sets_natural_filtration sets_vimage_algebra split: bool.split, simp, intro Int_stableI, clarsimp, metis sets.Int vimage_Int Int_commute Int_left_absorb Int_left_commute, force simp add: disjoint_family_on_def split: bool.split)
qed

subsection \<open>Adapted Process\<close>

text \<open>We call a collection a stochastic process \<^term>\<open>X\<close> adapted if \<^term>\<open>X i\<close> is \<^term>\<open>F i\<close>-borel-measurable for all indices \<^term>\<open>i :: 't\<close>.\<close>
                                
locale adapted_process = filtered_measure M F t\<^sub>0 for M F t\<^sub>0 and X :: "_ \<Rightarrow> _ \<Rightarrow> _ :: {second_countable_topology, banach}" +
  assumes adapted[measurable]: "\<And>i. t\<^sub>0 \<le> i \<Longrightarrow> X i \<in> borel_measurable (F i)"
begin

lemma adaptedE[elim]:
  assumes "\<lbrakk>\<And>j i. t\<^sub>0 \<le> j \<Longrightarrow> j \<le> i \<Longrightarrow> X j \<in> borel_measurable (F i)\<rbrakk> \<Longrightarrow> P"
  shows P
  using assms using adapted by (metis dual_order.trans borel_measurable_subalgebra sets_F_mono space_F)

lemma adaptedD:
  assumes "t\<^sub>0 \<le> j" "j \<le> i" 
  shows "X j \<in> borel_measurable (F i)" using assms adaptedE by meson

end

lemma (in filtered_measure) adapted_process_const_fun:
  assumes "f \<in> borel_measurable (F t\<^sub>0)"
  shows "adapted_process M F t\<^sub>0 (\<lambda>_. f)"
  using measurable_from_subalg subalgebra_F assms by (unfold_locales) blast

lemma (in filtered_measure) adapted_process_const:
  shows "adapted_process M F t\<^sub>0 (\<lambda>i _. c i)" by (unfold_locales) simp


text \<open>Again, we cover basic operations.\<close>

context adapted_process
begin

lemma compose_adapted:
  assumes "\<And>i. t\<^sub>0 \<le> i \<Longrightarrow> f i \<in> borel_measurable borel"
  shows "adapted_process M F t\<^sub>0 (\<lambda>i \<xi>. (f i) (X i \<xi>))"
  by (unfold_locales) (intro measurable_compose[OF adapted assms])

lemma norm_adapted: "adapted_process M F t\<^sub>0 (\<lambda>i \<xi>. norm (X i \<xi>))" by (fastforce intro: compose_adapted)

lemma scaleR_right_adapted:
  assumes "adapted_process M F t\<^sub>0 R"
  shows "adapted_process M F t\<^sub>0 (\<lambda>i \<xi>. (R i \<xi>) *\<^sub>R (X i \<xi>))"
  using adapted_process.adapted[OF assms] adapted by (unfold_locales) simp
  
lemma scaleR_right_const_fun_adapted:
  assumes "f \<in> borel_measurable (F t\<^sub>0)" 
  shows "adapted_process M F t\<^sub>0 (\<lambda>i \<xi>. f \<xi> *\<^sub>R (X i \<xi>))"
  using assms by (fast intro: scaleR_right_adapted adapted_process_const_fun)

lemma scaleR_right_const_adapted: "adapted_process M F t\<^sub>0 (\<lambda>i \<xi>. c i *\<^sub>R (X i \<xi>))" by (unfold_locales) simp

lemma add_adapted:
  assumes "adapted_process M F t\<^sub>0 Y"
  shows "adapted_process M F t\<^sub>0 (\<lambda>i \<xi>. X i \<xi> + Y i \<xi>)"
  using adapted_process.adapted[OF assms] adapted by (unfold_locales) simp

lemma diff_adapted:
  assumes "adapted_process M F t\<^sub>0 Y"
  shows "adapted_process M F t\<^sub>0 (\<lambda>i \<xi>. X i \<xi> - Y i \<xi>)"
  using adapted_process.adapted[OF assms] adapted by (unfold_locales) simp

lemma uminus_adapted: "adapted_process M F t\<^sub>0 (-X)" using scaleR_right_const_adapted[of "\<lambda>_. -1"] by (simp add: fun_Compl_def)

lemma partial_sum_adapted: "adapted_process M F t\<^sub>0 (\<lambda>n \<xi>. \<Sum>i\<in>{t\<^sub>0..n}. X i \<xi>)" 
proof (unfold_locales)
  fix i :: 'b
  have "X j \<in> borel_measurable (F i)" if "t\<^sub>0 \<le> j" "j \<le> i" for j using that adaptedE by meson
  thus "(\<lambda>\<xi>. \<Sum>i\<in>{t\<^sub>0..i}. X i \<xi>) \<in> borel_measurable (F i)" by simp
qed

lemma partial_sum'_adapted: "adapted_process M F t\<^sub>0 (\<lambda>n \<xi>. \<Sum>i\<in>{t\<^sub>0..<n}. X i \<xi>)" 
proof (unfold_locales)
  fix i :: 'b
  have "X j \<in> borel_measurable (F i)" if "t\<^sub>0 \<le> j" "j < i" for j using that adaptedE by fastforce
  thus "(\<lambda>\<xi>. \<Sum>i\<in>{t\<^sub>0..<i}. X i \<xi>) \<in> borel_measurable (F i)" by simp
qed

end

text \<open>In the dicrete time case, we have the following lemmas which will be useful later on.\<close>

lemma (in nat_filtered_measure) partial_sum_Suc_adapted:
  assumes "adapted_process M F 0 X"
  shows "adapted_process M F 0 (\<lambda>n \<xi>. \<Sum>i<n. X (Suc i) \<xi>)"
proof (unfold_locales)
  interpret adapted_process M F 0 X using assms by blast
  fix i
  have "X j \<in> borel_measurable (F i)" if "j \<le> i" for j using that adaptedD by blast
  thus "(\<lambda>\<xi>. \<Sum>i<i. X (Suc i) \<xi>) \<in> borel_measurable (F i)" by auto
qed

lemma (in enat_filtered_measure) partial_sum_eSuc_adapted:
  assumes "adapted_process M F 0 X"
  shows "adapted_process M F 0 (\<lambda>n \<xi>. \<Sum>i<n. X (eSuc i) \<xi>)"
proof (unfold_locales)
  interpret adapted_process M F 0 X using assms by blast
  fix i
  have "X (eSuc j) \<in> borel_measurable (F i)" if "j < i" for j using that adaptedD by (simp add: ileI1)
  thus "(\<lambda>\<xi>. \<Sum>i<i. X (eSuc i) \<xi>) \<in> borel_measurable (F i)" by auto
qed

lemma (in filtered_measure) adapted_process_sum:
  assumes "\<And>i. i \<in> I \<Longrightarrow> adapted_process M F t\<^sub>0 (X i)"
  shows "adapted_process M F t\<^sub>0 (\<lambda>k \<xi>. \<Sum>i \<in> I. X i k \<xi>)" 
proof -
  {
    fix i k assume "i \<in> I" and asm: "t\<^sub>0 \<le> k"
    then interpret adapted_process M F t\<^sub>0 "X i" using assms by simp
    have "X i k \<in> borel_measurable M" "X i k \<in> borel_measurable (F k)" using measurable_from_subalg subalgebras adapted asm by (blast, simp)
  }
  thus ?thesis by (unfold_locales) simp
qed

text \<open>An adapted process is necessarily a stochastic process.\<close>

sublocale adapted_process \<subseteq> stochastic_process using measurable_from_subalg subalgebras adapted by (unfold_locales) blast

text \<open>A stochastic process is always adapted to the natural filtration it generates.\<close>

lemma (in stochastic_process) adapted_process_natural_filtration: "adapted_process M (natural_filtration M t\<^sub>0 X) t\<^sub>0 X"
  using filtered_measure_natural_filtration
  by (intro_locales) (auto simp add: natural_filtration_def intro!: adapted_process_axioms.intro measurable_family_vimage_algebra) 

subsection \<open>Progressively Measurable Process\<close>

locale progressive_process = filtered_measure M F t\<^sub>0 for M F t\<^sub>0 and X :: "_ \<Rightarrow> _ \<Rightarrow> _ :: {second_countable_topology, banach}" +
  assumes progressive[measurable]: "\<And>t. t\<^sub>0 \<le> t \<Longrightarrow> (\<lambda>(i, x). X i x) \<in> borel_measurable (restrict_space borel {t\<^sub>0..t} \<Otimes>\<^sub>M F t)"
begin

lemma progressiveD:
  assumes "S \<in> borel"
  shows "(\<lambda>(j, \<xi>). X j \<xi>) -` S \<inter> ({t\<^sub>0..i} \<times> space M) \<in> (restrict_space borel {t\<^sub>0..i} \<Otimes>\<^sub>M F i)" 
  using measurable_sets[OF progressive, OF _ assms, of i]
  by (cases "t\<^sub>0 \<le> i") (auto simp add: space_restrict_space sets_pair_measure space_pair_measure)

end

lemma (in filtered_measure) progressive_process_const_fun:
  assumes "f \<in> borel_measurable (F t\<^sub>0)"
  shows "progressive_process M F t\<^sub>0 (\<lambda>_. f)"
proof (unfold_locales)
  fix i assume asm: "t\<^sub>0 \<le> i"
  have "f \<in> borel_measurable (F i)" using borel_measurable_mono[OF order.refl asm] assms by blast
  thus "case_prod (\<lambda>_. f) \<in> borel_measurable (restrict_space borel {t\<^sub>0..i} \<Otimes>\<^sub>M F i)" using measurable_compose[OF measurable_snd] by simp
qed

lemma (in filtered_measure) progressive_process_const:
  assumes "c \<in> borel_measurable borel"
  shows "progressive_process M F t\<^sub>0 (\<lambda>i _. c i)"
  using assms by (unfold_locales) (auto simp add: measurable_split_conv intro!: measurable_compose[OF measurable_fst] measurable_restrict_space1)

context progressive_process
begin

lemma compose_progressive:
  assumes "case_prod f \<in> borel_measurable borel"
  shows "progressive_process M F t\<^sub>0 (\<lambda>i \<xi>. (f i) (X i \<xi>))"
proof
  fix i assume asm: "t\<^sub>0 \<le> i"
  have "(\<lambda>(j, \<xi>). (j, X j \<xi>)) \<in> (restrict_space borel {t\<^sub>0..i} \<Otimes>\<^sub>M F i) \<rightarrow>\<^sub>M borel \<Otimes>\<^sub>M borel" 
    using progressive[OF asm] measurable_fst''[OF measurable_restrict_space1, OF measurable_id] 
    by (auto simp add: measurable_pair_iff measurable_split_conv)
  moreover have "(\<lambda>(j, \<xi>). f j (X j \<xi>)) = case_prod f o ((\<lambda>(j, y). (j, y)) o (\<lambda>(j, \<xi>). (j, X j \<xi>)))" by fastforce
  ultimately show "(\<lambda>(j, \<xi>). (f j) (X j \<xi>)) \<in> borel_measurable (restrict_space borel {t\<^sub>0..i} \<Otimes>\<^sub>M F i)" using assms by (simp add: borel_prod)
qed

lemma norm_progressive: "progressive_process M F t\<^sub>0 (\<lambda>i \<xi>. norm (X i \<xi>))" using measurable_compose[OF progressive borel_measurable_norm] by (unfold_locales) simp

lemma scaleR_right_progressive:
  assumes "progressive_process M F t\<^sub>0 R"
  shows "progressive_process M F t\<^sub>0 (\<lambda>i \<xi>. (R i \<xi>) *\<^sub>R (X i \<xi>))"
  using progressive_process.progressive[OF assms] by (unfold_locales) (simp add: progressive assms)
  
lemma scaleR_right_const_fun_progressive: 
  assumes "f \<in> borel_measurable (F t\<^sub>0)" 
  shows "progressive_process M F t\<^sub>0 (\<lambda>i \<xi>. f \<xi> *\<^sub>R (X i \<xi>))"
  using assms by (fast intro: scaleR_right_progressive progressive_process_const_fun)

lemma scaleR_right_const_progressive: 
  assumes "c \<in> borel_measurable borel"
  shows "progressive_process M F t\<^sub>0 (\<lambda>i \<xi>. c i *\<^sub>R (X i \<xi>))" 
  using assms by (fastforce intro: scaleR_right_progressive progressive_process_const)

lemma add_progressive:
  assumes "progressive_process M F t\<^sub>0 Y"
  shows "progressive_process M F t\<^sub>0 (\<lambda>i \<xi>. X i \<xi> + Y i \<xi>)"
  using progressive_process.progressive[OF assms] by (unfold_locales) (simp add: progressive assms)

lemma diff_progressive:
  assumes "progressive_process M F t\<^sub>0 Y"
  shows "progressive_process M F t\<^sub>0 (\<lambda>i \<xi>. X i \<xi> - Y i \<xi>)"
  using progressive_process.progressive[OF assms] by (unfold_locales) (simp add: progressive assms)

lemma uminus_progressive: "progressive_process M F t\<^sub>0 (-X)" using scaleR_right_const_progressive[of "\<lambda>_. -1"] by (simp add: fun_Compl_def)

end

text \<open>A progressively measurable process is also adapted.\<close>

sublocale progressive_process \<subseteq> adapted_process using measurable_compose_rev[OF progressive measurable_Pair1'] 
  unfolding prod.case space_restrict_space
  by unfold_locales simp

text \<open>In the discrete setting, adaptedness is equivalent to progressive measurability.\<close>

theorem (in nat_filtered_measure) progressive_iff_adapted: "progressive_process M F 0 X \<longleftrightarrow> adapted_process M F 0 X"
proof (intro iffI)
  assume asm: "progressive_process M F 0 X"
  interpret progressive_process M F 0 X by (rule asm)
  show "adapted_process M F 0 X" ..
next
  assume asm: "adapted_process M F 0 X"
  interpret adapted_process M F 0 X by (rule asm) 
  show "progressive_process M F 0 X"
  proof (unfold_locales, intro borel_measurableI)
    fix S :: "'b set" and i :: nat assume open_S: "open S"
    {
      fix j assume asm: "j \<le> i"
      hence "X j -` S \<inter> space M \<in> F i" using adaptedD[of j, THEN measurable_sets] space_F open_S by fastforce
      moreover have "case_prod X -` S \<inter> {j} \<times> space M = {j} \<times> (X j -` S \<inter> space M)" for j by fast
      moreover have "{j :: nat} \<in> restrict_space borel {0..i}" using asm by (simp add: sets_restrict_space_iff)
      ultimately have "case_prod X -` S \<inter> {j} \<times> space M \<in> restrict_space borel {0..i} \<Otimes>\<^sub>M F i" by simp
    }
    hence "(\<lambda>j. (\<lambda>(x, y). X x y) -` S \<inter> {j} \<times> space M) ` {..i} \<subseteq> restrict_space borel {0..i} \<Otimes>\<^sub>M F i" by blast
    moreover have "case_prod X -` S \<inter> space (restrict_space borel {0..i} \<Otimes>\<^sub>M F i) = (\<Union>j\<le>i. case_prod X -` S \<inter> {j} \<times> space M)" unfolding space_pair_measure space_restrict_space space_F by force  
    ultimately show "case_prod X -` S \<inter> space (restrict_space borel {0..i} \<Otimes>\<^sub>M F i) \<in> restrict_space borel {0..i} \<Otimes>\<^sub>M F i" by (metis sets.countable_UN)
  qed
qed

theorem (in enat_filtered_measure) progressive_iff_adapted: "progressive_process M F 0 X \<longleftrightarrow> adapted_process M F 0 X"
proof (intro iffI)
  assume asm: "progressive_process M F 0 X"
  interpret progressive_process M F 0 X by (rule asm)
  show "adapted_process M F 0 X" ..
next
  assume asm: "adapted_process M F 0 X"
  interpret adapted_process M F 0 X by (rule asm) 
  show "progressive_process M F 0 X"
  proof (unfold_locales, intro borel_measurableI)
    fix S :: "'b set" and i :: enat assume open_S: "open S"
    {
      fix j assume asm: "j \<le> i"
      hence "X j -` S \<inter> space M \<in> F i" using adaptedD[of j, THEN measurable_sets] space_F open_S by fastforce
      moreover have "case_prod X -` S \<inter> {j} \<times> space M = {j} \<times> (X j -` S \<inter> space M)" for j by fast
      moreover have "{j :: enat} \<in> restrict_space borel {0..i}" using asm by (simp add: sets_restrict_space_iff)
      ultimately have "case_prod X -` S \<inter> {j} \<times> space M \<in> restrict_space borel {0..i} \<Otimes>\<^sub>M F i" by simp
    }
    hence "(\<lambda>j. (\<lambda>(x, y). X x y) -` S \<inter> {j} \<times> space M) ` {..i} \<subseteq> restrict_space borel {0..i} \<Otimes>\<^sub>M F i" by blast
    moreover have "case_prod X -` S \<inter> space (restrict_space borel {0..i} \<Otimes>\<^sub>M F i) = (\<Union>j\<le>i. case_prod X -` S \<inter> {j} \<times> space M)" unfolding space_pair_measure space_restrict_space space_F by force  
    ultimately show "case_prod X -` S \<inter> space (restrict_space borel {0..i} \<Otimes>\<^sub>M F i) \<in> restrict_space borel {0..i} \<Otimes>\<^sub>M F i" by (metis sets.countable_UN)
  qed
qed

subsection \<open>Predictable Process\<close>

text \<open>We introduce the constant \<^term>\<open>\<Sigma>\<^sub>P\<close> to denote the predictable \<open>\<sigma>\<close>-algebra.\<close>

context linearly_filtered_measure
begin

definition \<Sigma>\<^sub>P :: "('b \<times> 'a) measure" where predictable_sigma: "\<Sigma>\<^sub>P \<equiv> sigma ({t\<^sub>0..} \<times> space M) ({{s<..t} \<times> A | A s t. A \<in> F s \<and> t\<^sub>0 \<le> s \<and> s < t} \<union> {{t\<^sub>0} \<times> A | A. A \<in> F t\<^sub>0})"

lemma space_predictable_sigma[simp]: "space \<Sigma>\<^sub>P = ({t\<^sub>0..} \<times> space M)" unfolding predictable_sigma space_measure_of_conv by blast

lemma sets_predictable_sigma: "sets \<Sigma>\<^sub>P = sigma_sets ({t\<^sub>0..} \<times> space M) ({{s<..t} \<times> A | A s t. A \<in> F s \<and> t\<^sub>0 \<le> s \<and> s < t} \<union> {{t\<^sub>0} \<times> A | A. A \<in> F t\<^sub>0})" 
  unfolding predictable_sigma using space_F sets.sets_into_space by (subst sets_measure_of) fastforce+

lemma measurable_predictable_sigma_snd:
  assumes "countable \<I>" "\<I> \<subseteq> {{s<..t} | s t. t\<^sub>0 \<le> s \<and> s < t}" "{t\<^sub>0<..} \<subseteq> (\<Union>\<I>)"
  shows "snd \<in> \<Sigma>\<^sub>P \<rightarrow>\<^sub>M F t\<^sub>0"
proof (intro measurableI)
  fix S :: "'a set" assume asm: "S \<in> F t\<^sub>0"
  have countable: "countable ((\<lambda>I. I \<times> S) ` \<I>)" using assms(1) by blast
  have "(\<lambda>I. I \<times> S) ` \<I> \<subseteq> {{s<..t} \<times> A | A s t. A \<in> F s \<and> t\<^sub>0 \<le> s \<and>  s < t}" using sets_F_mono[OF order_refl, THEN subsetD, OF _ asm] assms(2) by blast
  hence "(\<Union>I\<in>\<I>. I \<times> S) \<union> {t\<^sub>0} \<times> S \<in> \<Sigma>\<^sub>P" unfolding sets_predictable_sigma using asm by (intro sigma_sets_Un[OF sigma_sets_UNION[OF countable] sigma_sets.Basic] sigma_sets.Basic) blast+
  moreover have "snd -` S \<inter> space \<Sigma>\<^sub>P = {t\<^sub>0..} \<times> S" using sets.sets_into_space[OF asm] by fastforce
  moreover have "{t\<^sub>0} \<union> {t\<^sub>0<..} = {t\<^sub>0..}" by auto
  moreover have "(\<Union>I\<in>\<I>. I \<times> S) \<union> {t\<^sub>0} \<times> S = {t\<^sub>0..} \<times> S" using assms(2,3) calculation(3) by fastforce
  ultimately show "snd -` S \<inter> space \<Sigma>\<^sub>P \<in> \<Sigma>\<^sub>P" by argo
qed (auto)

lemma measurable_predictable_sigma_fst:
  assumes "countable \<I>" "\<I> \<subseteq> {{s<..t} | s t. t\<^sub>0 \<le> s \<and> s < t}" "{t\<^sub>0<..} \<subseteq> (\<Union>\<I>)"
  shows "fst \<in> \<Sigma>\<^sub>P \<rightarrow>\<^sub>M borel"
proof -
  have "A \<times> space M \<in> sets \<Sigma>\<^sub>P" if "A \<in> sigma_sets {t\<^sub>0..} {{s<..t} | s t. t\<^sub>0 \<le> s \<and> s < t}" for A unfolding sets_predictable_sigma using that 
  proof (induction rule: sigma_sets.induct)
    case (Basic a)
    thus ?case using space_F sets.top by blast
  next
    case (Compl a)
    have "({t\<^sub>0..} - a) \<times> space M = {t\<^sub>0..} \<times> space M - a \<times> space M" by blast
    then show ?case using Compl(2)[THEN sigma_sets.Compl] by presburger
  next
    case (Union a)
    have "\<Union> (range a) \<times> space M = \<Union> (range (\<lambda>i. a i \<times> space M))" by blast
    then show ?case using Union(2)[THEN sigma_sets.Union] by presburger
  qed (auto)
  moreover have "restrict_space borel {t\<^sub>0..} = sigma {t\<^sub>0..} {{s<..t} | s t. t\<^sub>0 \<le> s \<and> s < t}"
  proof -
    have "sigma_sets {t\<^sub>0..} ((\<inter>) {t\<^sub>0..} ` sigma_sets UNIV (range greaterThan)) = sigma_sets {t\<^sub>0..} {{s<..t} |s t. t\<^sub>0 \<le> s \<and> s < t}"
    proof (intro sigma_sets_eqI ; clarify)
      fix A :: "'b set" assume asm: "A \<in> sigma_sets UNIV (range greaterThan)"
      thus "{t\<^sub>0..} \<inter> A \<in> sigma_sets {t\<^sub>0..} {{s<..t} |s t. t\<^sub>0 \<le> s \<and> s < t}"
      proof (induction rule: sigma_sets.induct)
        case (Basic a)
        then obtain s where s: "a = {s<..}" by blast
        show ?case
        proof (cases "t\<^sub>0 \<le> s")
          case True
          hence *: "{t\<^sub>0..} \<inter> a = (\<Union>i \<in> \<I>. {s<..} \<inter> i)" using s assms(3) by force
          have "((\<inter>) {s<..} ` \<I>) \<subseteq> sigma_sets {t\<^sub>0..} {{s<..t} | s t. t\<^sub>0 \<le> s \<and> s < t}"
          proof (clarify)
            fix A assume "A \<in> \<I>"
            then obtain s' t' where A: "A = {s'<..t'}" "t\<^sub>0 \<le> s'" "s' < t'" using assms(2) by blast
            hence "{s<..} \<inter> A = {max s s'<..t'}" by fastforce
            moreover have "t\<^sub>0 \<le> max s s'" using A True by linarith
            moreover have "max s s' < t'" if "s < t'" using A that by linarith
            moreover have "{s<..} \<inter> A = {}" if "\<not> s < t'" using A that by force
            ultimately show "{s<..} \<inter> A \<in> sigma_sets {t\<^sub>0..} {{s<..t} |s t. t\<^sub>0 \<le> s \<and> s < t}" by (cases "s < t'") (blast, simp add: sigma_sets.Empty)
          qed
          thus ?thesis unfolding * using assms(1) by (intro sigma_sets_UNION) auto
        next
          case False
          hence "{t\<^sub>0..} \<inter> a = {t\<^sub>0..}" using s by force
          thus ?thesis using sigma_sets_top by auto
        qed
      next
        case (Compl a)
        have "{t\<^sub>0..} \<inter> (UNIV - a) = {t\<^sub>0..} - ({t\<^sub>0..} \<inter> a)" by blast
        then show ?case using Compl(2)[THEN sigma_sets.Compl] by presburger
      next
        case (Union a)
        have "{t\<^sub>0..} \<inter> \<Union> (range a) = \<Union> (range (\<lambda>i. {t\<^sub>0..} \<inter> a i))" by blast
        then show ?case using Union(2)[THEN sigma_sets.Union] by presburger
      qed (simp add: sigma_sets.Empty)
    next 
      fix s t assume asm: "t\<^sub>0 \<le> s" "s < t"
      hence *: "{s<..t} = {s<..} \<inter> ({t\<^sub>0..} - {t<..})" by force
      have "{s<..} \<in> sigma_sets {t\<^sub>0..} ((\<inter>) {t\<^sub>0..} ` sigma_sets UNIV (range greaterThan))" using asm by (intro sigma_sets.Basic) auto
      moreover have "{t\<^sub>0..} - {t<..} \<in> sigma_sets {t\<^sub>0..} ((\<inter>) {t\<^sub>0..} ` sigma_sets UNIV (range greaterThan))" using asm by (intro sigma_sets.Compl sigma_sets.Basic) auto
      ultimately show "{s<..t} \<in> sigma_sets {t\<^sub>0..} ((\<inter>) {t\<^sub>0..} ` sigma_sets UNIV (range greaterThan))" unfolding * Int_range_binary[of "{s<..}"] by (intro sigma_sets_Inter[OF _ binary_in_sigma_sets]) auto        
    qed
    thus ?thesis unfolding borel_Ioi restrict_space_def emeasure_sigma by (force intro: sigma_eqI)
  qed
  ultimately have "restrict_space borel {t\<^sub>0..} \<Otimes>\<^sub>M sigma (space M) {} \<subseteq> sets \<Sigma>\<^sub>P" 
    unfolding sets_pair_measure space_restrict_space space_measure_of_conv
    using space_predictable_sigma sets.sigma_algebra_axioms[of \<Sigma>\<^sub>P] 
    by (intro sigma_algebra.sigma_sets_subset) (auto simp add: sigma_sets_empty_eq sets_measure_of_conv)
  moreover have "space (restrict_space borel {t\<^sub>0..} \<Otimes>\<^sub>M sigma (space M) {}) = space \<Sigma>\<^sub>P" by (simp add: space_pair_measure)
  moreover have "fst \<in> restrict_space borel {t\<^sub>0..} \<Otimes>\<^sub>M sigma (space M) {} \<rightarrow>\<^sub>M borel" by (fastforce intro: measurable_fst''[OF measurable_restrict_space1, of "\<lambda>x. x"]) 
  ultimately show ?thesis by (meson borel_measurable_subalgebra)
qed

end

locale predictable_process = linearly_filtered_measure M F t\<^sub>0 for M F t\<^sub>0 and X :: "_ \<Rightarrow> _ \<Rightarrow> _ :: {second_countable_topology, banach}" +
  assumes predictable: "(\<lambda>(t, x). X t x) \<in> borel_measurable \<Sigma>\<^sub>P"
begin

lemmas predictableD = measurable_sets[OF predictable, unfolded space_predictable_sigma]

end

lemma (in nat_filtered_measure) measurable_predictable_sigma_snd':
  shows "snd \<in> \<Sigma>\<^sub>P \<rightarrow>\<^sub>M F 0"
  by (intro measurable_predictable_sigma_snd[of "range (\<lambda>x. {Suc x})"]) (force | simp add: greaterThan_0)+

lemma (in nat_filtered_measure) measurable_predictable_sigma_fst':
  shows "fst \<in> \<Sigma>\<^sub>P \<rightarrow>\<^sub>M borel"
  by (intro measurable_predictable_sigma_fst[of "range (\<lambda>x. {Suc x})"]) (force | simp add: greaterThan_0)+

lemma (in enat_filtered_measure) measurable_predictable_sigma_snd':
  shows "snd \<in> \<Sigma>\<^sub>P \<rightarrow>\<^sub>M F 0"
  by (intro measurable_predictable_sigma_snd[of "{{0<..\<infinity>}}"]) force+

lemma (in enat_filtered_measure) measurable_predictable_sigma_fst':
  shows "fst \<in> \<Sigma>\<^sub>P \<rightarrow>\<^sub>M borel"
  by (intro measurable_predictable_sigma_fst[of "{{0<..\<infinity>}}"]) force+

lemma (in real_filtered_measure) measurable_predictable_sigma_snd':
  shows "snd \<in> \<Sigma>\<^sub>P \<rightarrow>\<^sub>M F 0"
  using real_arch_simple by (intro measurable_predictable_sigma_snd[of "range (\<lambda>x::nat. {0<..real (Suc x)})"]) (fastforce intro: add_increasing)+

lemma (in real_filtered_measure) measurable_predictable_sigma_fst':
  shows "fst \<in> \<Sigma>\<^sub>P \<rightarrow>\<^sub>M borel"
  using real_arch_simple by (intro measurable_predictable_sigma_fst[of "range (\<lambda>x::nat. {0<..real (Suc x)})"]) (fastforce intro: add_increasing)+

lemma (in ennreal_filtered_measure) measurable_predictable_sigma_snd':
  shows "snd \<in> \<Sigma>\<^sub>P \<rightarrow>\<^sub>M F 0"
  by (intro measurable_predictable_sigma_snd[of "{{0<..\<infinity>}}"]) force+

lemma (in ennreal_filtered_measure) measurable_predictable_sigma_fst':
  shows "fst \<in> \<Sigma>\<^sub>P \<rightarrow>\<^sub>M borel"
  by (intro measurable_predictable_sigma_fst[of "{{0<..\<infinity>}}"]) force+

text \<open>We show sufficient conditions for functions constant in one argument to constitute a predictable process. In contrast to the cases before, this is not a triviality.\<close>

lemma (in linearly_filtered_measure) predictable_process_const_fun:
  assumes "snd \<in> \<Sigma>\<^sub>P \<rightarrow>\<^sub>M F t\<^sub>0" "f \<in> borel_measurable (F t\<^sub>0)"
    shows "predictable_process M F t\<^sub>0 (\<lambda>_. f)"
  using measurable_compose_rev[OF assms(2)] assms(1) by (unfold_locales) (auto simp add: measurable_split_conv)

lemma (in nat_filtered_measure) predictable_process_const_fun'[intro]:
  assumes "f \<in> borel_measurable (F 0)"
  shows "predictable_process M F 0 (\<lambda>_. f)"
  using assms by (intro predictable_process_const_fun[OF measurable_predictable_sigma_snd'])

lemma (in enat_filtered_measure) predictable_process_const_fun'[intro]:
  assumes "f \<in> borel_measurable (F 0)"
  shows "predictable_process M F 0 (\<lambda>_. f)"
  using assms by (intro predictable_process_const_fun[OF measurable_predictable_sigma_snd'])

lemma (in real_filtered_measure) predictable_process_const_fun'[intro]:
  assumes "f \<in> borel_measurable (F 0)"
  shows "predictable_process M F 0 (\<lambda>_. f)"
  using assms by (intro predictable_process_const_fun[OF measurable_predictable_sigma_snd'])

lemma (in ennreal_filtered_measure) predictable_process_const_fun'[intro]:
  assumes "f \<in> borel_measurable (F 0)"
  shows "predictable_process M F 0 (\<lambda>_. f)"
  using assms by (intro predictable_process_const_fun[OF measurable_predictable_sigma_snd'])

lemma (in linearly_filtered_measure) predictable_process_const:
  assumes "fst \<in> borel_measurable \<Sigma>\<^sub>P" "c \<in> borel_measurable borel"
  shows "predictable_process M F t\<^sub>0 (\<lambda>i _. c i)"
  using assms by (unfold_locales) (simp add: measurable_split_conv)

lemma (in linearly_filtered_measure) predictable_process_const_const[intro]:
  shows "predictable_process M F t\<^sub>0 (\<lambda>_ _. c)"
  by (unfold_locales) simp

lemma (in nat_filtered_measure) predictable_process_const'[intro]:
  assumes "c \<in> borel_measurable borel"
  shows "predictable_process M F 0 (\<lambda>i _. c i)"
  using assms by (intro predictable_process_const[OF measurable_predictable_sigma_fst'])

lemma (in enat_filtered_measure) predictable_process_const'[intro]:
  assumes "c \<in> borel_measurable borel"
  shows "predictable_process M F 0 (\<lambda>i _. c i)"
  using assms by (intro predictable_process_const[OF measurable_predictable_sigma_fst'])

lemma (in real_filtered_measure) predictable_process_const'[intro]:
  assumes "c \<in> borel_measurable borel"
  shows "predictable_process M F 0 (\<lambda>i _. c i)"
  using assms by (intro predictable_process_const[OF measurable_predictable_sigma_fst'])

lemma (in ennreal_filtered_measure) predictable_process_const'[intro]:
  assumes "c \<in> borel_measurable borel"
  shows "predictable_process M F 0 (\<lambda>i _. c i)"
  using assms by (intro predictable_process_const[OF measurable_predictable_sigma_fst'])

context predictable_process
begin

lemma compose_predictable:
  assumes "fst \<in> borel_measurable \<Sigma>\<^sub>P" "case_prod f \<in> borel_measurable borel"
  shows "predictable_process M F t\<^sub>0 (\<lambda>i \<xi>. (f i) (X i \<xi>))"
proof
  have "(\<lambda>(i, \<xi>). (i, X i \<xi>)) \<in> \<Sigma>\<^sub>P \<rightarrow>\<^sub>M borel \<Otimes>\<^sub>M borel" using predictable assms(1) by (auto simp add: measurable_pair_iff measurable_split_conv)
  moreover have "(\<lambda>(i, \<xi>). f i (X i \<xi>)) = case_prod f o (\<lambda>(i, \<xi>). (i, X i \<xi>))" by fastforce
  ultimately show "(\<lambda>(i, \<xi>). f i (X i \<xi>)) \<in> borel_measurable \<Sigma>\<^sub>P" unfolding borel_prod using assms by simp
qed

lemma norm_predictable: "predictable_process M F t\<^sub>0 (\<lambda>i \<xi>. norm (X i \<xi>))" using measurable_compose[OF predictable borel_measurable_norm] 
  by (unfold_locales) (simp add: prod.case_distrib)

lemma scaleR_right_predictable:
  assumes "predictable_process M F t\<^sub>0 R"
  shows "predictable_process M F t\<^sub>0 (\<lambda>i \<xi>. (R i \<xi>) *\<^sub>R (X i \<xi>))"
  using predictable predictable_process.predictable[OF assms] by (unfold_locales) (auto simp add: measurable_split_conv)

lemma scaleR_right_const_fun_predictable: 
  assumes "snd \<in> \<Sigma>\<^sub>P \<rightarrow>\<^sub>M F t\<^sub>0" "f \<in> borel_measurable (F t\<^sub>0)" 
  shows "predictable_process M F t\<^sub>0 (\<lambda>i \<xi>. f \<xi> *\<^sub>R (X i \<xi>))"
  using assms by (fast intro: scaleR_right_predictable predictable_process_const_fun)

lemma scaleR_right_const_predictable: 
  assumes "fst \<in> borel_measurable \<Sigma>\<^sub>P" "c \<in> borel_measurable borel"
  shows "predictable_process M F t\<^sub>0 (\<lambda>i \<xi>. c i *\<^sub>R (X i \<xi>))" 
  using assms by (fastforce intro: scaleR_right_predictable predictable_process_const)

lemma scaleR_right_const'_predictable: "predictable_process M F t\<^sub>0 (\<lambda>i \<xi>. c *\<^sub>R (X i \<xi>))" 
  by (fastforce intro: scaleR_right_predictable)

lemma add_predictable:
  assumes "predictable_process M F t\<^sub>0 Y"
  shows "predictable_process M F t\<^sub>0 (\<lambda>i \<xi>. X i \<xi> + Y i \<xi>)"
  using predictable predictable_process.predictable[OF assms] by (unfold_locales) (auto simp add: measurable_split_conv)

lemma diff_predictable:
  assumes "predictable_process M F t\<^sub>0 Y"
  shows "predictable_process M F t\<^sub>0 (\<lambda>i \<xi>. X i \<xi> - Y i \<xi>)"
  using predictable predictable_process.predictable[OF assms] by (unfold_locales) (auto simp add: measurable_split_conv)

lemma uminus_predictable: "predictable_process M F t\<^sub>0 (-X)" using scaleR_right_const'_predictable[of "-1"] by (simp add: fun_Compl_def)

end

text \<open>Every predictable process is also progressively measurable.\<close>

sublocale predictable_process \<subseteq> progressive_process
proof (unfold_locales)
  fix i :: 'b assume asm: "t\<^sub>0 \<le> i"
  {
    fix S :: "('b \<times> 'a) set" assume "S \<in> {{s<..t} \<times> A | A s t. A \<in> F s \<and> t\<^sub>0 \<le> s \<and> s < t} \<union> {{t\<^sub>0} \<times> A | A. A \<in> F t\<^sub>0}"
    hence "(\<lambda>x. x) -` S \<inter> ({t\<^sub>0..i} \<times> space M) \<in> restrict_space borel {t\<^sub>0..i} \<Otimes>\<^sub>M F i"
    proof
      assume "S \<in> {{s<..t} \<times> A | A s t. A \<in> F s \<and> t\<^sub>0 \<le> s \<and> s < t}"
      then obtain s t A where S_is: "S = {s<..t} \<times> A" "t\<^sub>0 \<le> s" "s < t" "A \<in> F s" by blast
      hence "(\<lambda>x. x) -` S \<inter> ({t\<^sub>0..i} \<times> space M) = {s<..min i t} \<times> A" using sets.sets_into_space[OF S_is(4)] by auto
      then show ?thesis using S_is sets_F_mono[of s i] by (cases "s \<le> i") (fastforce simp add: sets_restrict_space_iff)+
    next
      assume "S \<in> {{t\<^sub>0} \<times> A | A. A \<in> F t\<^sub>0}"
      then obtain A where S_is: "S = {t\<^sub>0} \<times> A" "A \<in> F t\<^sub>0" by blast
      hence "(\<lambda>x. x) -` S \<inter> ({t\<^sub>0..i} \<times> space M) = {t\<^sub>0} \<times> A" using asm sets.sets_into_space[OF S_is(2)] by auto
      thus ?thesis using S_is(2) sets_F_mono[OF order_refl asm] asm by (fastforce simp add: sets_restrict_space_iff)
    qed
    hence "(\<lambda>x. x) -` S \<inter> space (restrict_space borel {t\<^sub>0..i} \<Otimes>\<^sub>M F i) \<in> restrict_space borel {t\<^sub>0..i} \<Otimes>\<^sub>M F i" by (simp add: space_pair_measure space_F[OF asm])
  }
  moreover have "{{s<..t} \<times> A |A s t. A \<in> sets (F s) \<and> t\<^sub>0 \<le> s \<and> s < t} \<union> {{t\<^sub>0} \<times> A |A. A \<in> sets (F t\<^sub>0)} \<subseteq> Pow ({t\<^sub>0..} \<times> space M)" using sets.sets_into_space by force
  ultimately have "(\<lambda>x. x) \<in> restrict_space borel {t\<^sub>0..i} \<Otimes>\<^sub>M F i \<rightarrow>\<^sub>M \<Sigma>\<^sub>P" using space_F[OF asm] by (intro measurable_sigma_sets[OF sets_predictable_sigma]) (fast, force simp add: space_pair_measure)
  thus "case_prod X \<in> borel_measurable (restrict_space borel {t\<^sub>0..i} \<Otimes>\<^sub>M F i)" using predictable by simp
qed

text \<open>The following lemma characterizes predictability in a discrete-time setting.\<close>

lemma (in nat_filtered_measure) sets_in_filtration:
  assumes "(\<Union>i. {i} \<times> A i) \<in> \<Sigma>\<^sub>P"
  shows "A (Suc i) \<in> F i" "A 0 \<in> F 0"
  using assms unfolding sets_predictable_sigma
proof (induction "(\<Union>i. {i} \<times> A i)" arbitrary: A)
  case Basic
  {
    assume "\<exists>S. (\<Union>i. {i} \<times> A i) = {0} \<times> S"
    then obtain S where S: "(\<Union>i. {i} \<times> A i) = {0} \<times> S" by blast
    hence "S \<in> F 0" using Basic by (fastforce simp add: times_eq_iff)
    moreover have "A i = {}" if "i \<noteq> 0" for i using that S unfolding bot_nat_def[symmetric] by blast
    moreover have "A 0 = S" using S by blast
    ultimately have "A 0 \<in> F 0" "A (Suc i) \<in> F i" for i by auto
  }
  note * = this
  {
    assume "\<nexists>S. (\<Union>i. {i} \<times> A i) = {0} \<times> S"
    then obtain s t B where B: "(\<Union>i. {i} \<times> A i) = {s<..t} \<times> B" "B \<in> sets (F s)" "s < t" using Basic by auto
    hence "A i = B" if "i \<in> {s<..t}" for i using that by fast
    moreover have "A i = {}" if "i \<notin> {s<..t}" for i using B that by fastforce
    ultimately have "A 0 \<in> F 0" "A (Suc i) \<in> F i" for i using B sets_F_mono 
      by (simp, metis less_Suc_eq_le sets.empty_sets subset_eq bot_nat_0.extremum greaterThanAtMost_iff)
      
  }
  note ** = this
  show "A (Suc i) \<in> sets (F i)" "A 0 \<in> F 0" using *(2)[of i] *(1) **(2)[of i] **(1) by blast+
next
  case Empty
  {
    case 1
    then show ?case using Empty by simp
  next
    case 2
    then show ?case using Empty by simp
  }
next
  case (Compl a)
  have a_in: "a \<subseteq> {0..} \<times> space M" using Compl(1) sets.sets_into_space sets_predictable_sigma space_predictable_sigma by metis
  hence A_in: "A i \<subseteq> space M" for i using Compl(4) by blast
  have a: "a = {0..} \<times> space M - (\<Union>i. {i} \<times> A i)" using a_in Compl(4) by blast
  also have "... = - (\<Inter>j. - ({j} \<times> (space M - A j)))" by blast
  also have "... = (\<Union>j. {j} \<times> (space M - A j))" by blast
  finally have *: "(space M - A (Suc i)) \<in> F i" "(space M - A 0) \<in> F 0" using Compl(2,3) by auto
  {
    case 1
    then show ?case using * A_in by (metis bot_nat_0.extremum double_diff sets.Diff sets.top sets_F_mono sets_le_imp_space_le space_F)
 next
    case 2
    then show ?case using * A_in by (metis bot_nat_0.extremum double_diff sets.Diff sets.top sets_F_mono sets_le_imp_space_le space_F)
  }
next
  case (Union a)
  have a_in: "a i \<subseteq> {0..} \<times> space M" for i using Union(1) sets.sets_into_space sets_predictable_sigma space_predictable_sigma by metis
  hence A_in: "A i \<subseteq> space M" for i using Union(4) by blast
  have "snd x \<in> snd ` (a i \<inter> ({fst x} \<times> space M))" if "x \<in> a i" for i x using that a_in by fastforce
  hence a_i: "a i = (\<Union>j. {j} \<times> (snd ` (a i \<inter> ({j} \<times> space M))))" for i by force
  have A_i: "A i = snd ` (\<Union> (range a) \<inter> ({i} \<times> space M))" for i unfolding Union(4) using A_in by force 
  have *: "snd ` (a j \<inter> ({Suc i} \<times> space M)) \<in> F i" "snd ` (a j \<inter> ({0} \<times> space M)) \<in> F 0" for j using Union(2,3)[OF a_i] by auto
  {
    case 1
    have "(\<Union>j. snd ` (a j \<inter> ({Suc i} \<times> space M))) \<in> F i" using * by fast
    moreover have "(\<Union>j. snd ` (a j \<inter> ({Suc i} \<times> space M))) = snd ` (\<Union> (range a) \<inter> ({Suc i} \<times> space M))" by fast
    ultimately show ?case using A_i by metis
  next
    case 2
    have "(\<Union>j. snd ` (a j \<inter> ({0} \<times> space M))) \<in> F 0" using * by fast
    moreover have "(\<Union>j. snd ` (a j \<inter> ({0} \<times> space M))) = snd ` (\<Union> (range a) \<inter> ({0} \<times> space M))" by fast
    ultimately show ?case using A_i by metis
  }
qed

text \<open>This leads to the following useful fact.\<close>

lemma (in nat_filtered_measure) predictable_implies_adapted_Suc: 
  assumes "predictable_process M F 0 X"
  shows "adapted_process M F 0 (\<lambda>i. X (Suc i))"
proof (unfold_locales, intro borel_measurableI)
  interpret predictable_process M F 0 X by (rule assms)
  fix S :: "'b set" and i assume open_S: "open S"
  have "{Suc i} = {i<..Suc i}" by fastforce
  hence "{Suc i} \<times> space M \<in> \<Sigma>\<^sub>P" using space_F[symmetric, of i] unfolding sets_predictable_sigma by (intro sigma_sets.Basic) blast
  moreover have "case_prod X -` S \<inter> (UNIV \<times> space M) \<in> \<Sigma>\<^sub>P" unfolding atLeast_0[symmetric] using open_S by (intro predictableD, simp add: borel_open)
  ultimately have "case_prod X -` S \<inter> ({Suc i} \<times> space M) \<in> \<Sigma>\<^sub>P" unfolding sets_predictable_sigma using space_F sets.sets_into_space
    by (subst Times_Int_distrib1[of "{Suc i}" UNIV "space M", simplified], subst inf.commute, subst Int_assoc[symmetric], subst Int_range_binary) 
       (intro sigma_sets_Inter binary_in_sigma_sets, fast)+
  moreover have "case_prod X -` S \<inter> ({Suc i} \<times> space M) = {Suc i} \<times> (X (Suc i) -` S \<inter> space M)" by (auto simp add: le_Suc_eq)
  moreover have "... = (\<Union>j. {j} \<times> (if j = Suc i then (X (Suc i) -` S \<inter> space M) else {}))" by (force split: if_splits)
  ultimately have "(\<Union>j. {j} \<times> (if j = Suc i then (X (Suc i) -` S \<inter> space M) else {})) \<in> \<Sigma>\<^sub>P" by argo
  thus "X (Suc i) -` S \<inter> space (F i) \<in> sets (F i)" using sets_in_filtration[of "\<lambda>j. if j = Suc i then (X (Suc i) -` S \<inter> space M) else {}"] space_F[OF zero_le] by presburger
qed

text \<open>The following lemma characterizes predictability in the discrete setting.\<close>

theorem (in nat_filtered_measure) predictable_process_iff: "predictable_process M F 0 X \<longleftrightarrow> adapted_process M F 0 (\<lambda>i. X (Suc i)) \<and> X 0 \<in> borel_measurable (F 0)"
proof (intro iffI)
  assume asm: "adapted_process M F 0 (\<lambda>i. X (Suc i)) \<and> X 0 \<in> borel_measurable (F 0)"
  interpret adapted_process M F 0 "\<lambda>i. X (Suc i)" using asm by blast
  have "(\<lambda>(x, y). X x y) \<in> borel_measurable \<Sigma>\<^sub>P"
  proof (intro borel_measurableI)
    fix S :: "'b set" assume open_S: "open S"
    have "{i} \<times> (X i -` S \<inter> space M) \<in> sets \<Sigma>\<^sub>P" for i 
    proof (cases i)
      case 0
      then show ?thesis unfolding sets_predictable_sigma 
        using measurable_sets[OF _ borel_open[OF open_S], of "X 0" "F 0"] asm by auto
    next
      case (Suc i)
      have "{Suc i} = {i<..Suc i}" by fastforce
      then show ?thesis unfolding sets_predictable_sigma 
        using measurable_sets[OF adapted borel_open[OF open_S], of i] 
        by (intro sigma_sets.Basic, auto simp add: Suc)
    qed
    moreover have "(\<lambda>(x, y). X x y) -` S \<inter> space \<Sigma>\<^sub>P = (\<Union>i. {i} \<times> (X i -` S \<inter> space M))" by fastforce
    ultimately show "(\<lambda>(x, y). X x y) -` S \<inter> space \<Sigma>\<^sub>P \<in> sets \<Sigma>\<^sub>P" by simp
  qed
  thus "predictable_process M F 0 X" by (unfold_locales)
next
  assume asm: "predictable_process M F 0 X"
  interpret predictable_process M F 0 X using asm by blast
  show "adapted_process M F 0 (\<lambda>i. X (Suc i)) \<and> X 0 \<in> borel_measurable (F 0)" using predictable_implies_adapted_Suc asm by auto
qed

corollary (in nat_filtered_measure) predictable_processI[intro!]:
  assumes "X 0 \<in> borel_measurable (F 0)" "\<And>i. X (Suc i) \<in> borel_measurable (F i)"
  shows "predictable_process M F 0 X"
  unfolding predictable_process_iff
  using assms
  by (meson adapted_process.intro adapted_process_axioms_def filtered_measure_axioms)
                                                                            
end