theory MLWE

imports Lemmas_for_spmf
        "Game_Based_Crypto.CryptHOL_Tutorial"

begin

section \<open>Module Learning-with-Errors Problem (module-LWE)\<close>

text \<open>\<open>Berlekamp_Zassenhaus\<close> loads the vector type \<open>'a vec\<close> from \<open>Jordan_Normal_Form.Matrix\<close>. 
This doubles the symbols \<open>\$\<close> and \<open>\<chi>\<close> for \<open>vec_nth\<close> and \<open>vec_lambda\<close>. Thus we delete the
\<open>vec_index\<close> for type \<open>'a vec\<close>. Still some type ambiguities remain.\<close>

unbundle %invisible lifting_syntax
no_adhoc_overloading %invisible Monad_Syntax.bind \<rightleftharpoons> bind_pmf
declare %invisible [[names_short]]


text \<open>Here the actual theory starts.\<close>

text \<open>We introduce a locale \<open>module_lwe\<close> that represents the module-Learning-with-Errors 
(module-LWE) problem in the setting of Kyber. 
The locale takes as input:
\begin{itemize}
\item \<open>type_a\<close> the type of the quotient ring of Kyber. (This is a side effect of the Harrison 
    trick in the Kyber locale.)
\item \<open>type_k\<close> the finite type for indexing vectors in Kyber. The cardinality is exactely $k$.
   (This is a side effect of the Harrison trick in the Kyber locale.)
\item \<open>idx\<close> an indexing function from \<open>'k\<close> to \<open>{0..<k}\<close>
\item \<open>eta\<close> the specification value for the centered binomial distribution $\beta_\eta$
\end{itemize}\<close>

locale module_lwe = 
fixes type_a :: "('a :: qr_spec) itself" 
  and type_k :: "('k ::finite) itself"
  and k :: nat
  and idx :: "'k::finite \<Rightarrow> nat"
  and eta :: nat
assumes "k = CARD('k)"
  and bij_idx: "bij_betw idx (UNIV::'k set) {0..<k}"
 
begin

text \<open>The adversary in the module-LWE takes a matrix \<open>A::(('b, 'n) vec, 'm) vec\<close> and a vector
  \<open>t::('b, 'm) vec\<close> and returns a probability distribution on \<open>bool\<close> 
  guessing whether the given input was randomly generated or a valid module-LWE instance.\<close>
type_synonym ('b, 'n, 'm) adversary = 
  "(('b, 'n) vec, 'm) vec \<Rightarrow> ('b, 'm) vec \<Rightarrow>  bool spmf"

text \<open>Next, we want to define the centered binomial distributions $\beta_\eta$.
\<open>bit_set\<close> returns the set of all bit lists of length \<open>eta\<close>.
\<open>beta\<close> is the centered binomial distribution $\beta_\eta$ as a \<open>pmf\<close> on the quotient ring $R_q$.
\<open>beta_vec\<close> is then centered binomial distribution $\beta_\eta ^k$ on vectors in $R_q^k$.\<close>

definition bit_set :: "int list set" where
"bit_set = {xs:: int list. set xs \<subseteq> {0,1} \<and> length xs = eta}"

lemma finite_bit_set:
"finite bit_set"
unfolding bit_set_def 
by (simp add: finite_lists_length_eq)

lemma bit_set_nonempty:
"bit_set \<noteq> {}"
proof -
  have "replicate eta 0 \<in> bit_set" unfolding bit_set_def by auto
  then show ?thesis by auto
qed

adhoc_overloading %invisible Monad_Syntax.bind \<rightleftharpoons> bind_pmf 
definition beta :: "'a qr pmf" where
"beta = do {
    as \<leftarrow> pmf_of_set (bit_set);
    bs \<leftarrow> pmf_of_set (bit_set);
    return_pmf (to_module (\<Sum>i<eta. as ! i - bs! i))
  } "

definition beta_vec :: "('a qr , 'k) vec pmf" where
"beta_vec = do {
    (xs :: 'a qr list) \<leftarrow> replicate_pmf (k) (beta);
    return_pmf (\<chi> i.  xs ! (idx i))
  }"

text \<open>Since we work over \<open>spmf\<close>, we need to show that \<open>beta_vec\<close> is lossless.\<close>

lemma lossless_beta_vec[simp]:
  "lossless_spmf (spmf_of_pmf beta_vec)"
by (simp)

text \<open>We define the game versions of module-LWE.
Given an adversary \<open>\<A>\<close>, we have two games: 
in \<open>game\<close>, the instance given to the adversary is a module-LWE instance, 
whereas in \<open>game_random\<close>, the instance is chosen randomly.\<close>

definition game :: "('a qr,'k,'k) adversary \<Rightarrow> bool spmf" where
  "game \<A>  = do {
    A \<leftarrow> spmf_of_set (UNIV:: (('a qr, 'k) vec, 'k) vec set);
    s \<leftarrow> beta_vec;
    e \<leftarrow> beta_vec;
    b' \<leftarrow> \<A> A (A *v s + e);
    return_spmf (b')
  }"

definition game_random :: "('a qr,'k,'k) adversary \<Rightarrow> bool spmf" where
  "game_random \<A>  = do {
    A \<leftarrow> spmf_of_set (UNIV:: (('a qr, 'k) vec, 'k) vec set);
    b \<leftarrow> spmf_of_set (UNIV:: ('a qr, 'k) vec set);
    b' \<leftarrow> \<A> A b;
    return_spmf (b')
  }"

text \<open>The advantage of an adversary \<open>\<A>\<close> returns a value how good the adversary is at guessing 
whether the instance is generated by the module-LWE or uniformly at random.\<close>

definition advantage :: "('a qr,'k,'k) adversary \<Rightarrow> real" where 
"advantage \<A> = \<bar>spmf (game \<A>) True - spmf (game_random \<A>) True \<bar>"


text \<open>Since the reduction proof of Kyber uses the module-LWE problem for two different dimensions
(ie.\ $A\in R_q^{(k+1)\times k}$ and $A\in R_q^{k\times k}$), we need a second definition of
the index function, the centered binomial distribution, the game and random game, and the advantage.
Here the problem is that the dimension $k$ of the vectors is hard-coded in the type \<open>'k\<close>. 
That makes it hard to ``simply add'' another dimension. A trick how this can be formalised 
nevertheless is to use the option type on \<open>'k\<close> to encode a type with $k+1$ elements.
With the option type, we can embed a vector of dimension $k$ indexed by the type \<open>'k\<close> into
a vector of dimension $k+1$ by adding a value for the index \<open>None\<close> (an element \<open>a :: 'k\<close> 
is mapped to \<open>Some a\<close>).
Note also that the additional index appears only in one dimension of $A$, resulting in 
a non-quadratic matrix.\<close>

text \<open>Index function of the option type \<open>'k option\<close>.\<close>

fun idx' :: "'k option \<Rightarrow> nat" where
  "idx' None = 0" |
  "idx' (Some x) = idx x + 1"

lemma idx': "((x # xs) ! idx' i) = 
  ( if i = None then x else xs ! idx (the i))" 
  if "length xs = k" for xs and i::"'k option"
proof (cases i)
  case None
  then show ?thesis using nth_append_length[of xs x "[]"] that by (simp add: if_distrib)
next
  case (Some a)
  have "idx a < k" using bij_idx 
  by (meson UNIV_I atLeastLessThan_iff bij_betwE) 
  then show ?thesis using Some that by (simp add: if_distrib nth_append)
qed


lemma idx'_lambda:
  "(\<chi> i. (x # xs) ! idx' i) = 
   (\<chi> i. if i = None then x else xs ! idx (the i))" 
   if "length xs = k" for xs using idx'[OF that]
 by presburger

text \<open>Definition of the centered binomial distribution $\beta_\eta^{k+1}$ and lossless property.\<close>

definition beta_vec' :: "('a qr , 'k option) vec spmf" where
"beta_vec' = do {
    (xs :: 'a qr list) \<leftarrow> replicate_spmf (k+1) (beta);
    return_spmf (\<chi> i.  xs ! (idx' i))
  }"

lemma lossless_beta_vec'[simp]:
  "lossless_spmf beta_vec'"
unfolding beta_vec'_def beta_def
by (simp add: replicate_spmf_def)

text \<open>Some lemmas on replicate.\<close>

lemma replicate_pmf_same_length:
assumes "\<And> xs. length xs = m \<Longrightarrow> f xs = g xs"
shows "bind_pmf (replicate_pmf m p) f = bind_pmf (replicate_pmf m p) g"
by (metis (mono_tags, lifting) assms bind_pmf_cong mem_Collect_eq set_replicate_pmf)

lemma replicate_spmf_same_length:
assumes "\<And> xs. length xs = m \<Longrightarrow> f xs = g xs"
shows "(replicate_spmf m p \<bind> f) = (replicate_spmf m p \<bind> g)"
unfolding replicate_spmf_def bind_spmf_of_pmf 
by (simp add: replicate_pmf_same_length[OF assms])

text \<open>Lemma to split the \<open>replicate (k+1)\<close> function in \<open>beta_vec'\<close> into two parts: 
\<open>replicate k\<close> and a separate value. Note, that the \<open>xs\<close> in the \<open>do\<close> notation below are 
always of length $k$. \<close>

no_adhoc_overloading Monad_Syntax.bind \<rightleftharpoons> bind_pmf

lemma beta_vec':
  "beta_vec' = do {
    (xs :: 'a qr list) \<leftarrow> replicate_spmf (k) (beta);
    (x :: 'a qr) \<leftarrow> spmf_of_pmf beta;
    return_spmf (\<chi> i.  if i = None then x else xs ! (idx (the i)))
  }"                                                    
unfolding beta_vec'_def idx'_lambda[symmetric] replicate_spmf_Suc_cons 
  bind_spmf_assoc return_bind_spmf
by (subst replicate_spmf_same_length[where 
  f = "(\<lambda>y. spmf_of_pmf beta \<bind> (\<lambda>ya. return_spmf (vec_lambda (\<lambda>i. (ya # y) ! idx' i))))" and
  g = "(\<lambda>xs. spmf_of_pmf beta \<bind> (\<lambda>x. return_spmf (vec_lambda (\<lambda>i. if i = None then x else 
        xs ! idx (the i)))))"])
   (simp_all add: idx'_lambda)

adhoc_overloading Monad_Syntax.bind \<rightleftharpoons> bind_pmf

text \<open>Definition of the two games for the option type.\<close>

definition game' :: "('a qr,'k,'k option) adversary \<Rightarrow> bool spmf" where
  "game' \<A>  = do {
    A \<leftarrow> spmf_of_set (UNIV:: (('a qr, 'k) vec, 'k option) vec set);
    s \<leftarrow> beta_vec;
    e \<leftarrow> beta_vec';
    b' \<leftarrow> \<A> A (A *v s + e);
    return_spmf (b')
  }"

definition game_random' :: "('a qr,'k,'k option) adversary \<Rightarrow> bool spmf" where
  "game_random' \<A>  = do {
    A \<leftarrow> spmf_of_set (UNIV:: (('a qr, 'k) vec, 'k option) vec set);
    b \<leftarrow> spmf_of_set (UNIV:: ('a qr, 'k option) vec set);
    b' \<leftarrow> \<A> A b;
    return_spmf (b')
  }"

text \<open>Definition of the advantage for the option type.\<close>

definition advantage' :: "('a qr,'k,'k option) adversary \<Rightarrow> real" where 
"advantage' \<A> = \<bar>spmf (game' \<A>) True - spmf (game_random' \<A>) True \<bar>"

text \<open>Game and random game for finite type with one element only\<close>

definition beta1 :: "('a qr , 1) vec pmf" where
"beta1 = bind_pmf beta (\<lambda>x. return_pmf (\<chi> i. x))"

definition game1 :: "('a qr, 1, 1) adversary \<Rightarrow> bool spmf" where
  "game1 \<A>  = do {
    A \<leftarrow> spmf_of_set (UNIV:: (('a qr, 1) vec, 1) vec set);
    s \<leftarrow> spmf_of_pmf beta1;
    e \<leftarrow> spmf_of_pmf beta1;
    b' \<leftarrow> \<A> A (A *v s + e);
    return_spmf (b')
  }"

definition game_random1 :: "('a qr,1,1) adversary \<Rightarrow> bool spmf" where
  "game_random1 \<A>  = do {
    A \<leftarrow> spmf_of_set (UNIV:: (('a qr, 1) vec, 1) vec set);
    b \<leftarrow> spmf_of_set (UNIV:: ('a qr, 1) vec set);
    b' \<leftarrow> \<A> A b;
    return_spmf (b')
  }"

text \<open>The advantage of an adversary \<open>\<A>\<close> returns a value how good the adversary is at guessing 
whether the instance is generated by the module-LWE or uniformly at random.\<close>

definition advantage1 :: "('a qr,1,1) adversary \<Rightarrow> real" where 
"advantage1 \<A> = \<bar>spmf (game1 \<A>) True - spmf (game_random1 \<A>) True \<bar>"

end
end