(*******************************************************************************

  Project: Refining Authenticated Key Agreement with Strong Adversaries

  Module:  pfslvl1.thy (Isabelle/HOL 2016-1)
  ID:      $Id: pfslvl1.thy 133183 2017-01-31 13:55:43Z csprenge $
  Author:  Joseph Lallemand, INRIA Nancy <joseph.lallemand@loria.fr>
           Christoph Sprenger, ETH Zurich <sprenger@inf.ethz.ch>
  
  Level-1 protocol using ephemeral asymmetric keys to achieve forward secrecy.

  Copyright (c) 2015-2016 Joseph Lallemand and Christoph Sprenger
  Licence: LGPL

*******************************************************************************)

section \<open>Key Transport Protocol with PFS (L1)\<close>

theory pfslvl1
imports Runs Secrecy AuthenticationI Payloads
begin


declare option.split_asm [split]
declare domIff [simp, iff del] 

(**************************************************************************************************)
subsection \<open>State and Events\<close>
(**************************************************************************************************)

consts
  sk :: "nat"
  kE :: "nat"  
  Nend :: "nat"

text \<open>Proofs break if @{term "1"} is used, because @{method "simp"} replaces it with
@{term "Suc 0"}\dots\<close>
abbreviation
  "xpkE \<equiv> Var 0"

abbreviation
  "xskE \<equiv> Var 2"

abbreviation
  "xsk \<equiv> Var 3"

abbreviation
  "xEnd \<equiv> Var 4"


abbreviation 
  "End \<equiv> Number Nend"


text \<open>domain of each role (protocol dependent)\<close>
fun domain :: "role_t \<Rightarrow> var set" where
  "domain Init = {xpkE, xskE, xsk}"
| "domain Resp = {xpkE, xsk}"



consts
  test :: rid_t
  
consts
  guessed_runs :: "rid_t \<Rightarrow> run_t"
  guessed_frame :: "rid_t \<Rightarrow> frame"

text \<open>specification of the guessed frame\<close>
text \<open>\begin{enumerate}
\item Domain
\item Well-typedness.
  The messages in the frame of a run never contain implementation material
  even if the agents of the run are dishonest.
  Therefore we consider only well-typed frames.
  This is notably required for the session key compromise; it also helps proving
  the partitionning of ik,
  since we know that the messages added by the protocol do not contain ltkeys in their
  payload and are therefore valid implementations.
\item We also ensure that the values generated by the frame owner are correctly guessed.
\end{enumerate}\<close>
specification (guessed_frame) 
  guessed_frame_dom_spec [simp]:
    "dom (guessed_frame R) = domain (role (guessed_runs R))"
  guessed_frame_payload_spec [simp, elim]:
    "guessed_frame R x = Some y \<Longrightarrow> y \<in> payload"
  guessed_frame_Init_xpkE [simp]: 
    "role (guessed_runs R) = Init \<Longrightarrow> guessed_frame R xpkE = Some (epubKF (R$kE))"
  guessed_frame_Init_xskE [simp]: 
    "role (guessed_runs R) = Init \<Longrightarrow> guessed_frame R xskE = Some (epriKF (R$kE))"
  guessed_frame_Resp_xsk [simp]: 
    "role (guessed_runs R) = Resp \<Longrightarrow> guessed_frame R xsk = Some (NonceF (R$sk))"
apply (rule exI [of _ 
    "\<lambda>R.
      if role (guessed_runs R) = Init 
      then [xpkE \<mapsto> epubKF (R$kE), xskE \<mapsto> epriKF (R$kE), xsk \<mapsto> End]
      else [xpkE \<mapsto> End, xsk \<mapsto> NonceF (R$sk)]"],
  auto simp add: domIff intro: role_t.exhaust)  
done


abbreviation
  "test_owner \<equiv> owner (guessed_runs test)"

abbreviation
  "test_partner \<equiv> partner (guessed_runs test)"


text \<open>level 1 state\<close>
record l1_state = 
  s0_state +
  progress :: progress_t
  signals :: "signal \<Rightarrow> nat"


type_synonym l1_obs = "l1_state"


abbreviation
  run_ended :: "var set option \<Rightarrow> bool"
where
  "run_ended r \<equiv> in_progress r xsk"

lemma run_ended_not_None [elim]:
  "run_ended R \<Longrightarrow> R = None \<Longrightarrow> False"
by (fast dest: in_progress_Some)

text \<open>@{term "test_ended s"} $\longleftrightarrow$ the test run has ended in @{term "s"}\<close>
abbreviation
  test_ended :: "'a l1_state_scheme \<Rightarrow> bool"
where
  "test_ended s \<equiv> run_ended (progress s test)"

text \<open>a run can emit signals if it involves the same agents as the test run, and if the test run 
  has not ended yet\<close>
definition
  can_signal :: "'a l1_state_scheme \<Rightarrow> agent \<Rightarrow> agent \<Rightarrow> bool"
where
  "can_signal s A B \<equiv>
  ((A = test_owner \<and> B = test_partner) \<or> (B = test_owner \<and> A = test_partner)) \<and>
  \<not> test_ended s"


text \<open>events\<close>
definition
  l1_learn :: "msg \<Rightarrow> ('a l1_state_scheme * 'a l1_state_scheme) set"
where
  "l1_learn m \<equiv> {(s,s').
    \<comment> \<open>guard\<close>
    synth (analz (insert m (ik s))) \<inter> (secret s) = {}  \<and>
    \<comment> \<open>action\<close>
    s' = s \<lparr>ik := ik s \<union> {m}\<rparr>
  }"


text \<open>protocol events\<close>

text \<open>
\begin{itemize}
\item step 1: create @{term "Ra"}, @{term "A"} generates @{term "pkE"}, @{term "skE"}
\item step 2: create @{term "Rb"}, @{term "B"} reads @{term "pkE"} authentically,
  generates @{term "K"}, emits a running signal for @{term "A"}, @{term "B"},
  (@{term "pkE"}, @{term "K"})
\item step 3: @{term "A"} reads @{term "K"} and @{term "pkE"} authentically,
  emits a commit signal for @{term "A"}, @{term "B"}, (@{term "pkE"}, @{term "K"})
\end{itemize}
\<close>

definition
  l1_step1 :: "rid_t \<Rightarrow> agent \<Rightarrow> agent \<Rightarrow> ('a l1_state_scheme * 'a l1_state_scheme) set"
where
  "l1_step1 Ra A B \<equiv> {(s, s').
    \<comment> \<open>guards:\<close>
    Ra \<notin> dom (progress s) \<and>
    guessed_runs Ra = \<lparr>role=Init, owner=A, partner=B\<rparr> \<and>
    \<comment> \<open>actions:\<close>
    s' = s\<lparr>
      progress := (progress s)(Ra \<mapsto> {xpkE, xskE})
      \<rparr>
  }"


definition
  l1_step2 :: "rid_t \<Rightarrow> agent \<Rightarrow> agent \<Rightarrow> msg \<Rightarrow> ('a l1_state_scheme * 'a l1_state_scheme) set"
where
  "l1_step2 Rb A B KE \<equiv> {(s, s').
    \<comment> \<open>guards:\<close>
    guessed_runs Rb = \<lparr>role=Resp, owner=B, partner=A\<rparr> \<and>
    Rb \<notin> dom (progress s) \<and>
    guessed_frame Rb xpkE = Some KE \<and>
    (can_signal s A B \<longrightarrow> \<comment> \<open>authentication guard\<close>
      (\<exists> Ra. guessed_runs Ra = \<lparr>role=Init, owner=A, partner=B\<rparr> \<and>
             in_progress (progress s Ra) xpkE \<and> guessed_frame Ra xpkE = Some KE)) \<and>
    (Rb = test \<longrightarrow> NonceF (Rb$sk) \<notin> synth (analz (ik s))) \<and>
    \<comment> \<open>actions:\<close>
    s' = s\<lparr> progress := (progress s)(Rb \<mapsto> {xpkE, xsk}),
            secret := {x. x = NonceF (Rb$sk) \<and> Rb = test} \<union> secret s,
            signals := if can_signal s A B then
                          addSignal (signals s) (Running A B (\<langle> KE, NonceF (Rb$sk) \<rangle>))
                       else
                          signals s
          \<rparr>
  }"

definition
  l1_step3 :: "rid_t \<Rightarrow> agent \<Rightarrow> agent \<Rightarrow> msg \<Rightarrow> ('a l1_state_scheme * 'a l1_state_scheme) set"
where
  "l1_step3 Ra A B K \<equiv> {(s, s').
    \<comment> \<open>guards:\<close>
    guessed_runs Ra = \<lparr>role=Init, owner=A, partner=B\<rparr> \<and>
    progress s Ra = Some {xpkE, xskE} \<and>
    guessed_frame Ra xsk = Some K \<and>
    (can_signal s A B \<longrightarrow> \<comment> \<open>authentication guard\<close>
      (\<exists> Rb. guessed_runs Rb = \<lparr>role=Resp, owner=B, partner=A\<rparr> \<and>
             progress s Rb = Some {xpkE, xsk} \<and>
             guessed_frame Rb xpkE = Some (epubKF (Ra$kE)) \<and>
             guessed_frame Rb xsk = Some K)) \<and>
    (Ra = test \<longrightarrow> K \<notin> synth (analz (ik s))) \<and>

    \<comment> \<open>actions:\<close>
    s' = s\<lparr> progress := (progress s)(Ra \<mapsto> {xpkE, xskE, xsk}),
            secret := {x. x = K \<and> Ra = test} \<union> secret s,
            signals := if can_signal s A B then
                         addSignal (signals s) (Commit A B \<langle>epubKF (Ra$kE), K\<rangle>)
                       else
                         signals s
          \<rparr>
  }"




text \<open>specification\<close>

definition 
  l1_init :: "l1_state set"
where
  "l1_init \<equiv> { \<lparr>
    ik = {},
    secret = {},
    progress = Map.empty,
    signals = \<lambda>x. 0
    \<rparr> }"

definition 
  l1_trans :: "('a l1_state_scheme * 'a l1_state_scheme) set" where
  "l1_trans \<equiv> (\<Union>m Ra Rb A B K KE.
     l1_step1 Ra A B \<union>
     l1_step2 Rb A B KE \<union>
     l1_step3 Ra A B K \<union>
     l1_learn m \<union>
     Id
  )"

definition 
  l1 :: "(l1_state, l1_obs) spec" where
  "l1 \<equiv> \<lparr>
    init = l1_init,
    trans = l1_trans,
    obs = id
  \<rparr>"

lemmas l1_defs = 
  l1_def l1_init_def l1_trans_def
  l1_learn_def
  l1_step1_def l1_step2_def l1_step3_def

lemmas l1_nostep_defs =
  l1_def l1_init_def l1_trans_def

lemma l1_obs_id [simp]: "obs l1 = id"
by (simp add: l1_def)


declare domIff [iff]

lemma run_ended_trans:
  "run_ended (progress s R) \<Longrightarrow>
   (s, s') \<in> trans l1 \<Longrightarrow>
   run_ended (progress s' R)"
apply (auto simp add: l1_nostep_defs)
apply (simp add: l1_defs ik_dy_def, fast ?)+
done

declare domIff [iff del]

lemma can_signal_trans:
  "can_signal s' A B \<Longrightarrow>
  (s, s') \<in> trans l1 \<Longrightarrow>
  can_signal s A B"
by (auto simp add: can_signal_def run_ended_trans)


(**************************************************************************************************)
subsection \<open>Refinement: secrecy\<close>
(**************************************************************************************************)

text \<open>mediator function\<close>
definition 
  med01s :: "l1_obs \<Rightarrow> s0_obs"
where
  "med01s t \<equiv> \<lparr> ik = ik t, secret = secret t \<rparr>"


text \<open>relation between states\<close>
definition
  R01s :: "(s0_state * l1_state) set"
where
  "R01s \<equiv> {(s,s').
    s = \<lparr>ik = ik s', secret = secret s'\<rparr>
    }"


text \<open>protocol independent events\<close>

lemma l1_learn_refines_learn:
  "{R01s} s0_learn m, l1_learn m {>R01s}"
apply (simp add: PO_rhoare_defs R01s_def)
apply auto
apply (simp add: l1_defs s0_defs s0_secrecy_def)
done



text \<open>protocol events\<close>
lemma l1_step1_refines_skip:
  "{R01s} Id, l1_step1 Ra A B {>R01s}"
by (auto simp add: PO_rhoare_defs R01s_def l1_step1_def)

lemma l1_step2_refines_add_secret_skip:
  "{R01s} s0_add_secret (NonceF (Rb$sk)) \<union> Id, l1_step2 Rb A B KE {>R01s}"
apply (auto simp add: PO_rhoare_defs R01s_def s0_add_secret_def)
apply (auto simp add: l1_step2_def)
done

lemma l1_step3_refines_add_secret_skip:
  "{R01s} s0_add_secret K \<union> Id, l1_step3 Ra A B K {>R01s}"
apply (auto simp add: PO_rhoare_defs R01s_def s0_add_secret_def)
apply (auto simp add: l1_step3_def)
done


text \<open>refinement proof\<close>
lemmas l1_trans_refines_s0_trans = 
  l1_learn_refines_learn
  l1_step1_refines_skip l1_step2_refines_add_secret_skip l1_step3_refines_add_secret_skip 

lemma l1_refines_init_s0 [iff]:
  "init l1 \<subseteq> R01s `` (init s0)"
by (auto simp add: R01s_def s0_defs l1_defs s0_secrecy_def)


lemma l1_refines_trans_s0 [iff]:
  "{R01s} trans s0, trans l1 {> R01s}"
by (auto simp add: s0_def l1_def s0_trans_def l1_trans_def 
         intro: l1_trans_refines_s0_trans)


lemma obs_consistent_med01x [iff]: 
  "obs_consistent R01s med01s s0 l1"
by (auto simp add: obs_consistent_def R01s_def med01s_def)



text \<open>refinement result\<close>
lemma l1s_refines_s0 [iff]: 
  "refines 
     R01s
     med01s s0 l1"
by (auto simp add:refines_def PO_refines_def)

lemma  l1_implements_s0 [iff]: "implements med01s s0 l1"
by (rule refinement_soundness) (fast)


(**************************************************************************************************)
subsection \<open>Derived invariants: secrecy\<close>
(**************************************************************************************************)

abbreviation "l1_secrecy \<equiv> s0_secrecy"


lemma l1_obs_secrecy [iff]: "oreach l1 \<subseteq> l1_secrecy"
apply (rule external_invariant_translation 
         [OF s0_obs_secrecy _ l1_implements_s0])
apply (auto simp add: med01s_def s0_secrecy_def)
done

lemma l1_secrecy [iff]: "reach l1 \<subseteq> l1_secrecy"
by (rule external_to_internal_invariant [OF l1_obs_secrecy], auto)


(**************************************************************************************************)
subsection \<open>Invariants\<close>
(**************************************************************************************************)

subsubsection \<open>inv1\<close>
(**************************************************************************************************)
text \<open>if a commit signal for a nonce has been emitted,
 then there is a finished initiator run with this nonce.
\<close>



definition
  l1_inv1 :: "l1_state set"
where
  "l1_inv1 \<equiv> {s. \<forall> Ra A B K.
    signals s (Commit A B \<langle>epubKF (Ra$kE), K\<rangle>) > 0 \<longrightarrow>
      guessed_runs Ra = \<lparr>role=Init, owner=A, partner=B\<rparr> \<and>
      progress s Ra = Some {xpkE, xskE, xsk} \<and>
      guessed_frame Ra xsk = Some K
   }"
  
lemmas l1_inv1I = l1_inv1_def [THEN setc_def_to_intro, rule_format]
lemmas l1_inv1E [elim] = l1_inv1_def [THEN setc_def_to_elim, rule_format]
lemmas l1_inv1D = l1_inv1_def [THEN setc_def_to_dest, rule_format, rotated 1, simplified]


lemma l1_inv1_init [iff]:
  "init l1 \<subseteq> l1_inv1"
by (auto simp add: l1_def l1_init_def l1_inv1_def)

declare domIff [iff]

lemma l1_inv1_trans [iff]:
  "{l1_inv1} trans l1 {> l1_inv1}"
apply (auto simp add: PO_hoare_defs l1_nostep_defs intro!: l1_inv1I)
apply (auto simp add: l1_defs ik_dy_def l1_inv1_def)
done

lemma PO_l1_inv1 [iff]: "reach l1 \<subseteq> l1_inv1"
by (rule inv_rule_basic) (auto)


subsubsection \<open>inv2\<close>
(**************************************************************************************************)
text \<open>if a responder run knows a nonce, then a running signal for this nonce has been emitted\<close>

definition
  l1_inv2 :: "l1_state set"
where
  "l1_inv2 \<equiv> {s. \<forall> KE A B Rb.
    guessed_runs Rb = \<lparr>role=Resp, owner=B, partner=A\<rparr> \<longrightarrow>
    progress s Rb = Some {xpkE, xsk} \<longrightarrow>
    guessed_frame Rb xpkE = Some KE \<longrightarrow>
    can_signal s A B \<longrightarrow>
      signals s (Running A B \<langle>KE, NonceF (Rb$sk)\<rangle>) > 0
  }"

lemmas l1_inv2I = l1_inv2_def [THEN setc_def_to_intro, rule_format]
lemmas l1_inv2E [elim] = l1_inv2_def [THEN setc_def_to_elim, rule_format]
lemmas l1_inv2D = l1_inv2_def [THEN setc_def_to_dest, rule_format, rotated 1, simplified]


lemma l1_inv2_init [iff]:
  "init l1 \<subseteq> l1_inv2"
by (auto simp add: l1_def l1_init_def l1_inv2_def)

lemma l1_inv2_trans [iff]:
  "{l1_inv2} trans l1 {> l1_inv2}"
apply (auto simp add: PO_hoare_defs intro!: l1_inv2I)
apply (drule can_signal_trans, assumption)
apply (auto simp add: l1_nostep_defs)
apply (auto simp add: l1_defs ik_dy_def l1_inv2_def)
done

lemma PO_l1_inv2 [iff]: "reach l1 \<subseteq> l1_inv2"
by (rule inv_rule_basic) (auto)


subsubsection \<open>inv3 (derived)\<close>
(**************************************************************************************************)
text \<open>if an unfinished initiator run and a finished responder run both know the same nonce,
 then the number of running signals for this nonce is strictly greater than the number of commit
 signals.
 (actually, there are 0 commit and 1 running)
\<close>

definition
  l1_inv3 :: "l1_state set"
where
  "l1_inv3 \<equiv> {s. \<forall> A B Rb Ra.
    guessed_runs Rb = \<lparr>role=Resp, owner=B, partner=A\<rparr> \<longrightarrow>
    progress s Rb = Some {xpkE, xsk} \<longrightarrow>
    guessed_frame Rb xpkE = Some (epubKF (Ra$kE)) \<longrightarrow>
    guessed_runs Ra = \<lparr>role=Init, owner=A, partner=B\<rparr> \<longrightarrow>
    progress s Ra = Some {xpkE, xskE} \<longrightarrow>
    can_signal s A B \<longrightarrow>
      signals s (Commit A B (\<langle>epubKF (Ra$kE), NonceF (Rb$sk)\<rangle>)) 
    < signals s (Running A B (\<langle>epubKF (Ra$kE), NonceF (Rb$sk)\<rangle>)) 
  }"

lemmas l1_inv3I = l1_inv3_def [THEN setc_def_to_intro, rule_format]
lemmas l1_inv3E [elim] = l1_inv3_def [THEN setc_def_to_elim, rule_format]
lemmas l1_inv3D = l1_inv3_def [THEN setc_def_to_dest, rule_format, rotated 1, simplified]

lemma l1_inv3_derived: "l1_inv1 \<inter> l1_inv2 \<subseteq> l1_inv3"
apply (auto intro!: l1_inv3I)
apply (auto dest!: l1_inv2D)
apply (rename_tac x A B Rb Ra)
apply (case_tac "signals x (Commit A B \<langle>epubKF (Ra $ kE), NonceF (Rb $ sk)\<rangle>) > 0", auto)
apply (fastforce dest: l1_inv1D elim: equalityE)
done


(**************************************************************************************************)
subsection \<open>Refinement: injective agreement\<close>
(**************************************************************************************************)

text \<open>mediator function\<close>
definition 
  med01ia :: "l1_obs \<Rightarrow> a0i_obs"
where
  "med01ia t \<equiv> \<lparr>a0n_state.signals = signals t\<rparr>"


text \<open>relation between states\<close>
definition
  R01ia :: "(a0i_state * l1_state) set"
where
  "R01ia \<equiv> {(s,s').
    a0n_state.signals s = signals s'
    }"


text \<open>protocol independent events\<close>

lemma l1_learn_refines_a0_ia_skip:
  "{R01ia} Id, l1_learn m {>R01ia}"
apply (auto simp add: PO_rhoare_defs R01ia_def)
apply (simp add: l1_learn_def)
done



text \<open>protocol events\<close>
lemma l1_step1_refines_a0i_skip:
  "{R01ia} Id, l1_step1 Ra A B {>R01ia}"
by (auto simp add: PO_rhoare_defs R01ia_def l1_step1_def)


lemma l1_step2_refines_a0i_running_skip:
  "{R01ia} a0i_running A B \<langle>KE, NonceF (Rb$sk)\<rangle> \<union> Id, l1_step2 Rb A B KE {>R01ia}"
by (auto simp add: PO_rhoare_defs R01ia_def, simp_all add: l1_step2_def a0i_running_def, auto)

lemma l1_step3_refines_a0i_commit_skip:
  "{R01ia \<inter> (UNIV \<times> l1_inv3)} a0i_commit A B \<langle>epubKF (Ra$kE), K\<rangle> \<union> Id, l1_step3 Ra A B K {>R01ia}"
apply (auto simp add: PO_rhoare_defs R01ia_def)
apply (auto simp add: l1_step3_def a0i_commit_def)
apply (force elim!: l1_inv3E)+
done

text \<open>refinement proof\<close>
lemmas l1_trans_refines_a0i_trans = 
  l1_learn_refines_a0_ia_skip
  l1_step1_refines_a0i_skip l1_step2_refines_a0i_running_skip l1_step3_refines_a0i_commit_skip 

lemma l1_refines_init_a0i [iff]:
  "init l1 \<subseteq> R01ia `` (init a0i)"
by (auto simp add: R01ia_def a0i_defs l1_defs)


lemma l1_refines_trans_a0i [iff]:
  "{R01ia \<inter> (UNIV \<times> (l1_inv1 \<inter> l1_inv2))} trans a0i, trans l1 {> R01ia}"
proof -
  let ?pre' = "R01ia \<inter> (UNIV \<times> l1_inv3)"
  show ?thesis (is "{?pre} ?t1, ?t2 {>?post}")
  proof (rule relhoare_conseq_left)
    show "?pre \<subseteq> ?pre'"
      using l1_inv3_derived by blast
  next 
    show "{?pre'} ?t1, ?t2 {> ?post}"
      apply (auto simp add: a0i_def l1_def a0i_trans_def l1_trans_def)
      prefer 2 using l1_step2_refines_a0i_running_skip apply (simp add: PO_rhoare_defs, blast)
      prefer 2 using l1_step3_refines_a0i_commit_skip apply (simp add: PO_rhoare_defs, blast)
      apply (blast intro!:l1_trans_refines_a0i_trans)+
      done
  qed
qed


lemma obs_consistent_med01ia [iff]: 
  "obs_consistent R01ia med01ia a0i l1"
by (auto simp add: obs_consistent_def R01ia_def med01ia_def)



text \<open>refinement result\<close>
lemma l1_refines_a0i [iff]: 
  "refines 
     (R01ia \<inter> (reach a0i \<times> (l1_inv1 \<inter> l1_inv2)))
     med01ia a0i l1"
by (rule Refinement_using_invariants, auto)

lemma  l1_implements_a0i [iff]: "implements med01ia a0i l1"
by (rule refinement_soundness) (fast)


(**************************************************************************************************)
subsection \<open>Derived invariants: injective agreement\<close>
(**************************************************************************************************)

definition 
  l1_iagreement :: "('a l1_state_scheme) set"
where
  "l1_iagreement \<equiv> {s. \<forall> A B N. signals s (Commit A B N) \<le> signals s (Running A B N)}"

lemmas l1_iagreementI = l1_iagreement_def [THEN setc_def_to_intro, rule_format]
lemmas l1_iagreementE [elim] = l1_iagreement_def [THEN setc_def_to_elim, rule_format]


lemma l1_obs_iagreement [iff]: "oreach l1 \<subseteq> l1_iagreement"
apply (rule external_invariant_translation 
         [OF PO_a0i_obs_agreement _ l1_implements_a0i])
apply (auto simp add: med01ia_def l1_iagreement_def a0i_agreement_def)
done

lemma l1_iagreement [iff]: "reach l1 \<subseteq> l1_iagreement"
by (rule external_to_internal_invariant [OF l1_obs_iagreement], auto)



end
