section \<open>Implementation\<close>

theory HMM_Implementation
  imports
    Hidden_Markov_Model
    "Monad_Memo_DP.State_Main"
begin

subsection \<open>The Forward Algorithm\<close>

locale HMM4 = HMM3 _ _ _ \<O>\<^sub>s \<K> for \<O>\<^sub>s :: "'t set" and \<K> :: "'s \<Rightarrow> 's pmf" +
  assumes states_distinct: "distinct state_list"

context HMM3_defs
begin

context
  fixes os :: "'t iarray"
begin

text \<open>
  Alternative definition using indices into the list of states.
  The list of states is implemented as an immutable array for better performance.
\<close>

function forward_ix_rec where
  "forward_ix_rec s t_end n = (if n \<ge> IArray.length os then indicator {t_end} s else
    (\<Sum>t \<leftarrow> state_list.
      ennreal (pmf (\<O> t) (os !! n)) * ennreal (pmf (\<K> s) t) * forward_ix_rec t t_end (n + 1)))
  "
  by auto
termination
  by (relation "Wellfounded.measure (\<lambda>(_,_,n). IArray.length os - n)") auto

text \<open>Memoization\<close>

memoize_fun forward_ix\<^sub>m: forward_ix_rec
  with_memory dp_consistency_mapping
  monadifies (state) forward_ix_rec.simps[unfolded Let_def]
  term forward_ix\<^sub>m'
memoize_correct
  by memoize_prover

text \<open>The main theorems generated by memoization.\<close>
context
  includes state_monad_syntax
begin
thm forward_ix\<^sub>m'.simps forward_ix\<^sub>m_def
thm forward_ix\<^sub>m.memoized_correct
end

end (* Fixed IArray *)

definition
  "forward_ix os = forward_ix_rec (IArray os)"

definition
  "likelihood_compute s os \<equiv>
    if s \<in> set state_list then Some (\<Sum>t \<leftarrow> state_list. forward s t os) else None"

end (* HMM3 Defs *)

text \<open>Correctness of the alternative definition.\<close>

lemma (in HMM3) forward_ix_drop_one:
  "forward_ix (o # os) s t (n + 1) = forward_ix os s t n"
  by (induction "length os - n" arbitrary: s n; simp add: forward_ix_def)

lemma (in HMM4) forward_ix_forward:
  "forward_ix os s t 0 = forward s t os"
  unfolding forward_ix_def
proof (induction os arbitrary: s)
  case Nil
  then show ?case
    by simp
next
  case (Cons o os)
  show ?case
    using forward_ix_drop_one[unfolded forward_ix_def] states_distinct
    by (subst forward.simps, subst forward_ix_rec.simps)
       (simp add: Cons.IH state_list_\<S> sum_list_distinct_conv_sum_set
             del: forward_ix_rec.simps forward.simps
       )
qed

text \<open>
  Instructs the code generator to use this equation instead to execute \<open>forward\<close>.
  Uses the memoized version of \<open>forward_ix\<close>.
\<close>
lemma (in HMM4) forward_code [code]:
  "forward s t os = fst (run_state (forward_ix\<^sub>m' (IArray os) s t 0) Mapping.empty)"
  by (simp only:
      forward_ix_def forward_ix\<^sub>m.memoized_correct forward_ix_forward[symmetric]
      states_distinct
     )

theorem (in HMM4) likelihood_compute:
  "likelihood_compute s os = Some x \<longleftrightarrow> s \<in> \<S> \<and> x = likelihood s os"
  unfolding likelihood_compute_def
  by (auto simp: states_distinct state_list_\<S> sum_list_distinct_conv_sum_set likelihood_forward)


subsection \<open>The Viterbi Algorithm\<close>

context HMM3_defs
begin

context
  fixes os :: "'t iarray"
begin

text \<open>
  Alternative definition using indices into the list of states.
  The list of states is implemented as an immutable array for better performance.
\<close>

function viterbi_ix_rec where
  "viterbi_ix_rec s t_end n = (if n \<ge> IArray.length os then ([], indicator {t_end} s) else
  fst (
    argmax snd (map
      (\<lambda>t. let (xs, v) = viterbi_ix_rec t t_end (n + 1) in
        (t # xs, ennreal (pmf (\<O> t) (os !! n) * pmf (\<K> s) t) * v))
    state_list)))
  "
  by pat_completeness auto
termination
  by (relation "Wellfounded.measure (\<lambda>(_,_,n). IArray.length os - n)") auto

text \<open>Memoization\<close>

memoize_fun viterbi_ix\<^sub>m: viterbi_ix_rec
  with_memory dp_consistency_mapping
  monadifies (state) viterbi_ix_rec.simps[unfolded Let_def]

memoize_correct
  by memoize_prover

text \<open>The main theorems generated by memoization.\<close>
context
  includes state_monad_syntax
begin
thm viterbi_ix\<^sub>m'.simps viterbi_ix\<^sub>m_def
thm viterbi_ix\<^sub>m.memoized_correct
end

end (* Fixed IArray *)

definition
  "viterbi_ix os = viterbi_ix_rec (IArray os)"

end (* HMM3 Defs *)

context HMM3
begin

lemma viterbi_ix_drop_one:
  "viterbi_ix (o # os) s t (n + 1) = viterbi_ix os s t n"
  by (induction "length os - n" arbitrary: s n; simp add: viterbi_ix_def)

lemma viterbi_ix_viterbi:
  "viterbi_ix os s t 0 = viterbi s t os"
  unfolding viterbi_ix_def
proof (induction os arbitrary: s)
  case Nil
  then show ?case
    by simp
next
  case (Cons o os)
  show ?case
    using viterbi_ix_drop_one[unfolded viterbi_ix_def]
    by (subst viterbi.simps, subst viterbi_ix_rec.simps)
       (simp add: Cons.IH del: viterbi_ix_rec.simps viterbi.simps)
qed

lemma viterbi_code [code]:
  "viterbi s t os = fst (run_state (viterbi_ix\<^sub>m' (IArray os) s t 0) Mapping.empty)"
  by (simp only: viterbi_ix_def viterbi_ix\<^sub>m.memoized_correct viterbi_ix_viterbi[symmetric])

end (* Hidden Markov Model 3 *)

subsection \<open>Misc\<close>

lemma pmf_of_alist_support_aux_1:
  assumes "\<forall> (_, p) \<in> set \<mu>. p \<ge> 0"
  shows "(0 :: real) \<le> (case map_of \<mu> x of None \<Rightarrow> 0 | Some p \<Rightarrow> p)"
  using assms by (auto split: option.split dest: map_of_SomeD)

lemma pmf_of_alist_support_aux_2:
  assumes "\<forall> (_, p) \<in> set \<mu>. p \<ge> 0"
    and "sum_list (map snd \<mu>) = 1"
    and "distinct (map fst \<mu>)"
  shows "\<integral>\<^sup>+ x. ennreal (case map_of \<mu> x of None \<Rightarrow> 0 | Some p \<Rightarrow> p) \<partial>count_space UNIV = 1"
  using assms
  apply (subst nn_integral_count_space)
  subgoal
    by (rule finite_subset[where B = "fst ` set \<mu>"];
        force split: option.split_asm simp: image_iff dest: map_of_SomeD)
  apply (subst sum.mono_neutral_left[where T = "fst ` set \<mu>"])
     apply blast
  subgoal
    by (smt ennreal_less_zero_iff map_of_eq_None_iff mem_Collect_eq option.case(1) subsetI)
  subgoal
    by auto
  subgoal premises prems
  proof -
    have "(\<Sum>x = 0..<length \<mu>. snd (\<mu> ! x))
      = sum (\<lambda> x. case map_of \<mu> x of None \<Rightarrow> 0 | Some v \<Rightarrow> v) (fst ` set \<mu>)"
      apply (rule sym)
      apply (rule sum.reindex_cong[where l = "\<lambda> i. fst (\<mu> ! i)"])
        apply (auto split: option.split)
      subgoal
        using prems(3) by (intro inj_onI, auto simp: distinct_conv_nth)
      subgoal
        by (auto simp: in_set_conv_nth rev_image_eqI)
      subgoal
        by (simp add: map_of_eq_None_iff)
      subgoal
        using map_of_eq_Some_iff[OF prems(3)]
        by (metis fst_conv nth_mem option.inject prod_eqI snd_conv)
      done
    with prems(2) show ?thesis
      by (smt pmf_of_alist_support_aux_1[OF assms(1)] atLeastLessThan_iff ennreal_1
          length_map nth_map sum.cong sum_ennreal sum_list_sum_nth
          )
  qed
  done

lemma pmf_of_alist_support:
  assumes "\<forall> (_, p) \<in> set \<mu>. p \<ge> 0"
    and "sum_list (map snd \<mu>) = 1"
    and "distinct (map fst \<mu>)"
  shows "set_pmf (pmf_of_alist \<mu>) \<subseteq> fst ` set \<mu>"
  unfolding pmf_of_alist_def
  apply (subst set_embed_pmf)
  subgoal for x
    using assms(1) by (auto split: option.split dest: map_of_SomeD)
  subgoal
    using pmf_of_alist_support_aux_2[OF assms] .
  apply (force split: option.split_asm simp: image_iff dest: map_of_SomeD)+
  done

text \<open>Defining a Markov kernel from an association list.\<close>
locale Closed_Kernel_From =
  fixes K :: "('s \<times> ('t \<times> real) list) list"
    and S :: "'t list"
  assumes wellformed: "S \<noteq> []"
      and closed: "\<forall> (s, \<mu>) \<in> set K. \<forall> (t, _) \<in> set \<mu>. t \<in> set S"
      and is_pmf:
        "\<forall> (_, \<mu>) \<in> set K. \<forall> (_, p) \<in> set \<mu>. p \<ge> 0"
        "\<forall> (_, \<mu>) \<in> set K. distinct (map fst \<mu>)"
        "\<forall> (s, \<mu>) \<in> set K. sum_list (map snd \<mu>) = 1"
      and is_unique:
        "distinct (map fst K)"
begin

definition
  "K' s \<equiv> case map_of (map (\<lambda> (s, \<mu>). (s, PMF_Impl.pmf_of_alist \<mu>)) K) s of
  None \<Rightarrow> return_pmf (hd S) |
  Some s \<Rightarrow> s"

sublocale Closed_Kernel K' "set S"
  using wellformed closed is_pmf pmf_of_alist_support
  unfolding K'_def by - (standard; fastforce split: option.split_asm dest: map_of_SomeD)

definition [code]:
  "K1 = map_of (map (\<lambda> (s, \<mu>). (s, map_of \<mu>)) K)"

lemma pmf_of_alist_aux:
  assumes "(s, \<mu>) \<in> set K"
  shows
    "pmf (pmf_of_alist \<mu>) t = (case map_of \<mu> t of
      None \<Rightarrow> 0
    | Some p \<Rightarrow> p)"
  using assms is_pmf unfolding pmf_of_alist_def
  by (intro pmf_embed_pmf pmf_of_alist_support_aux_2) 
     (auto 4 3 split: option.split dest: map_of_SomeD)

lemma unique: "\<mu> = \<mu>'" if "(s, \<mu>) \<in> set K" "(s, \<mu>') \<in> set K"
  using that is_unique
  by (smt Pair_inject distinct_conv_nth fst_conv in_set_conv_nth length_map nth_map)

lemma (in -) map_of_NoneD:
  "x \<notin> fst ` set M" if "map_of M x = None"
  using that by (auto dest: weak_map_of_SomeI)

lemma K'_code [code_post]:
  "pmf (K' s) t = (case K1 s of
      None \<Rightarrow> (if t = hd S then 1 else 0)
    | Some \<mu> \<Rightarrow> case \<mu> t of
        None \<Rightarrow> 0
      | Some p \<Rightarrow> p
  )"
  unfolding K'_def K1_def
  apply (clarsimp split: option.split, safe)
                 apply (drule map_of_SomeD, drule map_of_NoneD, force)+
         apply (fastforce dest: unique map_of_SomeD simp: pmf_of_alist_aux)+
  done

end

subsection \<open>Executing Concrete HMMs\<close>

locale Concrete_HMM_defs =
  fixes \<K> :: "('s \<times> ('s \<times> real) list) list"
    and \<O> :: "('s \<times> ('t \<times> real) list) list"
    and \<O>\<^sub>s :: "'t list"
    and \<K>\<^sub>s :: "'s list"
begin

definition
  "\<K>' s \<equiv> case map_of (map (\<lambda> (s, \<mu>). (s, PMF_Impl.pmf_of_alist \<mu>)) \<K>) s of
    None \<Rightarrow> return_pmf (hd \<K>\<^sub>s) |
    Some s \<Rightarrow> s"

definition
  "\<O>' s \<equiv> case map_of (map (\<lambda> (s, \<mu>). (s, PMF_Impl.pmf_of_alist \<mu>)) \<O>) s of
    None \<Rightarrow> return_pmf (hd \<O>\<^sub>s) |
    Some s \<Rightarrow> s"

end

locale Concrete_HMM = Concrete_HMM_defs +
  assumes observations_wellformed': "\<O>\<^sub>s \<noteq> []"
      and observations_closed': "\<forall> (s, \<mu>) \<in> set \<O>. \<forall> (t, _) \<in> set \<mu>. t \<in> set \<O>\<^sub>s"
      and observations_form_pmf':
        "\<forall> (_, \<mu>) \<in> set \<O>. \<forall> (_, p) \<in> set \<mu>. p \<ge> 0"
        "\<forall> (_, \<mu>) \<in> set \<O>. distinct (map fst \<mu>)"
        "\<forall> (s, \<mu>) \<in> set \<O>. sum_list (map snd \<mu>) = 1"
      and observations_unique:
        "distinct (map fst \<O>)"
  assumes states_wellformed: "\<K>\<^sub>s \<noteq> []"
      and states_closed: "\<forall> (s, \<mu>) \<in> set \<K>. \<forall> (t, _) \<in> set \<mu>. t \<in> set \<K>\<^sub>s"
      and states_form_pmf:
        "\<forall> (_, \<mu>) \<in> set \<K>. \<forall> (_, p) \<in> set \<mu>. p \<ge> 0"
        "\<forall> (_, \<mu>) \<in> set \<K>. distinct (map fst \<mu>)"
        "\<forall> (s, \<mu>) \<in> set \<K>. sum_list (map snd \<mu>) = 1"
      and states_unique:
        "distinct (map fst \<K>)" "distinct \<K>\<^sub>s"
begin

interpretation O: Closed_Kernel_From \<O> \<O>\<^sub>s
  rewrites "O.K' = \<O>'"
proof -
  show \<open>Closed_Kernel_From \<O> \<O>\<^sub>s\<close>
    using observations_wellformed' observations_closed' observations_form_pmf' observations_unique
    by unfold_locales auto
  show \<open>Closed_Kernel_From.K' \<O> \<O>\<^sub>s = \<O>'\<close>
    unfolding Closed_Kernel_From.K'_def[OF \<open>Closed_Kernel_From \<O> \<O>\<^sub>s\<close>] \<O>'_def
    by auto
qed

interpretation K: Closed_Kernel_From \<K> \<K>\<^sub>s
  rewrites "K.K' = \<K>'"
proof -
  show \<open>Closed_Kernel_From \<K> \<K>\<^sub>s\<close>
    using states_wellformed states_closed states_form_pmf states_unique by unfold_locales auto
  show \<open>Closed_Kernel_From.K' \<K> \<K>\<^sub>s = \<K>'\<close>
    unfolding Closed_Kernel_From.K'_def[OF \<open>Closed_Kernel_From \<K> \<K>\<^sub>s\<close>] \<K>'_def
    by auto
qed

lemmas O_code = O.K'_code O.K1_def
lemmas K_code = K.K'_code K.K1_def

sublocale HMM_interp: HMM4 \<O>' "set \<K>\<^sub>s" \<K>\<^sub>s "set \<O>\<^sub>s" \<K>'
  using O.Closed_Kernel_axioms K.Closed_Kernel_axioms states_unique(2)
  by (intro_locales; intro HMM4_axioms.intro HMM3_axioms.intro HOL.refl)

end (* Concrete HMM *)

end