(*  Title:      CoW/Submonoids.thy
    Author:     Štěpán Holub, Charles University
    Author:     Martin Raška, Charles University
    Author:     Štěpán Starosta, CTU in Prague

Part of Combinatorics on Words Formalized. See https://gitlab.com/formalcow/combinatorics-on-words-formalized/
*)


theory Submonoids
  imports CoWBasic
begin

chapter \<open>Submonoids of a free monoid\<close>

text\<open>This chapter deals with properties of submonoids of a free monoid, that is, with monoids of words.
See more in Chapter 1 of @{cite Lo83}.
\<close>

section \<open>Hull\<close>

text\<open>First, we define the hull of a set of words, that is, the monoid generated by them.\<close>

inductive_set hull :: "'a list set \<Rightarrow> 'a list set" (\<open>\<langle>_\<rangle>\<close>)
  for G where
    emp_in[simp]:  "\<epsilon> \<in> \<langle>G\<rangle>" |
    prod_cl:  "w1 \<in> G \<Longrightarrow> w2 \<in> \<langle>G\<rangle> \<Longrightarrow> w1 \<cdot> w2 \<in> \<langle>G\<rangle>"

lemmas [intro] = hull.intros

lemma hull_closed[intro]: "w1 \<in> \<langle>G\<rangle> \<Longrightarrow> w2 \<in> \<langle>G\<rangle> \<Longrightarrow> w1 \<cdot> w2 \<in> \<langle>G\<rangle>"
  by (rule hull.induct[of w1 G "\<lambda> x. (x\<cdot>w2)\<in>\<langle>G\<rangle>"]) auto+

lemma gen_in [intro]: "w \<in> G \<Longrightarrow> w \<in> \<langle>G\<rangle>"
  using hull.prod_cl by force

lemma hull_induct: assumes "x \<in> \<langle>G\<rangle>" "P \<epsilon>" "\<And>w. w \<in> G \<Longrightarrow> P w"
  "\<And>w1 w2. w1 \<in> \<langle>G\<rangle> \<Longrightarrow> P w1 \<Longrightarrow> w2 \<in> \<langle>G\<rangle> \<Longrightarrow> P w2 \<Longrightarrow> P (w1 \<cdot> w2)" shows  "P x"
  using hull.induct[of _ _ P, OF \<open>x \<in> \<langle>G\<rangle>\<close>  \<open>P \<epsilon>\<close>]
    assms by (simp add: gen_in)

lemma genset_sub[simp]: "G \<subseteq> \<langle>G\<rangle>"
  using gen_in ..

lemma genset_sub_lists: "ws \<in> lists G \<Longrightarrow> ws \<in> lists \<langle>G\<rangle>"
  using sub_lists_mono[OF genset_sub].

lemma in_lists_conv_set_subset: "set ws \<subseteq> G \<longleftrightarrow> ws \<in> lists G"
  by blast

lemma concat_in_hull [intro]:
  assumes "set ws \<subseteq> G"
  shows   "concat ws \<in> \<langle>G\<rangle>"
  using assms by (induction ws) auto

lemma concat_in_hull' [intro]:
  assumes "ws \<in> lists G"
  shows   "concat ws \<in> \<langle>G\<rangle>"
  using assms by (induction ws) auto

lemma hull_concat_lists0: "w \<in> \<langle>G\<rangle> \<Longrightarrow> (\<exists> ws \<in> lists G. concat ws = w)"
proof(rule hull.induct[of _ G])
  show "\<exists>ws\<in>lists G. concat ws = \<epsilon>"
    using concat.simps(1) lists.Nil[of G] exI[of "\<lambda> x. concat x = \<epsilon>", OF concat.simps(1)] by blast
  show " \<And>w1 w2. w1 \<in> G \<Longrightarrow> w2 \<in> \<langle>G\<rangle> \<Longrightarrow> \<exists>ws\<in>lists G. concat ws = w2 \<Longrightarrow> \<exists>ws\<in>lists G. concat ws = w1 \<cdot> w2"
    using Cons_in_lists_iff concat.simps(2) by metis
qed simp

lemma hull_concat_listsE: assumes "w \<in> \<langle>G\<rangle>"
  obtains ws where "ws \<in> lists G" and "concat ws = w"
  using assms hull_concat_lists0 by blast

lemma hull_concat_lists: "\<langle>G\<rangle> = concat ` lists G"
  using hull_concat_lists0 by blast

lemma concat_tl: "x # xs \<in> lists G \<Longrightarrow> concat xs \<in> \<langle>G\<rangle>"
  by (simp add: hull_concat_lists)

lemma nemp_concat_hull: assumes "us \<noteq> \<epsilon>" and "us \<in> lists (G - {\<epsilon>})"
  shows "concat us \<in> \<langle>G\<rangle>" and "concat us \<noteq> \<epsilon>"
  using assms by fastforce+

lemma hull_mono: "A \<subseteq> B \<Longrightarrow> \<langle>A\<rangle> \<subseteq> \<langle>B\<rangle>"
proof
  fix x assume "A \<subseteq> B" "x \<in> \<langle>A\<rangle>"
  thus "x \<in> \<langle>B\<rangle>"
    unfolding image_def hull_concat_lists using sub_lists_mono[OF \<open>A \<subseteq> B\<close>]
    by blast
qed

lemma emp_gen_set: "\<langle>{}\<rangle> = {\<epsilon>}"
  unfolding hull_concat_lists by auto

lemma concat_lists_minus[simp]: "concat ` lists (G - {\<epsilon>}) = concat ` lists G"
proof
  show "concat ` lists G \<subseteq> concat ` lists (G - {\<epsilon>})"
  proof
    fix x assume "x \<in> concat ` lists G"
    from imageE[OF this]
    obtain y where "x = concat y" "y \<in> lists G".
    from lists_minus'[OF \<open>y \<in> lists G\<close>] del_emp_concat[of y, folded \<open>x = concat y\<close>]
    show "x \<in> concat ` lists (G - {\<epsilon>})"
      by blast
  qed
qed (simp add: image_mono lists_mono)

lemma hull_drop_one: "\<langle>G - {\<epsilon>}\<rangle> = \<langle>G\<rangle>"
proof (intro equalityI subsetI)
  fix x assume "x \<in> \<langle>G\<rangle>" thus "x \<in> \<langle>G - {\<epsilon>}\<rangle>"
    unfolding  hull_concat_lists by simp
next
  fix x assume "x \<in> \<langle>G - {\<epsilon>}\<rangle>" thus "x \<in> \<langle>G\<rangle>"
    unfolding  hull_concat_lists image_iff by auto
qed

lemma sing_gen_power: "u \<in> \<langle>{x}\<rangle> \<Longrightarrow> \<exists>k. u = x\<^sup>@k"
  unfolding hull_concat_lists  using one_generated_list_power by auto

lemma sing_gen[intro]: "w \<in> \<langle>{z}\<rangle> \<Longrightarrow> w \<in> z*"
  using rootI sing_gen_power by blast

lemma pow_sing_gen[simp]: "x\<^sup>@k \<in> \<langle>{x}\<rangle>"
  using concat_in_hull[OF sing_pow_set_sub, unfolded concat_sing_pow].

lemma root_sing_gen: "w \<in> z* \<Longrightarrow> w \<in> \<langle>{z}\<rangle>"
  by (elim rootE) force

lemma sing_genE[elim]:
  assumes "u \<in> \<langle>{x}\<rangle>"
  obtains k where "x\<^sup>@k = u"
  using assms using sing_gen_power by blast

lemma sing_gen_root_conv: "w \<in> \<langle>{z}\<rangle> \<longleftrightarrow> w \<in> z*"
  using root_sing_gen by blast

lemma lists_gen_to_hull: "us \<in> lists (G - {\<epsilon>}) \<Longrightarrow> us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})"
  using lists_mono genset_sub by force

lemma rev_hull: "rev`\<langle>G\<rangle> = \<langle>rev`G\<rangle>"
proof
  show "rev ` \<langle>G\<rangle> \<subseteq> \<langle>rev ` G\<rangle>"
  proof
    fix x assume "x \<in> rev ` \<langle>G\<rangle>"
    then obtain xs where "x = rev (concat xs)" and "xs \<in> lists G"
      unfolding hull_concat_lists by auto
    from rev_in_lists[OF \<open>xs \<in> lists G\<close>]
    have "(map rev (rev xs)) \<in> lists (rev ` G)"
      by fastforce
    thus "x \<in> \<langle>rev ` G\<rangle>"
      unfolding image_iff hull_concat_lists
      using \<open>x = rev (concat xs)\<close>[unfolded rev_concat] by blast
  qed
  show "\<langle>rev ` G\<rangle> \<subseteq> rev ` \<langle>G\<rangle>"
  proof
    fix x assume  "x \<in>  \<langle>rev ` G\<rangle>"
    then obtain xs where "x = concat xs" and "xs \<in> lists (rev ` G)"
      unfolding hull_concat_lists by blast
    from rev_in_lists[OF \<open>xs \<in> lists ((rev ` G))\<close>]
    have "map rev (rev xs) \<in> lists G"
      by fastforce
    hence "rev x \<in> \<langle>G\<rangle>"
      unfolding \<open>x = concat xs\<close> rev_concat
      by fast
    thus "x \<in> rev ` \<langle>G\<rangle>"
      unfolding rev_in_conv.
  qed
qed

lemma power_in[intro]: "x \<in> \<langle>G\<rangle> \<Longrightarrow> x\<^sup>@k \<in> \<langle>G\<rangle>"
  by (induction k, auto)

lemma hull_closed_lists:  "us \<in> lists \<langle>G\<rangle> \<Longrightarrow> concat us \<in> \<langle>G\<rangle>"
  by (induct us, auto)

lemma hull_I [intro]:
  "\<epsilon> \<in> H \<Longrightarrow> (\<And> x y. x \<in> H \<Longrightarrow> y \<in> H \<Longrightarrow> x \<cdot> y \<in> H) \<Longrightarrow> \<langle>H\<rangle> = H"
  by (standard, use hull.induct[of _ H "\<lambda> x. x \<in> H"] in force) (simp only: genset_sub)

lemma self_gen: "\<langle>\<langle>G\<rangle>\<rangle> = \<langle>G\<rangle>"
  using image_subsetI[of "lists \<langle>G\<rangle>" concat "\<langle>G\<rangle>", unfolded hull_concat_lists[of "\<langle>G\<rangle>", symmetric],
      THEN subset_antisym[OF _ genset_sub[of "\<langle>G\<rangle>"]]] hull_closed_lists[of _ G] by blast

lemma hull_mono'[intro]: "A \<subseteq> \<langle>B\<rangle> \<Longrightarrow> \<langle>A\<rangle> \<subseteq> \<langle>B\<rangle>"
  using hull_mono self_gen by blast

lemma hull_conjug [elim]: "w \<in> \<langle>{r\<cdot>s,s\<cdot>r}\<rangle> \<Longrightarrow> w \<in> \<langle>{r,s}\<rangle>"
  using hull_mono[of "{r\<cdot>s,s\<cdot>r}" "\<langle>{r,s}\<rangle>", unfolded self_gen] by blast

text\<open>Intersection of hulls is a hull.\<close>

lemma hulls_inter: "\<langle>\<Inter> {\<langle>G\<rangle> | G. G \<in> S}\<rangle> = \<Inter> {\<langle>G\<rangle> | G. G \<in> S}"
proof
  {fix G assume "G \<in> S"
    hence "\<langle>\<Inter> {\<langle>G\<rangle> |G. G \<in> S}\<rangle> \<subseteq> \<langle>G\<rangle>"
      using Inter_lower[of "\<langle>G\<rangle>" "{\<langle>G\<rangle> |G. G \<in> S}"] mem_Collect_eq[of "\<langle>G\<rangle>" "\<lambda> A. \<exists> G. G \<in> S \<and> A = \<langle>G\<rangle>"]
        hull_mono[of "\<Inter> {\<langle>G\<rangle> |G. G \<in> S}" "\<langle>G\<rangle>"] unfolding self_gen by auto}
  thus "\<langle>\<Inter> {\<langle>G\<rangle> |G. G \<in> S}\<rangle> \<subseteq> \<Inter> {\<langle>G\<rangle> |G. G \<in> S}"  by blast
next
  show "\<Inter> {\<langle>G\<rangle> |G. G \<in> S} \<subseteq> \<langle>\<Inter> {\<langle>G\<rangle> |G. G \<in> S}\<rangle>"
    by simp
qed

lemma hull_keeps_root: "\<forall> u \<in> A. u \<in> r* \<Longrightarrow>  w \<in> \<langle>A\<rangle> \<Longrightarrow> w \<in> r*"
  by (rule hull.induct[of _ _ "\<lambda> x. x \<in> r*"], auto)

lemma bin_hull_keeps_root [intro]: "u \<in> r* \<Longrightarrow> v \<in> r* \<Longrightarrow> w \<in> \<langle>{u,v}\<rangle> \<Longrightarrow> w \<in> r*"
  by (rule hull.induct[of _ _ "\<lambda> x. x \<in> r*"], auto)

lemma bin_comm_hull_comm: "x \<cdot> y = y \<cdot> x \<Longrightarrow> u \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> v \<in> \<langle>{x,y}\<rangle> \<Longrightarrow>  u \<cdot> v = v \<cdot> u"
  unfolding comm_root using bin_hull_keeps_root by blast

lemma[reversal_rule]: "rev ` \<langle>{rev u, rev v}\<rangle> = \<langle>{u,v}\<rangle>"
  by (simp add: rev_hull)

lemma[reversal_rule]: "rev w \<in>  \<langle>rev ` G\<rangle> \<equiv> w \<in> \<langle>G\<rangle>"
  unfolding rev_in_conv rev_hull rev_rev_image_eq.


section "Factorization into generators"

text\<open>We define a decomposition (or a factorization) of a into elements of a given generating set. Such a decomposition is well defined only
if the decomposed word is an element of the hull. Even int that case, however, the decomposition need not be unique.\<close>

definition decompose :: "'a list set  \<Rightarrow> 'a list \<Rightarrow> 'a list list" (\<open>Dec _ _\<close> [55,55] 56) where
  "decompose G u = (SOME us. us \<in> lists (G - {\<epsilon>}) \<and> concat us = u)"

lemma dec_ex:  assumes "u \<in> \<langle>G\<rangle>" shows "\<exists> us. (us \<in> lists (G - {\<epsilon>}) \<and> concat us = u)"
  using assms unfolding image_def  hull_concat_lists[of G] mem_Collect_eq
  using del_emp_concat lists_minus' by metis

lemma dec_in_lists': "u \<in> \<langle>G\<rangle> \<Longrightarrow> (Dec G u) \<in> lists (G - {\<epsilon>})"
  unfolding decompose_def using someI_ex[OF dec_ex] by blast

lemma concat_dec[simp, intro] : "u \<in> \<langle>G\<rangle> \<Longrightarrow> concat (Dec G u) = u"
  unfolding decompose_def using someI_ex[OF dec_ex] by blast

lemma dec_emp [simp]: "Dec G \<epsilon> = \<epsilon>"
proof-
  have ex:  "\<epsilon> \<in> lists (G - {\<epsilon>}) \<and> concat \<epsilon> = \<epsilon>"
    by simp
  have all: "(us \<in> lists (G - {\<epsilon>}) \<and> concat us = \<epsilon>) \<Longrightarrow> us = \<epsilon>" for us
    using emp_concat_emp by auto
  show  ?thesis
    unfolding decompose_def
    using all[OF someI[of "\<lambda> x. x \<in> lists (G - {\<epsilon>}) \<and> concat x = \<epsilon>", OF ex]].
qed

lemma dec_nemp: "u \<in> \<langle>G\<rangle> - {\<epsilon>} \<Longrightarrow>  Dec G u \<noteq> \<epsilon>"
  using concat_dec[of u G] by force

lemma dec_nemp'[simp, intro]: "u \<noteq> \<epsilon> \<Longrightarrow> u \<in> \<langle>G\<rangle> \<Longrightarrow> Dec G u \<noteq> \<epsilon>"
  using dec_nemp by blast

lemma dec_eq_emp_iff [simp]: assumes "u \<in> \<langle>G\<rangle>" shows "Dec G u = \<epsilon> \<longleftrightarrow> u = \<epsilon>"
  using dec_nemp'[OF _ \<open>u \<in> \<langle>G\<rangle>\<close>] by auto

lemma dec_in_lists[simp]: "u \<in> \<langle>G\<rangle> \<Longrightarrow> Dec G u \<in> lists G"
  using dec_in_lists' by auto

lemma set_dec_sub: "x \<in> \<langle>G\<rangle> \<Longrightarrow> set (Dec G x) \<subseteq> G"
  using dec_in_lists by blast

lemma dec_hd: "u \<noteq> \<epsilon> \<Longrightarrow> u \<in> \<langle>G\<rangle> \<Longrightarrow> hd (Dec G u) \<in> G"
  by simp

lemma non_gen_dec: assumes  "u \<in> \<langle>G\<rangle>" "u \<notin> G" shows "Dec G u  \<noteq> [u]"
  using dec_in_lists[OF \<open>u \<in> \<langle>G\<rangle>\<close>] Cons_in_lists_iff[of u \<epsilon> G] \<open>u \<notin> G\<close> by argo

subsection \<open>Refinement into a specific decomposition\<close>

text\<open>We extend the decomposition to lists of words. This can be seen as a refinement of a previous decomposition of some word.\<close>

definition refine :: "'a list set \<Rightarrow> 'a list list \<Rightarrow> 'a list list" (\<open>Ref _ _\<close> [51,51] 65) where
  "refine G us = concat(map (decompose G) us)"

lemma ref_morph: "us \<in> lists \<langle>G\<rangle> \<Longrightarrow> vs \<in> lists \<langle>G\<rangle> \<Longrightarrow> refine G (us \<cdot> vs) = refine G us \<cdot> refine G vs"
  unfolding refine_def by simp

lemma ref_conjug:
  "u \<sim> v \<Longrightarrow> (Ref G u) \<sim> Ref G v"
  unfolding refine_def by (intro conjug_concat_conjug map_conjug)

lemma ref_morph_plus: "us \<in> lists (\<langle>G\<rangle> - {\<epsilon>}) \<Longrightarrow> vs \<in> lists (\<langle>G\<rangle> - {\<epsilon>}) \<Longrightarrow> refine G (us \<cdot> vs) = refine G us \<cdot> refine G vs"
  unfolding refine_def by simp

lemma ref_pref_mono: "ws \<in> lists \<langle>G\<rangle> \<Longrightarrow> us \<le>p ws \<Longrightarrow> Ref G us \<le>p Ref G ws"
  unfolding prefix_def using ref_morph append_in_lists_dest' append_in_lists_dest by metis

lemma ref_suf_mono: "ws \<in> lists \<langle>G\<rangle> \<Longrightarrow> us \<le>s ws \<Longrightarrow> (Ref G us) \<le>s Ref G ws"
  unfolding suffix_def using ref_morph append_in_lists_dest' append_in_lists_dest by metis

lemma ref_fac_mono: "ws \<in> lists \<langle>G\<rangle> \<Longrightarrow> us \<le>f ws \<Longrightarrow> (Ref G us) \<le>f Ref G ws"
  unfolding sublist_altdef' using ref_pref_mono ref_suf_mono  suf_in_lists by metis

lemma ref_pop_hd: "us \<noteq> \<epsilon> \<Longrightarrow> us \<in> lists \<langle>G\<rangle> \<Longrightarrow> refine G us = decompose G (hd us) \<cdot> refine G (tl us)"
  unfolding  refine_def  using list.simps(9)[of "decompose G" "hd us" "tl us"] by simp

lemma ref_in: "us \<in> lists \<langle>G\<rangle> \<Longrightarrow> (Ref G us) \<in> lists (G - {\<epsilon>})"
proof (induction us)
  case (Cons a us)
  then show ?case
    using Cons.IH Cons.prems dec_in_lists' by (auto simp add: refine_def)
qed (simp add: refine_def)

lemma ref_in'[intro]: "us \<in> lists \<langle>G\<rangle> \<Longrightarrow> (Ref G us) \<in> lists G"
  using ref_in by auto

lemma concat_ref: "us \<in> lists \<langle>G\<rangle> \<Longrightarrow> concat (Ref G us) = concat us"
proof (induction us)
  case (Cons a us)
  then show ?case
    using Cons.IH Cons.prems concat_dec refine_def by (auto simp add: refine_def)
qed (simp add: refine_def)

lemma ref_gen: "us \<in> lists B \<Longrightarrow> B \<subseteq> \<langle>G\<rangle> \<Longrightarrow> Ref G us \<in> \<langle>decompose G ` B\<rangle>"
  by (induct us, auto simp add: refine_def)

lemma ref_set: "ws \<in> lists \<langle>G\<rangle> \<Longrightarrow> set (Ref G ws) = \<Union> (set`(decompose G)`set ws)"
  by (simp add: refine_def)

lemma emp_ref: assumes "us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})" and  "Ref G us = \<epsilon>" shows "us = \<epsilon>"
  using emp_concat_emp[OF \<open>us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})\<close>]
    concat_ref [OF lists_minus[OF assms(1)], unfolded \<open>Ref G us = \<epsilon>\<close> concat.simps(1),symmetric] by blast

lemma sing_ref_sing:
  assumes "us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})" and "refine G us = [b]"
  shows "us = [b]"
proof-
  have "us \<noteq> \<epsilon>"
    using \<open>refine G us = [b]\<close> by (auto simp add: refine_def)
  have "tl us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})" and "hd us \<in> \<langle>G\<rangle> - {\<epsilon>}"
    using list.collapse[OF \<open>us \<noteq> \<epsilon>\<close>]  \<open>us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})\<close> Cons_in_lists_iff[of "hd us" "tl us" "\<langle>G\<rangle> - {\<epsilon>}"]
    by auto
  have "Dec G (hd us) \<noteq> \<epsilon>"
    using dec_nemp[OF \<open>hd us \<in> \<langle>G\<rangle> - {\<epsilon>}\<close>].
  have "us \<in> lists \<langle>G\<rangle>"
    using \<open>us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})\<close> lists_minus by auto
  have "concat us = b"
    using \<open>us \<in> lists \<langle>G\<rangle>\<close> assms(2) concat_ref by force
  have "refine G (tl us) = \<epsilon>"
    using ref_pop_hd[OF \<open>us \<noteq> \<epsilon>\<close> \<open>us \<in> lists \<langle>G\<rangle>\<close>]  unfolding  \<open>refine G us = [b]\<close>
    using \<open>Dec G (hd us) \<noteq> \<epsilon>\<close> Cons_eq_append_conv[of b \<epsilon> "(Dec G (hd us))" "(Ref G (tl us))"]
      Cons_eq_append_conv[of b \<epsilon> "(Dec G (hd us))" "(Ref G (tl us))"]  append_is_Nil_conv[of _ "(Ref G (tl us))"]
    by blast
  from  emp_ref[OF \<open>tl us \<in> lists (\<langle>G\<rangle> - {\<epsilon>})\<close> this, symmetric]
  have "\<epsilon> = tl us".
  from this[unfolded Nil_tl]
  show ?thesis
    using \<open>us \<noteq> \<epsilon>\<close> \<open>concat us = b\<close> by auto
qed

lemma ref_ex: assumes "Q \<subseteq> \<langle>G\<rangle>" and "us \<in> lists Q"
  shows "Ref G us \<in> lists (G - {\<epsilon>})" and "concat (Ref G us) = concat us"
  using ref_in[OF sub_lists_mono[OF assms]] concat_ref[OF sub_lists_mono[OF assms]].

section "Basis"

text\<open>An important property of monoids of words is that they have a unique minimal generating set. Which is the set consisting of indecomposable elements.\<close>

text\<open>The simple element is defined as a word which has only trivial decomposition into generators: a singleton.\<close>

definition simple_element :: "'a list \<Rightarrow> 'a list set  \<Rightarrow> bool" (\<open> _ \<in>B _ \<close> [51,51] 50) where
  "simple_element b G = (b \<in> G \<and> (\<forall> us. us \<in> lists (G - {\<epsilon>}) \<and> concat us = b \<longrightarrow> \<^bold>|us\<^bold>| = 1))"

lemma simp_el_el: "b \<in>B G \<Longrightarrow> b \<in> G"
  unfolding simple_element_def by blast

lemma simp_elD: "b \<in>B G \<Longrightarrow> us \<in> lists (G - {\<epsilon>}) \<Longrightarrow> concat us = b \<Longrightarrow> \<^bold>|us\<^bold>| = 1"
  unfolding simple_element_def by blast

lemma simp_el_sing: assumes "b \<in>B G" "us \<in> lists (G - {\<epsilon>})" "concat us = b" shows "us = [b]"
  using \<open>concat us = b\<close> concat_len_one[OF simp_elD[OF assms]] sing_word[OF simp_elD[OF assms]] by simp

lemma nonsimp: "us \<in> lists (G - {\<epsilon>}) \<Longrightarrow> concat us \<in>B G \<Longrightarrow>  us = [concat us]"
  using simp_el_sing[of "concat us" G us]   unfolding simple_element_def
  by blast

lemma emp_nonsimp: assumes "b \<in>B G" shows "b \<noteq> \<epsilon>"
  using simp_elD[OF assms, of \<epsilon>] by force

lemma basis_no_fact: assumes "u \<in> \<langle>G\<rangle>" and "v \<in> \<langle>G\<rangle>" and "u \<cdot> v \<in>B G" shows "u = \<epsilon> \<or> v = \<epsilon>"
proof-
  have eq1: "concat ((Dec G u) \<cdot> (Dec G v)) = u \<cdot> v"
    using concat_morph[of "Dec G u" "Dec G v"]
    unfolding concat_dec[OF \<open>u \<in> \<langle>G\<rangle>\<close>] concat_dec[OF \<open>v \<in> \<langle>G\<rangle>\<close>].
  have eq2: "(Dec G u) \<cdot> (Dec G v) = [u \<cdot> v]"
    using  \<open>u \<cdot> v \<in>B G\<close> nonsimp[of "(Dec G u) \<cdot> (Dec G v)"]
    unfolding eq1 append_in_lists_conv[of "(Dec G u)" "(Dec G v)" "G - {\<epsilon>}"]
    using dec_in_lists'[OF \<open>u \<in> \<langle>G\<rangle>\<close>] dec_in_lists'[OF \<open>v \<in> \<langle>G\<rangle>\<close>]
    by (meson append_in_lists_conv)
  have "Dec G u = \<epsilon> \<or> Dec G v = \<epsilon>"
    using butlast_append[of "Dec G u" "Dec G v"]  unfolding eq2 butlast.simps(2)[of "u\<cdot>v" \<epsilon>]
    using   Nil_is_append_conv[of "Dec G u" "butlast (Dec G v)"] by auto
  thus ?thesis
    using concat_dec[OF \<open>u \<in> \<langle>G\<rangle>\<close>] concat_dec[OF \<open>v \<in> \<langle>G\<rangle>\<close>]
      concat.simps(1)
    by auto
qed

lemma simp_elI:
  assumes "b \<in> G" and "b \<noteq> \<epsilon>"  and all: "\<forall> u v. u \<noteq> \<epsilon> \<and> u \<in> \<langle>G\<rangle> \<and> v \<noteq> \<epsilon> \<and> v \<in> \<langle>G\<rangle> \<longrightarrow> u \<cdot> v \<noteq> b"
  shows "b \<in>B G"
  unfolding simple_element_def
proof(rule conjI)
  show "\<forall>us. us \<in> lists (G - {\<epsilon>}) \<and> concat us = b \<longrightarrow> \<^bold>|us\<^bold>| = 1"
  proof (rule allI, rule impI, elim conjE)
    fix us assume "us \<in> lists (G - {\<epsilon>})" "concat us = b"
    hence "us \<noteq> \<epsilon>" using \<open>b \<noteq> \<epsilon>\<close> concat.simps(1) by blast
    hence "hd us \<in> \<langle>G\<rangle>" and "hd us \<noteq> \<epsilon>"
      using \<open>us \<in> lists (G - {\<epsilon>})\<close> lists_hd_in_set  gen_in by auto
    have "tl us = \<epsilon>"
    proof(rule ccontr)
      assume "tl us \<noteq> \<epsilon>"
      from nemp_concat_hull[of "tl us", OF this tl_in_lists[OF \<open>us \<in> lists (G - {\<epsilon>})\<close>]]
      show False
        using all \<open>hd us \<noteq> \<epsilon>\<close> \<open>hd us \<in> \<langle>G\<rangle>\<close>  concat.simps(2)[of "hd us" "tl us", symmetric]
        unfolding list.collapse[OF \<open>us \<noteq> \<epsilon>\<close>] \<open>concat us = b\<close>
        by blast
    qed
    thus "\<^bold>|us\<^bold>| = 1"
      using  long_list_tl[of us] Nitpick.size_list_simp(2)[of us] \<open>us \<noteq> \<epsilon>\<close> by fastforce
  qed
qed (simp add: \<open>b \<in> G\<close>)

lemma simp_el_indecomp:
  assumes "b \<in>B G" "u \<noteq> \<epsilon>" "u \<in> \<langle>G\<rangle>" "v \<noteq> \<epsilon>" "v \<in> \<langle>G\<rangle>"
  shows "u \<cdot> v \<noteq> b"
  using basis_no_fact[OF \<open>u \<in> \<langle>G\<rangle>\<close> \<open>v \<in> \<langle>G\<rangle>\<close>] \<open>u \<noteq> \<epsilon>\<close> \<open>v \<noteq> \<epsilon>\<close> \<open>b \<in>B G\<close> by blast

text\<open>We are ready to define the \emph{basis} as the set of all simple elements.\<close>

definition basis :: "'a list set  \<Rightarrow> 'a list set" (\<open>\<BB> _\<close> [51] ) where
  "basis G = {x. x \<in>B G}"

lemma basis_inI: "x \<in>B G \<Longrightarrow> x \<in> \<BB> G"
  unfolding basis_def by simp

lemma basisD: "x \<in> \<BB> G \<Longrightarrow> x \<in>B G"
  unfolding basis_def by simp

lemma emp_not_basis: "x \<in> \<BB> G \<Longrightarrow> x \<noteq> \<epsilon>"
  using basisD emp_nonsimp by blast

lemma basis_sub: "\<BB> G \<subseteq> G"
  unfolding  basis_def simple_element_def by simp

lemma basis_drop_emp: "(\<BB> G) - {\<epsilon>} = \<BB> G"
  using emp_not_basis by blast

lemma simp_el_hull':  assumes "b \<in>B \<langle>G\<rangle>"  shows "b \<in>B G"
proof-
  have all: "\<forall>us. us \<in> lists (G - {\<epsilon>}) \<and> concat us = b \<longrightarrow> \<^bold>|us\<^bold>| = 1"
    using assms lists_gen_to_hull unfolding simple_element_def by metis
  have "b \<in> \<langle>G\<rangle>"
    using assms simp_elD unfolding simple_element_def by blast
  obtain bs where "bs \<in> lists (G - {\<epsilon>})" and "concat bs = b"
    using  dec_ex[OF \<open>b \<in> \<langle>G\<rangle>\<close>] by blast
  have "b \<in> G"
    using
      lists_minus[OF \<open>bs \<in> lists (G - {\<epsilon>})\<close>]
      lists_gen_to_hull[OF \<open>bs \<in> lists (G - {\<epsilon>})\<close>, THEN nonsimp[of bs "\<langle>G\<rangle>"],
        unfolded \<open>concat bs = b\<close>, OF \<open>b \<in>B \<langle>G\<rangle>\<close>] by force
  thus "b \<in>B G"
    by (simp add: all simple_element_def)
qed

lemma simp_el_hull:  assumes "b \<in>B G" shows "b \<in>B \<langle>G\<rangle>"
  using simp_elI[of b "\<langle>G\<rangle>", OF _ emp_nonsimp[OF assms]] unfolding self_gen
  using simp_el_indecomp[OF \<open>b \<in>B G\<close>] gen_in[OF simp_el_el[OF assms]] by presburger

lemma concat_tl_basis: "x # xs \<in> lists \<BB> G \<Longrightarrow> concat xs \<in> \<langle>G\<rangle>"
  unfolding hull_concat_lists basis_def simple_element_def by auto

text\<open>The basis generates the hull\<close>

lemma set_concat_len: assumes "us \<in> lists (G - {\<epsilon>})" "1 < \<^bold>|us\<^bold>|" "u \<in> set us" shows "\<^bold>|u\<^bold>| < \<^bold>|concat us\<^bold>|"
proof-
  obtain x y where "us = x \<cdot> [u] \<cdot> y" and "x \<cdot> y \<noteq> \<epsilon>"
    using split_list_long[OF \<open>1 < \<^bold>|us\<^bold>|\<close> \<open>u \<in> set us\<close>].
  hence "x \<cdot> y \<in> lists (G - {\<epsilon>})"
    using \<open>us \<in> lists (G - {\<epsilon>})\<close> by auto
  hence "\<^bold>|concat (x \<cdot> y)\<^bold>| \<noteq> 0"
    using \<open>x \<cdot> y \<noteq> \<epsilon>\<close> in_lists_conv_set  by force
  hence "\<^bold>|concat us\<^bold>| = \<^bold>|u\<^bold>| + \<^bold>|concat x\<^bold>| + \<^bold>|concat y\<^bold>|"
    using lenmorph \<open>us = x \<cdot> [u] \<cdot> y\<close> by simp
  thus ?thesis
    using \<open>\<^bold>|concat (x \<cdot> y)\<^bold>| \<noteq> 0\<close> by auto
qed

lemma non_simp_dec: assumes "w \<notin> \<BB> G" "w \<noteq> \<epsilon>" "w \<in> G"
  obtains us where "us \<in> lists (G - {\<epsilon>})" "1 < \<^bold>|us\<^bold>|" "concat us = w"
  using \<open>w \<noteq> \<epsilon>\<close> \<open>w \<in> G\<close> \<open>w \<notin> \<BB> G\<close>  basis_inI[of w G, unfolded simple_element_def]
  using concat.simps(1) nemp_le_len nless_le by metis


lemma basis_gen: "w \<in> G \<Longrightarrow>  w \<in> \<langle>\<BB> G\<rangle>"
proof (induct "length w" arbitrary: w rule: less_induct)
  case less
  show ?case
  proof (cases "w \<in> \<BB> G \<or> w = \<epsilon>", blast)
    assume "\<not> (w \<in> \<BB> G \<or> w = \<epsilon>)"
    with \<open>w \<in> G\<close>
    obtain us where "us \<in> lists (G - {\<epsilon>})" "1 < \<^bold>|us\<^bold>|" "concat us = w"
      using non_simp_dec by blast
    have "u \<in> set us \<Longrightarrow> u \<in> \<langle>\<BB> G\<rangle>" for u
      using  lists_minus[OF \<open>us \<in> lists (G - {\<epsilon>})\<close>] less(1)[OF set_concat_len[OF \<open>us \<in> lists (G - {\<epsilon>})\<close> \<open>1 < \<^bold>|us\<^bold>|\<close>, unfolded \<open>concat us = w\<close>], of u]
      by blast
    thus "w \<in> \<langle>\<BB> G\<rangle> "
      unfolding \<open>concat us = w\<close>[symmetric]
      using hull_closed_lists[OF in_listsI] by blast
  qed
qed

lemmas basis_concat_listsE = hull_concat_listsE[OF basis_gen]

theorem basis_gen_hull: "\<langle>\<BB> G\<rangle> = \<langle>G\<rangle>"
proof
  show "\<langle>\<BB> G\<rangle> \<subseteq> \<langle>G\<rangle>"
    unfolding hull_concat_lists basis_def simple_element_def by auto
  show  "\<langle>G\<rangle> \<subseteq> \<langle>\<BB> G\<rangle>"
  proof
    fix x  show  "x \<in> \<langle>G\<rangle> \<Longrightarrow> x \<in> \<langle>\<BB> G\<rangle>"
    proof (induct rule: hull.induct)
      show "\<And>w1 w2. w1 \<in> G \<Longrightarrow> w2 \<in> \<langle>\<BB> G\<rangle> \<Longrightarrow> w1 \<cdot> w2 \<in> \<langle>\<BB> G\<rangle>"
        using hull_closed[of _ "\<BB> G"] basis_gen[of _ G]  by blast
    qed auto
  qed
qed

lemma basis_gen_hull': "\<langle>\<BB> \<langle>G\<rangle>\<rangle> = \<langle>G\<rangle>"
  using basis_gen_hull self_gen by blast

theorem basis_of_hull: "\<BB> \<langle>G\<rangle> = \<BB> G"
proof
  show "\<BB> G  \<subseteq>  \<BB> \<langle>G\<rangle>"
    using basisD basis_inI simp_el_hull by blast
  show "\<BB> \<langle>G\<rangle>  \<subseteq>  \<BB> G"
    using basisD basis_inI simp_el_hull' by blast
qed

lemma basis_hull_sub: "\<BB> \<langle>G\<rangle> \<subseteq> G"
  using basis_of_hull basis_sub by blast

text\<open>The basis is the smallest generating set.\<close>
theorem basis_sub_gen:  "\<langle>S\<rangle> = \<langle>G\<rangle> \<Longrightarrow> \<BB> G \<subseteq> S"
  using basis_of_hull basis_sub by metis

lemma basis_min_gen: "S \<subseteq> \<BB> G \<Longrightarrow> \<langle>S\<rangle> = G \<Longrightarrow> S = \<BB> G"
  using basis_of_hull basis_sub by blast

lemma basisI: "(\<And> B. \<langle>B\<rangle> = \<langle>C\<rangle> \<Longrightarrow> C \<subseteq> B) \<Longrightarrow> \<BB> \<langle>C\<rangle> = C"
  using basis_gen_hull basis_min_gen basis_of_hull by metis

thm basis_inI

text\<open>An arbitrary set between basis and the hull is generating...\<close>
lemma gen_sets: assumes "\<BB> G \<subseteq> S" and "S \<subseteq> \<langle>G\<rangle>" shows "\<langle>S\<rangle> = \<langle>G\<rangle>"
  using  image_mono[OF lists_mono[of S "\<langle>G\<rangle>"], of concat, OF \<open>S \<subseteq> \<langle>G\<rangle>\<close>] image_mono[OF lists_mono[of "\<BB> G" S], of concat, OF \<open>\<BB> G \<subseteq> S\<close>]
  unfolding sym[OF hull_concat_lists]  basis_gen_hull
  using  subset_antisym[of "\<langle>S\<rangle>" "\<langle>G\<rangle>"] self_gen by metis

text\<open>... and has the same basis\<close>
lemma basis_sets: "\<BB> G \<subseteq> S \<Longrightarrow> S \<subseteq> \<langle>G\<rangle> \<Longrightarrow> \<BB> G = \<BB> S"
  by (metis basis_of_hull  gen_sets)

text\<open>Any nonempty composed element has a decomposition into basis elements with many useful properties\<close>

lemma non_simp_fac: assumes "w \<noteq> \<epsilon>" and "w \<in> \<langle>G\<rangle>" and "w \<notin> \<BB> G"
  obtains us where "1 < \<^bold>|us\<^bold>|" and "us \<noteq> \<epsilon>" and  "us \<in> lists \<BB> G" and
    "hd us \<noteq> \<epsilon>" and "hd us \<in> \<langle>G\<rangle>" and
    "concat(tl us) \<noteq> \<epsilon>" and "concat(tl us) \<in> \<langle>G\<rangle>" and
    "w = hd us \<cdot> concat(tl us)"
proof-
  obtain us where "us \<in> lists \<BB> G" and "concat us = w"
    using \<open>w \<in> \<langle>G\<rangle>\<close> dec_in_lists[of w "\<BB> G"] concat_dec[of w "\<BB> G"]
    unfolding basis_gen_hull
    by blast
  hence "us \<noteq> \<epsilon>"
    using  \<open>w \<noteq> \<epsilon>\<close> concat.simps(1)
    by blast
  from lists_hd_in_set[OF this \<open>us \<in> lists \<BB> G\<close>, THEN emp_not_basis]
    lists_hd_in_set[OF this \<open>us \<in> lists \<BB> G\<close>, THEN gen_in[of "hd us" "\<BB> G", unfolded basis_gen_hull]]
  have "hd us \<noteq> \<epsilon>" and "hd us \<in> \<langle>G\<rangle>".
  have  "1 < \<^bold>|us\<^bold>|"
    using \<open>w \<notin> \<BB> G\<close> lists_hd_in_set[OF \<open>us \<noteq> \<epsilon>\<close> \<open>us \<in> lists \<BB> G\<close>] \<open>w \<noteq> \<epsilon>\<close> \<open>w \<in> \<langle>G\<rangle>\<close>
      concat_len_one[of us, unfolded \<open>concat us = w\<close>]
      \<open>us \<noteq> \<epsilon>\<close> leI nemp_le_len order_antisym_conv by metis
  from nemp_concat_hull[OF long_list_tl[OF this], of "\<BB> G", unfolded basis_drop_emp basis_gen_hull, OF tl_in_lists[OF \<open>us \<in> lists \<BB> G\<close>]]
  have "concat (tl us) \<in> \<langle>G\<rangle>" and "concat(tl us) \<noteq> \<epsilon>".
  have "w = hd us \<cdot> concat(tl us)"
    using \<open>us \<noteq> \<epsilon>\<close> \<open>us \<in> lists \<BB> G\<close> \<open>concat us = w\<close> concat.simps(2)[of "hd us" "tl us"] list.collapse[of us]
    by argo
  from that[OF \<open>1 < \<^bold>|us\<^bold>|\<close> \<open>us \<noteq> \<epsilon>\<close> \<open>us \<in> lists \<BB> G\<close> \<open>hd us \<noteq> \<epsilon>\<close> \<open>hd us \<in> \<langle>G\<rangle>\<close> \<open>concat (tl us) \<noteq> \<epsilon>\<close> \<open>concat (tl us) \<in> \<langle>G\<rangle>\<close> this]
  show thesis.
qed

lemma basis_dec: "p \<in> \<langle>G\<rangle> \<Longrightarrow> s \<in> \<langle>G\<rangle> \<Longrightarrow> p \<cdot> s \<in> \<BB> G \<Longrightarrow> p = \<epsilon> \<or> s = \<epsilon>"
  using basis_no_fact[of p G s] unfolding basis_def by simp

lemma non_simp_fac': "w \<notin> \<BB> G \<Longrightarrow> w \<noteq> \<epsilon> \<Longrightarrow> w \<in> \<langle>G\<rangle> \<Longrightarrow> \<exists>us. us \<in> lists (G - {\<epsilon>}) \<and> w = concat us \<and> \<^bold>|us\<^bold>| \<noteq> 1"
  by (metis basis_inI concat_len_one dec_in_lists' dec_in_lists concat_dec dec_nemp lists_hd_in_set nemp_elem_setI simple_element_def)

lemma emp_gen_iff: "(G - {\<epsilon>}) = {} \<longleftrightarrow> \<langle>G\<rangle> = {\<epsilon>}"
proof
  assume "G - {\<epsilon>} = {}" show "\<langle>G\<rangle> = {\<epsilon>}"
    using  hull_drop_one[of G, unfolded \<open>G - {\<epsilon>} = {}\<close> emp_gen_set] by simp
next
  assume "\<langle>G\<rangle> = {\<epsilon>}" thus"G - {\<epsilon>} = {}" by blast
qed

lemma emp_basis_iff:  "\<BB> G = {} \<longleftrightarrow> G - {\<epsilon>} = {}"
  using emp_gen_iff[of "\<BB> G", unfolded basis_gen_hull basis_drop_emp, folded emp_gen_iff].

section "Code"

locale nemp_words =
  fixes G
  assumes emp_not_in: "\<epsilon> \<notin> G"

begin
lemma drop_empD: "G - {\<epsilon>} = G"
  using emp_not_in by simp

lemmas emp_concat_emp' = emp_concat_emp[of _ G, unfolded drop_empD]

thm disjE[OF ruler[OF take_is_prefix take_is_prefix]]

lemma concat_take_mono: assumes "ws \<in> lists G" and "concat (take i ws) \<le>p concat (take j ws)"
  shows "take i ws \<le>p take j ws"
proof (rule disjE[OF ruler[OF take_is_prefix take_is_prefix]])
  assume "take j ws \<le>p take i ws"
  from prefixE[OF this]
  obtain us where "take i ws = take j ws \<cdot> us".
  hence "us \<in> lists G" using \<open>ws \<in> lists G\<close>
    using append_in_lists_conv take_in_lists by metis
  have "concat (take j ws) = concat (take i ws)"
    using pref_concat_pref[OF \<open>take j ws \<le>p take i ws\<close>] assms(2) by simp
  from arg_cong[OF \<open>take i ws = take j ws \<cdot> us\<close>, of concat, unfolded concat_morph, unfolded this]
  have "us = \<epsilon>"
    using \<open>us \<in> lists G\<close> emp_concat_emp' by blast
  thus  "take i ws \<le>p take j ws"
    using \<open>take i ws = take j ws \<cdot> us\<close> by force
qed

lemma nemp: "x \<in> G \<Longrightarrow> x \<noteq> \<epsilon>"
  using emp_not_in by blast

lemma code_concat_eq_emp_iff [simp]: "us \<in> lists G \<Longrightarrow> concat us = \<epsilon> \<longleftrightarrow> us = \<epsilon>"
  unfolding in_lists_conv_set concat_eq_Nil_conv
  by (simp add: nemp)

lemma root_dec_inj_on: "inj_on (\<lambda> x. [\<rho> x]\<^sup>@(e\<^sub>\<rho> x)) G"
  unfolding inj_on_def using primroot_exp_eq
  unfolding concat_sing_pow[of "\<rho> _", symmetric] by metis


lemma concat_root_dec_eq_concat:
  assumes "ws \<in> lists G"
  shows "concat (concat (map (\<lambda> x. [\<rho> x]\<^sup>@(e\<^sub>\<rho> x)) ws)) = concat ws"
    (is "concat(concat (map ?R ws)) = concat ws")
  using assms
  by (induction ws, simp_all add: nemp)

end

text\<open>A basis freely generating its hull is called a \emph{code}. By definition,
this means that generated elements have unique factorizations into the elements of the code.\<close>

locale code =
  fixes \<C>
  assumes is_code: "xs \<in> lists \<C> \<Longrightarrow> ys \<in> lists \<C> \<Longrightarrow> concat xs = concat ys \<Longrightarrow> xs = ys"
begin

lemma code_not_comm: "x \<in> \<C> \<Longrightarrow> y \<in> \<C> \<Longrightarrow> x \<noteq> y \<Longrightarrow> x \<cdot> y \<noteq> y \<cdot> x"
  using is_code[of "[x,y]" "[y,x]"]  by auto

lemma emp_not_in: "\<epsilon> \<notin> \<C>"
proof
  assume "\<epsilon> \<in> \<C>"
  hence "[] \<in> lists \<C>" and "[\<epsilon>] \<in> lists \<C>" and "concat [] = concat [\<epsilon>]" and "[] \<noteq> [\<epsilon>]"
    by simp+
  thus False
    using is_code by blast
qed

lemma nemp: "u \<in> \<C> \<Longrightarrow> u \<noteq> \<epsilon>"
  using emp_not_in by force


sublocale nemp_words \<C>
  using emp_not_in by unfold_locales

lemma code_simple: "c \<in> \<C> \<Longrightarrow> c \<in>B \<C>"
  unfolding   simple_element_def
proof
  fix c assume "c \<in> \<C>"
  hence "[c] \<in> lists \<C>"
    by simp
  show "\<forall>us. us \<in> lists (\<C> - {\<epsilon>}) \<and> concat us = c \<longrightarrow> \<^bold>|us\<^bold>| = 1"
  proof
    fix us
    {assume "us \<in> lists (\<C> - {\<epsilon>})" "concat us = c"
      hence "us \<in> lists \<C>" by blast
      hence  "us = [c]"
        using \<open>concat us = c\<close> \<open>c \<in> \<C>\<close> is_code[of "[c]", OF \<open>[c] \<in> lists \<C>\<close> \<open>us \<in> lists \<C>\<close>] emp_not_in by auto}
    thus "us \<in> lists (\<C> - {\<epsilon>}) \<and> concat us = c \<longrightarrow> \<^bold>|us\<^bold>| = 1"
      using sing_len[of c] by fastforce
  qed
qed

lemma code_is_basis: "\<BB> \<C> = \<C>"
  using code_simple basis_def[of \<C>] basis_sub by blast

lemma code_unique_dec': "us \<in> lists \<C> \<Longrightarrow> Dec \<C> (concat us) = us"
  using dec_in_lists[of "concat us" \<C>, THEN is_code, of us]
    concat_dec[of "concat us" \<C>] hull_concat_lists[of \<C>] image_eqI[of "concat us" concat us "lists \<C>"]
  by argo

lemma code_unique_dec [intro!]: "us \<in> lists \<C> \<Longrightarrow> concat us = u \<Longrightarrow>  Dec \<C> u = us"
  using code_unique_dec' by blast

lemma triv_refine[intro!] : "us \<in> lists \<C> \<Longrightarrow> concat us = u \<Longrightarrow>  Ref \<C> [u] = us"
  using code_unique_dec' by (auto simp add: refine_def)

lemma code_unique_ref: "us \<in> lists \<langle>\<C>\<rangle> \<Longrightarrow> refine \<C> us = decompose \<C> (concat us)"
proof-
  assume "us \<in> lists \<langle>\<C>\<rangle>"
  hence "concat (refine \<C> us) = concat us"
    using concat_ref by blast
  hence eq: "concat (refine \<C> us) = concat (decompose \<C> (concat us))"
    using  concat_dec[OF hull_closed_lists[OF \<open>us \<in> lists \<langle>\<C>\<rangle>\<close>]] by auto
  have dec: "Dec \<C> (concat us) \<in> lists \<C>"
    using \<open>us \<in> lists \<langle>\<C>\<rangle>\<close> dec_in_lists hull_closed_lists
    by metis
  have "Ref \<C> us \<in> lists \<C>"
    using lists_minus[OF ref_in[OF \<open>us \<in> lists \<langle>\<C>\<rangle>\<close>]].
  from  is_code[OF this dec eq]
  show ?thesis.
qed

lemma refI [intro]: "us \<in> lists \<langle>\<C>\<rangle> \<Longrightarrow> vs \<in> lists \<C> \<Longrightarrow> concat vs = concat us \<Longrightarrow> Ref \<C> us = vs"
  unfolding code_unique_ref code_unique_dec..

lemma code_dec_morph: assumes "x \<in> \<langle>\<C>\<rangle>" "y \<in> \<langle>\<C>\<rangle>"
  shows "(Dec \<C> x) \<cdot> (Dec \<C> y) = Dec \<C> (x\<cdot>y)"
proof-
  have eq: "(Dec \<C> x) \<cdot> (Dec \<C> y) = Dec \<C> (concat ((Dec \<C> x) \<cdot> (Dec \<C> y)))"
    using dec_in_lists[OF \<open>x \<in> \<langle>\<C>\<rangle>\<close>] dec_in_lists[OF \<open>y \<in> \<langle>\<C>\<rangle>\<close>]
      code.code_unique_dec[OF code_axioms, of "(Dec \<C> x) \<cdot> (Dec \<C> y)", unfolded append_in_lists_conv, symmetric]
    by presburger
  moreover have "concat ((Dec \<C> x) \<cdot> (Dec \<C> y)) = (x \<cdot> y)"
    using concat_morph[of "Dec \<C> x" "Dec \<C> y"]
    unfolding concat_dec[OF \<open>x \<in> \<langle>\<C>\<rangle>\<close>] concat_dec[OF \<open>y \<in> \<langle>\<C>\<rangle>\<close>].
  ultimately show "(Dec \<C> x) \<cdot> (Dec \<C> y) = Dec \<C> (x\<cdot>y)"
    by argo
qed

lemma dec_pow: "rs \<in> \<langle>\<C>\<rangle> \<Longrightarrow> Dec \<C> (rs\<^sup>@k) = (Dec \<C> rs)\<^sup>@k"
proof(induction k arbitrary: rs, fastforce)
  case (Suc k)
  then show ?case
    using code_dec_morph pow_Suc power_in by metis
qed

lemma code_el_dec: "c \<in> \<C> \<Longrightarrow> decompose \<C> c = [c]"
  by fastforce

lemma code_ref_list: "us \<in> lists \<C> \<Longrightarrow> refine \<C> us = us"
proof (induct us)
  case (Cons a us)
  then show ?case using code_el_dec
    unfolding refine_def by simp
qed (simp add: refine_def)

lemma code_ref_gen: assumes "G \<subseteq> \<langle>\<C>\<rangle>" "u \<in> \<langle>G\<rangle>"
  shows "Dec \<C> u \<in> \<langle>decompose \<C> ` G\<rangle>"
proof-
  have "refine \<C> (Dec G u) = Dec \<C> u"
    using  dec_in_lists[OF \<open>u \<in> \<langle>G\<rangle>\<close>]  \<open>G \<subseteq> \<langle>\<C>\<rangle>\<close> code_unique_ref[of "Dec G u", unfolded concat_dec[OF \<open>u \<in> \<langle>G\<rangle>\<close>]] by blast
  from ref_gen[of "Dec G u" G, OF dec_in_lists[OF \<open>u \<in> \<langle>G\<rangle>\<close>], of \<C>, unfolded this, OF \<open>G \<subseteq> \<langle>\<C>\<rangle>\<close>]
  show ?thesis.
qed

find_theorems "\<rho> ?x \<^sup>@ ?k = ?x" "0 < ?k"

lemma code_rev_code: "code (rev ` \<C>)"
proof
  fix xs ys assume "xs \<in> lists (rev ` \<C>)" "ys \<in> lists (rev ` \<C>)" "concat xs = concat ys"
  have "map rev (rev xs) \<in> lists \<C>" and "map rev (rev ys) \<in> lists \<C>"
    using rev_in_lists[OF \<open>xs \<in> lists (rev ` \<C>)\<close>] rev_in_lists[OF \<open>ys \<in> lists (rev ` \<C>)\<close>] map_rev_lists_rev
    by (metis imageI)+
  moreover have "concat (map rev (rev xs)) = concat (map rev (rev ys))"
    unfolding rev_concat[symmetric] using \<open>concat xs = concat ys\<close> by blast
  ultimately have "map rev (rev xs) = map rev (rev ys)"
    using is_code by blast
  thus "xs = ys"
     using \<open>concat xs = concat ys\<close> by simp
qed

lemma dec_rev [simp, reversal_rule]:
  "u \<in> \<langle>\<C>\<rangle> \<Longrightarrow> Dec rev ` \<C> (rev u) = rev (map rev (Dec \<C> u))"
  by (auto simp only: rev_map lists_image rev_in_lists rev_concat[symmetric] dec_in_lists
      intro!: code_rev_code code.code_unique_dec imageI del: in_listsI)

lemma elem_comm_sing_set: assumes "ws \<in> lists \<C>" and "ws \<noteq> \<epsilon>" and "u \<in> \<C>" and "concat ws \<cdot> u = u \<cdot> concat ws"
  shows  "set ws = {u}"
  using assms
proof-
  have "concat (ws \<cdot> [u]) = concat ([u] \<cdot> ws)"
    using assms by simp
  have "ws \<cdot> [u] = [u] \<cdot> ws"
    using  \<open>u \<in> \<C>\<close> \<open>ws \<in> lists \<C>\<close> is_code[OF _ _  \<open>concat (ws \<cdot> [u]) = concat ([u] \<cdot> ws)\<close>]
    by simp
  from comm_nemp_pows_posE[OF this \<open>ws \<noteq> \<epsilon>\<close> not_Cons_self2[of u \<epsilon>]]
  obtain t k m where "ws = t\<^sup>@k" "[u] = t\<^sup>@m" "0 < k" "0 < m" "primitive t".
  hence "t = [u]"
    by force
  show "set ws = {u}"
    using sing_pow_set[OF \<open>0 < k\<close>] unfolding \<open>ws = t\<^sup>@k\<close> \<open>t = [u]\<close>.
qed

lemma  pure_code_pres_prim:  assumes pure: "\<forall>u \<in> \<langle>\<C>\<rangle>. \<rho> u \<in> \<langle>\<C>\<rangle>" and
  "w \<in> \<langle>\<C>\<rangle>" and "primitive (Dec \<C> w)"
shows "primitive w"
proof-
  obtain k where "(\<rho> w)\<^sup>@k = w"
    using primroot_expE by blast

  have "\<rho> w \<in> \<langle>\<C>\<rangle>"
    using assms(2) pure by auto

  have "(Dec \<C> (\<rho> w))\<^sup>@k \<in> lists \<C>"
    by (metis \<open>\<rho> w \<in> \<langle>\<C>\<rangle>\<close> concat_sing_pow dec_in_lists flatten_lists order_refl sing_pow_lists)

  have "(Dec \<C> (\<rho> w))\<^sup>@k = Dec \<C> w"
    using \<open>(Dec \<C> (\<rho> w)) \<^sup>@ k \<in> lists \<C>\<close>  code.code_unique_dec code_axioms concat_morph_power \<open>(\<rho> w) \<^sup>@ k = w\<close> concat_dec[OF \<open>\<rho> w \<in> \<langle>\<C>\<rangle>\<close>] by metis
  hence "k = 1"
    using \<open>primitive (Dec \<C> w)\<close> unfolding primitive_def by blast
  thus "primitive w"
    by (metis pow_1 \<open>\<rho> w \<^sup>@ k = w\<close> assms(3) dec_emp prim_nemp primroot_prim)
qed

lemma inj_on_dec: "inj_on (decompose \<C>) \<langle>\<C>\<rangle>"
  by (rule inj_onI) (use concat_dec in force)

end \<comment> \<open>end context code\<close>

lemma emp_is_code: "code {}"
  using code.intro empty_iff insert_iff lists_empty by metis

lemma code_induct_hd: assumes "\<epsilon> \<notin> C" and
  "\<And> xs ys. xs \<in> lists C \<Longrightarrow> ys \<in> lists C \<Longrightarrow> concat xs = concat ys \<Longrightarrow> hd xs = hd ys"
shows "code C"
proof
  show "xs \<in> lists C \<Longrightarrow> ys \<in> lists C \<Longrightarrow> concat xs = concat ys \<Longrightarrow> xs = ys" for xs ys
  proof (induct xs ys rule: list_induct2')
    case (4 x xs y ys)
    from assms(2)[OF "4.prems"]
    have "x = y" by force
    from "4.prems"[unfolded this]
    have "xs \<in> lists C" and "ys \<in> lists C" and "concat xs = concat ys"
      by simp_all
    from "4.hyps"[OF this] \<open>x = y\<close>
    show ?case
      by simp
  qed (auto simp add: \<open>\<epsilon> \<notin> C\<close>)
qed

lemma ref_set_primroot: assumes "ws \<in> lists (G - {\<epsilon>})" and "code (\<rho>`G)"
  shows "set (Ref \<rho>`G ws) = \<rho>`(set ws)"
proof-
  have "G \<subseteq> \<langle>\<rho>`G\<rangle>"
  proof
    fix x
    assume "x \<in> G"
    show "x \<in> \<langle>\<rho> ` G\<rangle>"
      by (metis \<open>x \<in> G\<close> genset_sub image_subset_iff power_in primroot_expE)
  qed
  hence "ws \<in> lists \<langle>\<rho>`G\<rangle>"
    using assms by blast

  have "set (decompose (\<rho>`G) a) = {\<rho> a}" if "a \<in> set ws" for a
  proof-
    have "\<rho> a \<in> \<rho>`G"
      using \<open>a \<in> set ws\<close> \<open>ws \<in> lists (G - {\<epsilon>})\<close> by blast
    have "(Dec (\<rho>`G) a) \<in> [\<rho> a]*"
      using code.code_unique_dec[OF \<open>code (\<rho> ` G)\<close> sing_pow_lists concat_sing_pow, OF \<open>\<rho> a \<in> \<rho> ` G\<close>]
        primroot_expE rootI by metis
    from sing_pow_set'[OF this dec_nemp']
    show "set (decompose (\<rho>`G) a) = {\<rho> a}"
      using \<open>a \<in> set ws\<close> \<open>ws \<in> lists \<langle>\<rho> ` G\<rangle>\<close> \<open>ws \<in> lists (G - {\<epsilon>})\<close> by blast
  qed

  have "(set`(decompose (\<rho>`G))`set ws) = {{\<rho> a} |a. a \<in> set ws}" (is "?L = ?R")
  proof
    show "?L \<subseteq> ?R"
      using \<open>\<And>a. a \<in> set ws \<Longrightarrow> set (Dec \<rho> ` G a) = {\<rho> a}\<close> by blast
    show "?R \<subseteq> ?L"
      using \<open>\<And>a. a \<in> set ws \<Longrightarrow> set (Dec \<rho> ` G a) = {\<rho> a}\<close> by blast
  qed

  show ?thesis
    using ref_set[OF \<open>ws \<in> lists \<langle>\<rho>`G\<rangle>\<close>]
      Setcompr_eq_image \<open>set ` decompose (\<rho> ` G) ` set ws = {{\<rho> a} |a. a \<in> set ws}\<close> by (auto simp add: refine_def)
qed


section \<open>Prefix code\<close>

locale pref_code =
  fixes \<C>
  assumes
    emp_not_in: "\<epsilon> \<notin> \<C>" and
    pref_free: "u \<in> \<C> \<Longrightarrow> v \<in> \<C> \<Longrightarrow> u \<le>p v \<Longrightarrow> u = v"

begin

lemma nemp: "u \<in> \<C> \<Longrightarrow> u \<noteq> \<epsilon>"
  using emp_not_in by force

lemma concat_pref_concat:
  assumes "us \<in> lists \<C>" "vs \<in> lists \<C>" "concat us \<le>p concat vs"
  shows "us \<le>p vs"
using assms proof (induction us vs rule: list_induct2')
  case (4 x xs y ys)
  from "4.prems"
  have "x = y"
    by (auto elim!: ruler_prefE intro: pref_free sym del: in_listsD)
  with "4" show "x # xs \<le>p y # ys"
  by simp
qed (simp_all add: nemp)

lemma concat_pref_concat_conv:
  assumes "us \<in> lists \<C>" "vs \<in> lists \<C>"
  shows "concat us \<le>p concat vs \<longleftrightarrow> us \<le>p vs"
using concat_pref_concat[OF assms] pref_concat_pref..

sublocale code
  by standard (simp_all add: pref_antisym concat_pref_concat)

lemmas is_code = is_code and
  code = code_axioms

lemma dec_pref_unique:
  "w \<in> \<langle>\<C>\<rangle> \<Longrightarrow> p \<in> \<langle>\<C>\<rangle> \<Longrightarrow> p \<le>p w \<Longrightarrow> Dec \<C> p \<le>p Dec \<C> w"
  using concat_pref_concat_conv[of "Dec \<C> p" "Dec \<C> w", THEN iffD1]
  by simp

end

subsection \<open>Suffix code\<close>

locale suf_code = pref_code "(rev ` \<C>)" for \<C>
begin

thm dec_rev
    code

sublocale code
  using code_rev_code unfolding rev_rev_image_eq.

lemmas concat_suf_concat = concat_pref_concat[reversed] and
       concat_suf_concat_conv = concat_pref_concat_conv[reversed] and
       nemp = nemp[reversed] and
       suf_free = pref_free[reversed] and
       dec_suf_unique = dec_pref_unique[reversed]

thm is_code
    code_axioms
    code

end

section \<open>Marked code\<close>

locale marked_code =
  fixes \<C>
  assumes
    emp_not_in: "\<epsilon> \<notin> \<C>" and
    marked: "u \<in> \<C> \<Longrightarrow> v \<in> \<C> \<Longrightarrow> hd u = hd v \<Longrightarrow> u = v"

begin

lemma nemp: "u \<in> \<C> \<Longrightarrow> u \<noteq> \<epsilon>"
  using emp_not_in by blast

sublocale pref_code
  by (unfold_locales) (simp_all add: emp_not_in marked nemp pref_hd_eq)


lemma marked_concat_lcp: "us \<in> lists \<C> \<Longrightarrow> vs \<in> lists \<C> \<Longrightarrow> concat (us \<and>\<^sub>p vs) = (concat us) \<and>\<^sub>p (concat vs)"
proof (induct us vs rule: list_induct2')
  case (4 x xs y ys)
  hence "x \<in> \<C>" and "y \<in> \<C>" and "xs \<in> lists \<C>" and "ys \<in> lists \<C>"
    by simp_all
  show ?case
  proof (cases)
    assume "x = y"
    thus "concat (x # xs \<and>\<^sub>p y # ys) = concat (x # xs) \<and>\<^sub>p concat (y # ys)"
      using "4.hyps"[OF \<open>xs \<in> lists \<C>\<close> \<open>ys \<in> lists \<C>\<close>] by (simp add: lcp_ext_left)
  next
    assume "x \<noteq> y"
    with marked[OF \<open>x \<in> \<C>\<close> \<open>y \<in> \<C>\<close>] have "hd x \<noteq> hd y" by blast
    hence "concat (x # xs) \<and>\<^sub>p concat (y # ys) = \<epsilon>"
      by (simp add: \<open>x \<in> \<C>\<close> \<open>y \<in> \<C>\<close> nemp lcp_distinct_hd)
    moreover have "concat (x # xs \<and>\<^sub>p y # ys) = \<epsilon>"
      using \<open>x \<noteq> y\<close> by simp
    ultimately show ?thesis by presburger
  qed
qed simp_all

lemma hd_concat_hd: assumes "xs \<in> lists \<C>" and "ys \<in> lists \<C>" and "xs \<noteq> \<epsilon>" and "ys \<noteq> \<epsilon>" and
  "hd (concat xs) = hd (concat ys)"
shows "hd xs = hd ys"
proof-
  have "hd (hd xs) = hd (hd ys)"
    using assms  hd_concat[OF \<open>xs \<noteq> \<epsilon>\<close> lists_hd_in_set[THEN nemp]] hd_concat[OF \<open>ys \<noteq> \<epsilon>\<close> lists_hd_in_set[THEN nemp]]
    by presburger

  from marked[OF lists_hd_in_set lists_hd_in_set this] assms(1-4)
  show "hd xs = hd ys"
    by simp
qed

end

section "Non-overlapping code"

locale non_overlapping =
  fixes \<C>
  assumes
    emp_not_in: "\<epsilon> \<notin> \<C>" and
    no_overlap: "u \<in> \<C> \<Longrightarrow> v \<in> \<C> \<Longrightarrow> z \<le>p u \<Longrightarrow> z \<le>s v \<Longrightarrow> z \<noteq> \<epsilon> \<Longrightarrow> u = v" and
    no_fac: "u \<in> \<C> \<Longrightarrow> v \<in> \<C> \<Longrightarrow> u \<le>f v \<Longrightarrow>  u = v"
begin

lemma nemp: "u \<in> \<C>  \<Longrightarrow> u \<noteq> \<epsilon>"
  using emp_not_in by force

sublocale pref_code
  using nemp non_overlapping.no_fac non_overlapping_axioms pref_code.intro by fastforce

lemma rev_non_overlapping: "non_overlapping (rev ` \<C>)"
proof
  show "\<epsilon> \<notin> rev ` \<C>"
    using nemp by force
  show "u \<in> rev ` \<C> \<Longrightarrow> v \<in> rev ` \<C> \<Longrightarrow> z \<le>p u \<Longrightarrow> z \<le>s v \<Longrightarrow> z \<noteq> \<epsilon> \<Longrightarrow> u = v" for u v z
    using no_overlap[reversed] unfolding rev_in_conv..
  show "u \<in> rev ` \<C> \<Longrightarrow> v \<in> rev ` \<C> \<Longrightarrow> u \<le>f v \<Longrightarrow> u = v" for u v
    using no_fac[reversed] unfolding rev_in_conv by presburger
qed

sublocale suf: suf_code \<C>
proof-
  interpret i: non_overlapping "rev ` \<C>"
    using rev_non_overlapping.
  from i.pref_code_axioms
  show "suf_code \<C>"
    by unfold_locales
qed

lemma overlap_concat_last: assumes  "u \<in> \<C>" and "vs \<in> lists \<C>" and "vs \<noteq> \<epsilon>" and
      "r \<noteq> \<epsilon>" and "r \<le>p u" and  "r \<le>s p \<cdot> concat vs"
  shows  "u = last vs"
proof-
  from suffix_same_cases[OF suf_ext[OF concat_last_suf[OF \<open>vs \<noteq> \<epsilon>\<close>]] \<open>r \<le>s p \<cdot> concat vs\<close>]
  show "u = last vs"
  proof (rule disjE)
    assume "r \<le>s last vs"
    from no_overlap[OF \<open>u \<in> \<C>\<close> _ \<open>r \<le>p u\<close> this \<open>r \<noteq> \<epsilon>\<close>]
    show "u = last vs"
      using \<open>vs \<in> lists \<C>\<close> \<open>vs \<noteq> \<epsilon>\<close> by force
  next
    assume "last vs \<le>s r"
    from no_fac[OF _ \<open>u \<in> \<C>\<close> pref_suf_fac, OF _ \<open>r \<le>p u\<close>  this]
    show "u = last vs"
      using \<open>vs \<in> lists \<C>\<close> \<open>vs \<noteq> \<epsilon>\<close> by force
  qed
qed

lemma overlap_concat_hd: assumes  "u \<in> \<C>" and "vs \<in> lists \<C>" and "vs \<noteq> \<epsilon>" and "r \<noteq> \<epsilon>" and "r \<le>s u" and  "r \<le>p concat vs \<cdot> s"
  shows  "u = hd vs"
proof-
  interpret c: non_overlapping "rev ` \<C>" by (simp add: rev_non_overlapping)
  from c.overlap_concat_last[reversed, OF assms]
  show ?thesis.
qed

lemma fac_concat_fac:
  assumes "us \<in> lists \<C>" "vs \<in> lists \<C>"
    and "1 < card (set us)"
    and "concat vs = p \<cdot> concat us \<cdot> s"
  obtains ps ss where  "concat ps = p" and "concat ss = s" and "ps \<cdot> us \<cdot> ss = vs"
proof-
  define us1 where "us1 = takeWhile (\<lambda> a. a = hd us) us"
  define us2 where "us2 = dropWhile (\<lambda> a. a = hd us) us"
  from card_set_decompose[OF \<open>1 < card (set us)\<close>]
  have "us = us1 \<cdot> us2" "us1 \<noteq> \<epsilon>" "us2 \<noteq> \<epsilon>" "set us1 = {hd us}" "last us1 \<noteq> hd us2"
    unfolding us1_def us2_def by simp_all
  have "us2 \<in> lists \<C>" "us1 \<in> lists \<C>"
    using \<open>us = us1 \<cdot> us2\<close> \<open>us \<in> lists \<C>\<close> by simp_all
  hence "concat us2 \<noteq> \<epsilon>"
    using \<open>us2 \<noteq> \<epsilon>\<close> nemp by force
  hence "p \<cdot> concat us1 <p concat vs"
    using \<open>us = us1 \<cdot> us2\<close> unfolding \<open>concat vs = p \<cdot> concat us \<cdot> s\<close> by simp
  from pref_mod_list[OF this]
  obtain j r where "j < \<^bold>|vs\<^bold>|" "r <p vs ! j" "concat (take j vs) \<cdot> r = p \<cdot> concat us1".
  have "r = \<epsilon>"
  proof (rule ccontr)
    assume "r \<noteq> \<epsilon>"
    from spref_exE[OF \<open>r <p vs ! j\<close>]
    obtain z where "r \<cdot> z = vs ! j" "z \<noteq> \<epsilon>".
    from overlap_concat_last[OF _ \<open>us1 \<in> lists \<C>\<close> \<open>us1 \<noteq> \<epsilon>\<close> \<open>r \<noteq> \<epsilon>\<close> sprefD1[OF \<open>r <p vs ! j\<close>] sufI[OF \<open>concat (take j vs) \<cdot> r = p \<cdot> concat us1\<close>]]
    have "vs ! j = last us1"
      using  nth_in_lists[OF \<open>j < \<^bold>|vs\<^bold>|\<close> \<open>vs \<in> lists \<C>\<close>].

    have concat_vs: "concat vs = concat (take j vs) \<cdot> vs!j \<cdot> concat (drop (Suc j) vs)"
      unfolding lassoc concat_take_Suc[OF \<open>j < \<^bold>|vs\<^bold>|\<close>] concat_morph[symmetric] by force
    from this[folded \<open>r \<cdot> z = vs ! j\<close>]
    have "z \<cdot> concat (drop (Suc j) vs) = concat us2 \<cdot> s"
      unfolding \<open>concat vs = p \<cdot> concat us \<cdot> s\<close> lassoc \<open>concat (take j vs) \<cdot> r = p \<cdot> concat us1\<close>            \<open>us = us1 \<cdot> us2\<close> concat_morph
      unfolding rassoc cancel by simp
    from overlap_concat_hd[OF _ \<open>us2 \<in> lists \<C>\<close> \<open>us2 \<noteq> \<epsilon>\<close> \<open>z \<noteq> \<epsilon>\<close> sufI[OF \<open>r \<cdot> z = vs ! j\<close>]  prefI[OF this]]
    have "vs ! j = hd us2"
      using  nth_in_lists[OF \<open>j < \<^bold>|vs\<^bold>|\<close> \<open>vs \<in> lists \<C>\<close>].

    thus False
      unfolding \<open>vs ! j = last us1\<close> using  \<open>last us1 \<noteq> hd us2\<close> by contradiction
  qed

  have "drop j vs \<in> lists \<C>" and "take j vs \<in> lists \<C>"
    using \<open>vs \<in> lists \<C>\<close> by inlists
  have "concat us2 \<cdot> s = concat (drop j vs)"
    using arg_cong[OF takedrop[of j vs], of concat] \<open>concat (take j vs) \<cdot> r = p \<cdot> concat us1\<close>
    unfolding \<open>concat vs = p \<cdot> concat us \<cdot> s\<close> concat_morph \<open>r = \<epsilon>\<close> emp_simps \<open>us = us1 \<cdot> us2\<close> by auto
  from prefI[OF this]
  have "us2 \<le>p drop j vs"
    using concat_pref_concat_conv[OF \<open>us2 \<in> lists \<C>\<close> \<open>drop j vs \<in> lists \<C>\<close>] by blast
  hence s: "concat (us2\<inverse>\<^sup>>drop j vs) = s"
    using \<open>concat us2 \<cdot> s = concat (drop j vs)\<close> concat_morph_lq lqI by blast

  from \<open>concat (take j vs) \<cdot> r = p \<cdot> concat us1\<close>[unfolded \<open>r = \<epsilon>\<close> emp_simps]
  have "concat us1 \<le>s concat (take j vs)"
    by fastforce
  hence "us1 \<le>s take j vs"
    using suf.concat_pref_concat_conv[reversed, OF \<open>us1 \<in> lists \<C>\<close> \<open>take j vs \<in> lists \<C>\<close>] by blast
  from arg_cong[OF rq_suf[OF this], of concat, unfolded concat_morph]
  have p: "concat (take j vs\<^sup><\<inverse>us1 ) = p"
    using rqI[OF \<open>concat (take j vs) = p \<cdot> concat us1\<close>[symmetric]]
    rq_triv by metis

  have "take j vs\<^sup><\<inverse>us1  \<cdot> us \<cdot> us2\<inverse>\<^sup>>drop j vs = vs"
    unfolding \<open>us = us1 \<cdot> us2\<close> rassoc lq_pref[OF \<open>us2 \<le>p drop j vs\<close>]
    unfolding lassoc  rq_suf[OF \<open>us1 \<le>s take j vs\<close>] by simp

  from that[OF p s this]
  show thesis.
qed

theorem prim_morph:
  assumes "ws \<in> lists \<C>"
    and "\<^bold>|ws\<^bold>| \<noteq> 1"
    and "primitive ws"
  shows "primitive (concat ws)"
proof (rule ccontr)
  have "ws \<in> lists \<C>" and "ws \<cdot> ws \<in> lists \<C>"
    using \<open>ws \<in> lists \<C>\<close> by simp_all
  moreover have "1 < card (set ws)" using \<open>primitive ws\<close> \<open>\<^bold>|ws\<^bold>| \<noteq> 1\<close>
    by (rule prim_card_set)
  moreover assume "\<not> primitive (concat ws)"
  then obtain k z where "2 \<le> k" and "z \<^sup>@ k = concat ws" by (elim not_prim_primroot_expE)
  have "concat (ws \<cdot> ws) = z \<cdot> concat ws \<cdot> z\<^sup>@(k-1)"
    unfolding concat_morph \<open>z \<^sup>@ k = concat ws\<close>[symmetric] add_exps[symmetric] pow_Suc[symmetric]
    using \<open>2 \<le> k\<close> by simp
  ultimately obtain ps ss where "concat ps = z" and "concat ss = z\<^sup>@(k-1)" and  "ps \<cdot> ws \<cdot> ss = ws \<cdot> ws"
    by (rule fac_concat_fac)
  have "ps \<^sup>@ k \<in> lists \<C>"
    using \<open>ps \<cdot> ws \<cdot> ss = ws \<cdot> ws\<close> \<open>ws \<cdot> ws \<in> lists \<C>\<close> by inlists
  moreover have "concat (ps \<^sup>@ k) = concat ws"
    unfolding concat_pow \<open>concat ps = z\<close> \<open>z \<^sup>@ k = concat ws\<close>..
  ultimately have "ps \<^sup>@ k = ws" using \<open>ws \<in> lists \<C>\<close> by (intro is_code)
  show False
    using prim_exp_one[OF \<open>primitive ws\<close> \<open>ps \<^sup>@ k = ws\<close>] \<open>2 \<le> k\<close> by presburger
qed

lemma prim_concat_conv:
  assumes "ws \<in> lists \<C>"
    and "\<^bold>|ws\<^bold>| \<noteq> 1"
  shows "primitive (concat ws) \<longleftrightarrow> primitive ws"
  using prim_concat_prim prim_morph[OF assms]..

end

lemma (in code) code_roots_non_overlapping: "non_overlapping ((\<lambda> x. [\<rho> x]\<^sup>@(e\<^sub>\<rho> x)) ` \<C>)"
proof
  show "\<epsilon> \<notin> (\<lambda>x. [\<rho> x] \<^sup>@ e\<^sub>\<rho> x) ` \<C>"
  proof
    assume "\<epsilon> \<in> (\<lambda>x. [\<rho> x] \<^sup>@ e\<^sub>\<rho> x) ` \<C> "
    from this[unfolded image_iff]
    obtain u where "u \<in> \<C>" and "\<epsilon> = [\<rho> u] \<^sup>@ e\<^sub>\<rho> u"
      by blast
    from arg_cong[OF this(2), of concat]
    show False
      unfolding concat.simps(1) concat_sing_pow primroot_exp_eq
      using emp_not_in \<open>u \<in> \<C>\<close> by blast
  qed
  fix us vs
  assume us': "us \<in> (\<lambda>x. [\<rho> x] \<^sup>@ e\<^sub>\<rho> x) ` \<C>" and vs': "vs \<in> (\<lambda>x. [\<rho> x] \<^sup>@ e\<^sub>\<rho> x) ` \<C>"
    from us'[unfolded image_iff]
    obtain u where "u \<in> \<C>" and us: "us = [\<rho> u] \<^sup>@ e\<^sub>\<rho> u"
      by blast
    from vs'[unfolded image_iff]
    obtain v where "v \<in> \<C>" and vs: "vs = [\<rho> v] \<^sup>@ e\<^sub>\<rho> v"
      by blast
  note sing_set = sing_pow_set[OF primroot_exp_nemp[OF nemp]]
  show "us = vs" if "zs \<le>p us" and "zs \<le>s vs" and "zs \<noteq> \<epsilon>" for zs
  proof-
    from set_mono_prefix[OF \<open>zs \<le>p us\<close>] \<open>zs \<noteq> \<epsilon>\<close>[folded set_empty2]
    have "set zs = {\<rho> u}"
      using subset_singletonD  unfolding \<open>us = [\<rho> u] \<^sup>@ e\<^sub>\<rho> u\<close> sing_set[OF \<open>u \<in> \<C>\<close>]
      by metis
    from set_mono_suffix[OF \<open>zs \<le>s vs\<close>] \<open>zs \<noteq> \<epsilon>\<close>[folded set_empty2]
    have "set zs = {\<rho> v}"
      using subset_singletonD  unfolding \<open>vs = [\<rho> v] \<^sup>@ e\<^sub>\<rho> v\<close>  sing_set[OF \<open>v \<in> \<C>\<close>]
      by metis
    hence "\<rho> u = \<rho> v"
      unfolding \<open>set zs = {\<rho> u}\<close> by simp
    from same_primroots_comm[OF this]
    have "u = v"
      using code_not_comm [OF \<open>u \<in> \<C>\<close> \<open>v \<in> \<C>\<close>] by blast
    thus "us = vs"
      unfolding \<open>us = [\<rho> u] \<^sup>@ e\<^sub>\<rho> u\<close> \<open>vs = [\<rho> v] \<^sup>@ e\<^sub>\<rho> v\<close> by blast
  qed
  show "us = vs"  if  "us \<le>f vs"
  proof-
    from sing_set[OF \<open>u \<in> \<C>\<close>, of "\<rho> u"] sing_set[OF \<open>v \<in> \<C>\<close>, of "\<rho> v"]
    have "\<rho> u = \<rho> v"
      unfolding  us[symmetric] vs[symmetric] using set_mono_sublist[OF \<open>us \<le>f vs\<close>]
      by force
    from same_primroots_comm[OF this]
    have "u = v"
      using code_not_comm [OF \<open>u \<in> \<C>\<close> \<open>v \<in> \<C>\<close>] by blast
    thus "us = vs"
      unfolding \<open>us = [\<rho> u] \<^sup>@ e\<^sub>\<rho> u\<close> \<open>vs = [\<rho> v] \<^sup>@ e\<^sub>\<rho> v\<close> by blast
  qed
qed

theorem (in code) roots_prim_morph:
  assumes "ws \<in> lists \<C>"
    and "\<^bold>|ws\<^bold>| \<noteq> 1"
    and "primitive ws"
  shows "primitive (concat (map (\<lambda> x. [\<rho> x]\<^sup>@(e\<^sub>\<rho> x)) ws))"
    (is "primitive (concat (map ?R ws))")
proof-
  interpret rc: non_overlapping "?R ` \<C>"
    using code_roots_non_overlapping.

  show ?thesis
  proof (rule rc.prim_morph)
    show "primitive (map ?R ws)"
      using  inj_map_prim[OF root_dec_inj_on
          \<open>ws \<in> lists \<C>\<close> \<open>primitive ws\<close>].
    show "map ?R ws \<in> lists (?R ` \<C>)"
      using \<open>ws \<in> lists \<C>\<close> lists_image[of ?R \<C>] by force
    show "\<^bold>|map (\<lambda>x. [\<rho> x] \<^sup>@ e\<^sub>\<rho> x) ws\<^bold>| \<noteq> 1"
      using \<open>\<^bold>|ws\<^bold>| \<noteq> 1\<close> by simp
  qed
qed

section \<open>Binary code\<close>

text\<open>We pay a special attention to two element codes.
In particular, we show that two words form a code if and only if they do not commute. This means that two
words either commute, or do not satisfy any nontrivial relation.
\<close>

definition  bin_lcp  where "bin_lcp x y  =  x\<cdot>y \<and>\<^sub>p y\<cdot>x"
definition  bin_lcs  where "bin_lcs x y  =  x\<cdot>y \<and>\<^sub>s y\<cdot>x"

definition  bin_mismatch where "bin_mismatch x y =  (x\<cdot>y)!\<^bold>|bin_lcp x y\<^bold>|"
definition  bin_mismatch_suf where " bin_mismatch_suf x y = bin_mismatch (rev y) (rev x)"

value[nbe] "[0::nat,1,0]!3"

lemma bin_lcs_rev: "bin_lcs x y = rev (bin_lcp (rev x) (rev y))"
  unfolding bin_lcp_def bin_lcs_def  longest_common_suffix_def rev_append using lcp_sym by fastforce

lemma bin_lcp_sym: "bin_lcp x y = bin_lcp y x"
  unfolding bin_lcp_def using lcp_sym.

lemma bin_mismatch_comm: "(bin_mismatch x y = bin_mismatch y x) \<longleftrightarrow> (x \<cdot> y = y \<cdot> x)"
  unfolding bin_mismatch_def bin_lcp_def lcp_sym[of "y \<cdot> x"]
  using  lcp_mismatch'[of "x \<cdot> y" "y \<cdot> x", unfolded comm_comp_eq_conv[of x y]]   by fastforce

lemma bin_lcp_pref_fst_snd: "bin_lcp x y \<le>p x \<cdot> y"
  unfolding bin_lcp_def using lcp_pref.

lemma bin_lcp_pref_snd_fst: "bin_lcp x y \<le>p y \<cdot> x"
  using bin_lcp_pref_fst_snd[of y x, unfolded bin_lcp_sym[of y x]].

lemma bin_lcp_bin_lcs [reversal_rule]:  "bin_lcp (rev x) (rev y) = rev (bin_lcs x y)"
  unfolding bin_lcp_def bin_lcs_def rev_append[symmetric] lcs_lcp
    lcs_sym[of "x \<cdot> y"]..

lemmas bin_lcs_sym = bin_lcp_sym[reversed]

lemma bin_lcp_len: "x \<cdot> y \<noteq> y \<cdot> x \<Longrightarrow> \<^bold>|bin_lcp x y\<^bold>| < \<^bold>|x \<cdot> y\<^bold>|"
  unfolding bin_lcp_def
  using lcp_len' pref_comm_eq by blast

lemmas bin_lcs_len = bin_lcp_len[reversed]

lemma bin_mismatch_pref_suf'[reversal_rule]:
  "bin_mismatch (rev y) (rev x) =  bin_mismatch_suf x y"
  unfolding bin_mismatch_suf_def..

subsection \<open>Binary code locale\<close>

locale binary_code =
  fixes u\<^sub>0 u\<^sub>1
  assumes non_comm: "u\<^sub>0 \<cdot> u\<^sub>1 \<noteq> u\<^sub>1 \<cdot> u\<^sub>0"

begin

text\<open>A crucial property of two element codes is the constant decoding delay given by the word $\alpha$,
which is a prefix of any generating word (sufficiently long), while the letter
immediately after this common prefix indicates the first element of the decomposition.
\<close>

definition uu where "uu a = (if a then u\<^sub>0 else u\<^sub>1)"

lemma bin_code_set_bool: "{uu a,uu (\<not> a)} = {u\<^sub>0,u\<^sub>1}"
  by (induct a, unfold uu_def, simp_all add: insert_commute)

lemma bin_code_set_bool': "{uu a,uu (\<not> a)} = {u\<^sub>1,u\<^sub>0}"
  by (induct a, unfold uu_def, simp_all add: insert_commute)

lemma bin_code_swap: "binary_code u\<^sub>1 u\<^sub>0"
  using binary_code.intro[OF non_comm[symmetric]].

lemma bin_code_bool: "binary_code (uu a)  (uu (\<not> a))"
  unfolding uu_def by (induct a, simp_all add: bin_code_swap binary_code_axioms)

lemma bin_code_neq: "u\<^sub>0 \<noteq> u\<^sub>1"
  using non_comm by auto

lemma bin_code_neq_bool: "uu a \<noteq> uu (\<not> a)"
  unfolding uu_def by (induct a) (use bin_code_neq in fastforce)+

lemma bin_fst_nemp: "u\<^sub>0 \<noteq> \<epsilon>" and bin_snd_nemp: "u\<^sub>1 \<noteq> \<epsilon>" and bin_nemp_bool: "uu a \<noteq> \<epsilon>"
  using non_comm uu_def by auto

lemma bin_not_comp: "\<not> u\<^sub>0 \<cdot> u\<^sub>1 \<bowtie> u\<^sub>1 \<cdot> u\<^sub>0"
  using comm_comp_eq_conv non_comm by blast

lemma bin_not_comp_bool: "\<not> (uu a \<cdot> uu (\<not> a) \<bowtie> uu (\<not> a) \<cdot> uu a)"
  unfolding uu_def by (induct a, use bin_not_comp pref_comp_sym in auto)

lemma bin_not_comp_suf: "\<not> u\<^sub>0 \<cdot> u\<^sub>1 \<bowtie>\<^sub>s u\<^sub>1 \<cdot> u\<^sub>0"
  using comm_comp_eq_conv_suf non_comm[reversed] by blast

lemma bin_not_comp_suf_bool: "\<not> (uu a \<cdot> uu (\<not> a) \<bowtie>\<^sub>s uu (\<not> a) \<cdot> uu a)"
  unfolding uu_def by (induct a, use bin_not_comp_suf suf_comp_sym in auto)

lemma bin_mismatch_neq: "bin_mismatch u\<^sub>0 u\<^sub>1 \<noteq> bin_mismatch u\<^sub>1 u\<^sub>0"
  using non_comm[folded bin_mismatch_comm].

abbreviation bin_code_lcp (\<open>\<alpha>\<close>) where  "bin_code_lcp \<equiv> bin_lcp u\<^sub>0 u\<^sub>1"
abbreviation bin_code_lcs where "bin_code_lcs \<equiv> bin_lcs u\<^sub>0 u\<^sub>1"
abbreviation bin_code_mismatch_fst (\<open>c\<^sub>0\<close>) where "bin_code_mismatch_fst \<equiv> bin_mismatch u\<^sub>0 u\<^sub>1"
abbreviation bin_code_mismatch_snd (\<open>c\<^sub>1\<close>) where "bin_code_mismatch_snd \<equiv> bin_mismatch u\<^sub>1 u\<^sub>0"

definition cc where "cc a = (if a then c\<^sub>0 else c\<^sub>1)"


lemmas bin_lcp_swap = bin_lcp_sym[of u\<^sub>0 u\<^sub>1, symmetric] and
       bin_lcp_pref = bin_lcp_pref_fst_snd[of u\<^sub>0 u\<^sub>1] and
       bin_lcp_pref' = bin_lcp_pref_snd_fst[of u\<^sub>0 u\<^sub>1] and
       bin_lcp_short = bin_lcp_len[OF non_comm, unfolded lenmorph]

lemmas bin_code_simps = cc_def uu_def if_True if_False bool_simps

lemma bin_lcp_bool: "bin_lcp (uu a) (uu (\<not> a)) = bin_code_lcp"
  unfolding uu_def by (induct a, simp_all add: bin_lcp_swap)

lemma bin_lcp_spref: "\<alpha> <p u\<^sub>0 \<cdot> u\<^sub>1"
  using bin_lcp_pref bin_lcp_pref' bin_not_comp by fastforce

lemma bin_lcp_spref': "\<alpha> <p u\<^sub>1 \<cdot> u\<^sub>0"
  using bin_lcp_pref bin_lcp_pref' bin_not_comp by fastforce

lemma bin_lcp_spref_bool: "\<alpha> <p uu a \<cdot> uu (\<not> a)"
  unfolding uu_def by (induct a, use  bin_lcp_spref bin_lcp_spref' in auto)

lemma bin_mismatch_bool': "\<alpha> \<cdot> [cc a] \<le>p uu a \<cdot> uu (\<not> a)"
  using add_nth_pref[OF bin_lcp_spref_bool, of a]
  unfolding uu_def cc_def bin_mismatch_def bin_lcp_bool bin_lcp_swap
  by (induct a) simp_all

lemma bin_mismatch_bool: "\<alpha> \<cdot> [cc a] \<le>p uu a \<cdot> \<alpha>"
proof-
  from bin_mismatch_bool'
  have "\<alpha> \<cdot> [cc a] \<le>p uu a \<cdot> (uu (\<not> a) \<cdot> uu a)"
    using pref_prolong by blast
  from pref_prod_pref_short[OF this bin_lcp_pref_snd_fst, unfolded bin_lcp_bool lenmorph sing_len]
  show ?thesis
    using nemp_len[OF bin_nemp_bool, of a] by linarith
qed

lemmas bin_fst_mismatch = bin_mismatch_bool[of True, unfolded bin_code_simps] and
       bin_fst_mismatch' = bin_mismatch_bool'[of True, unfolded bin_code_simps] and
       bin_snd_mismatch = bin_mismatch_bool[of False, unfolded bin_code_simps] and
       bin_snd_mismatch' = bin_mismatch_bool'[of False, unfolded bin_code_simps]

lemma bin_lcp_pref_all: "xs \<in> lists {u\<^sub>0,u\<^sub>1} \<Longrightarrow> \<alpha> \<le>p concat xs \<cdot> \<alpha>"
proof (induct xs)
  case (Cons a xs)
  have "a \<in> {u\<^sub>0,u\<^sub>1}" and "xs \<in> lists {u\<^sub>0, u\<^sub>1}"
    using \<open>a # xs \<in> lists {u\<^sub>0, u\<^sub>1}\<close> by simp_all
  show ?case
  proof (rule two_elemP[OF \<open>a \<in> {u\<^sub>0,u\<^sub>1}\<close>], simp_all)
    show "\<alpha> \<le>p u\<^sub>0 \<cdot> concat xs \<cdot> \<alpha>"
      using pref_extD[OF bin_fst_mismatch] Cons.hyps[OF \<open>xs \<in> lists {u\<^sub>0, u\<^sub>1}\<close>] pref_prolong by blast
  next
    show "\<alpha> \<le>p u\<^sub>1 \<cdot> concat xs \<cdot> \<alpha>"
      using pref_extD[OF bin_snd_mismatch] Cons.hyps[OF \<open>xs \<in> lists {u\<^sub>0, u\<^sub>1}\<close>] pref_prolong by blast
  qed
qed simp

lemma bin_lcp_pref_all_hull: "w \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle> \<Longrightarrow> \<alpha> \<le>p w \<cdot> \<alpha>"
  using bin_lcp_pref_all using hull_concat_listsE by metis

lemma bin_lcp_mismatch_pref_all_bool: assumes "q \<le>p w" and "w \<in> \<langle>{uu b,uu (\<not> b)}\<rangle>" and "\<^bold>|\<alpha>\<^bold>| < \<^bold>|uu a \<cdot> q\<^bold>|"
  shows "\<alpha> \<cdot> [cc a] \<le>p uu a \<cdot> q"
proof-
  have aux: "uu a \<cdot> w \<cdot> \<alpha> = (uu a \<cdot> q) \<cdot> (q\<inverse>\<^sup>>w \<cdot> \<alpha>)" "{uu b,uu (\<not> b)} = {u\<^sub>0,u\<^sub>1}"
    using lq_pref[OF \<open>q \<le>p w\<close>] bin_code_set_bool by force+
  have "\<^bold>|\<alpha> \<cdot> [cc a]\<^bold>| \<le> \<^bold>|uu a \<cdot> q\<^bold>|"
    using \<open>\<^bold>|\<alpha>\<^bold>| < \<^bold>|uu a \<cdot> q\<^bold>|\<close> by auto
  thus ?thesis
    using pref_prolong[OF bin_mismatch_bool  bin_lcp_pref_all_hull[OF \<open>w \<in> \<langle>{uu b,uu (\<not> b)}\<rangle>\<close>[unfolded aux]], of a]
    unfolding aux by blast
qed

lemmas bin_lcp_mismatch_pref_all_fst = bin_lcp_mismatch_pref_all_bool[of _ _ True True, unfolded bin_code_simps] and
       bin_lcp_mismatch_pref_all_snd = bin_lcp_mismatch_pref_all_bool[of _ _ True False, unfolded bin_code_simps]

lemma bin_lcp_pref_all_len: assumes "q \<le>p w" and "w \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>" and "\<^bold>|\<alpha>\<^bold>| \<le> \<^bold>|q\<^bold>|"
  shows "\<alpha> \<le>p q"
  using bin_lcp_pref_all_hull[OF \<open>w \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close>] pref_ext[OF \<open>q \<le>p w\<close>] prefix_length_prefix[OF _ _ \<open>\<^bold>|bin_code_lcp\<^bold>| \<le> \<^bold>|q\<^bold>|\<close>] by blast

lemma bin_mismatch_all_bool: assumes "xs \<in> lists {uu b, uu (\<not> b)}" shows "\<alpha> \<cdot> [cc a] \<le>p (uu a) \<cdot> concat xs \<cdot> \<alpha>"
  using pref_prolong[OF bin_mismatch_bool bin_lcp_pref_all, of xs a] assms unfolding  bin_code_set_bool[of b].

lemmas bin_fst_mismatch_all = bin_mismatch_all_bool[of _ True True, unfolded bin_code_simps] and
       bin_snd_mismatch_all = bin_mismatch_all_bool[of _ True False, unfolded bin_code_simps]

lemma bin_mismatch_all_hull_bool: assumes "w \<in> \<langle>{uu b,uu (\<not> b)}\<rangle>" shows "\<alpha> \<cdot> [cc a] \<le>p uu a \<cdot> w \<cdot> \<alpha>"
  using bin_mismatch_all_bool hull_concat_listsE[OF assms] by metis

lemmas bin_fst_mismatch_all_hull = bin_mismatch_all_hull_bool[of _ True True, unfolded bin_code_simps] and
       bin_snd_mismatch_all_hull = bin_mismatch_all_hull_bool[of _ True False, unfolded bin_code_simps]

lemma bin_mismatch_all_len_bool: assumes "q \<le>p uu a \<cdot> w" and "w \<in> \<langle>{uu b,uu (\<not> b)}\<rangle>" and "\<^bold>|\<alpha>\<^bold>| < \<^bold>|q\<^bold>|"
  shows "\<alpha> \<cdot> [cc a] \<le>p q"
proof-
  have "\<^bold>|\<alpha> \<cdot> [cc a]\<^bold>| \<le> \<^bold>|uu a \<cdot> w\<^bold>|" "\<^bold>|\<alpha> \<cdot> [cc a]\<^bold>| \<le> \<^bold>|q\<^bold>|"
    using less_le_trans[OF \<open>\<^bold>|\<alpha>\<^bold>| < \<^bold>|q\<^bold>|\<close> pref_len[OF \<open>q \<le>p uu a \<cdot> w\<close>]] \<open>\<^bold>|\<alpha>\<^bold>| < \<^bold>|q\<^bold>|\<close> by force+
  from pref_prod_le[OF bin_mismatch_all_hull_bool[OF assms(2), unfolded lassoc], OF this(1)]
  show ?thesis
    by (rule prefix_length_prefix) fact+
qed

lemmas bin_fst_mismatch_all_len = bin_mismatch_all_len_bool[of _ True _ True, unfolded bin_code_simps] and
       bin_snd_mismatch_all_len = bin_mismatch_all_len_bool[of _ False _ True, unfolded bin_code_simps]

lemma bin_code_delay: assumes "\<^bold>|\<alpha>\<^bold>| \<le> \<^bold>|q\<^sub>0\<^bold>|" and "\<^bold>|\<alpha>\<^bold>| \<le> \<^bold>|q\<^sub>1\<^bold>|" and
      "q\<^sub>0 \<le>p u\<^sub>0 \<cdot> w\<^sub>0" and "q\<^sub>1 \<le>p u\<^sub>1 \<cdot> w\<^sub>1" and
      "w\<^sub>0 \<in> \<langle>{u\<^sub>0, u\<^sub>1}\<rangle>" and "w\<^sub>1 \<in> \<langle>{u\<^sub>0, u\<^sub>1}\<rangle>"
  shows "q\<^sub>0 \<and>\<^sub>p q\<^sub>1 = \<alpha>"
proof-
  have p1: "\<alpha> \<cdot> [c\<^sub>0] \<le>p u\<^sub>0 \<cdot> w\<^sub>0 \<cdot> \<alpha>"
    using assms(5) using bin_fst_mismatch_all_hull by auto
  have p2: "\<alpha> \<cdot> [c\<^sub>1] \<le>p u\<^sub>1 \<cdot> w\<^sub>1 \<cdot> \<alpha>"
    using assms(6) using bin_snd_mismatch_all_hull by auto
  have lcp: "u\<^sub>0 \<cdot> w\<^sub>0 \<cdot> \<alpha> \<and>\<^sub>p u\<^sub>1 \<cdot> w\<^sub>1 \<cdot> \<alpha> = \<alpha>"
    using lcp_first_mismatch_pref[OF p1 p2 bin_mismatch_neq].
  from lcp_extend_eq[of "q\<^sub>0" "u\<^sub>0 \<cdot> w\<^sub>0 \<cdot> \<alpha>" "q\<^sub>1" "u\<^sub>1 \<cdot> w\<^sub>1 \<cdot> \<alpha>",
       unfolded this,OF _ _ assms(1-2)]
  show ?thesis
    using pref_ext[OF \<open>q\<^sub>0 \<le>p u\<^sub>0 \<cdot> w\<^sub>0\<close>] pref_ext[OF \<open>q\<^sub>1 \<le>p u\<^sub>1 \<cdot> w\<^sub>1\<close>] by force
qed

lemma hd_lq_mismatch_fst: "hd (\<alpha>\<inverse>\<^sup>>(u\<^sub>0 \<cdot> \<alpha>)) = c\<^sub>0"
  using hd_lq_conv_nth[OF prefix_snocD[OF bin_fst_mismatch]] bin_fst_mismatch
  by (auto simp add: prefix_def)

lemma hd_lq_mismatch_snd: "hd (\<alpha>\<inverse>\<^sup>>(u\<^sub>1 \<cdot> \<alpha>)) = c\<^sub>1"
  using hd_lq_conv_nth[OF prefix_snocD[OF bin_snd_mismatch]] bin_snd_mismatch
  by (auto simp add: prefix_def)

lemma hds_bin_mismatch_neq: "hd (\<alpha>\<inverse>\<^sup>>(u\<^sub>0 \<cdot> \<alpha>)) \<noteq> hd (\<alpha>\<inverse>\<^sup>>(u\<^sub>1 \<cdot> \<alpha>))"
  unfolding hd_lq_mismatch_fst hd_lq_mismatch_snd
  using bin_mismatch_neq.

lemma bin_lcp_fst_pow_pref: assumes "0 < k" shows "\<alpha> \<cdot> [c\<^sub>0] \<le>p u\<^sub>0\<^sup>@k \<cdot> u\<^sub>1 \<cdot> z"
  using assms
proof (induct k rule: nat_induct_non_zero)
  case 1
  then show ?case
    unfolding pow_1 using  pref_prolong[OF bin_fst_mismatch' triv_pref].
next
  case (Suc n)
  show ?case
    unfolding pow_Suc rassoc
    by (rule pref_prolong[OF bin_fst_mismatch])
    (use append_prefixD[OF Suc.hyps(2)] in blast)
qed

lemmas bin_lcp_snd_pow_pref = binary_code.bin_lcp_fst_pow_pref[OF bin_code_swap, unfolded bin_lcp_swap]

lemma bin_lcp_fst_lcp: "\<alpha> \<le>p u\<^sub>0 \<cdot> \<alpha>" and bin_lcp_snd_lcp: "\<alpha> \<le>p u\<^sub>1 \<cdot> \<alpha>"
  using pref_extD[OF bin_fst_mismatch]  pref_extD[OF bin_snd_mismatch].

lemma bin_lcp_pref_all_set: assumes "set ws = {u\<^sub>0,u\<^sub>1}"
  shows "\<alpha> \<le>p concat ws"
proof-
  have "ws \<in> lists {u\<^sub>0, u\<^sub>1}"
    using assms by blast
  have "\<^bold>|u\<^sub>0\<^bold>| + \<^bold>|u\<^sub>1\<^bold>| \<le> \<^bold>|concat ws\<^bold>|"
    using assms two_in_set_concat_len[OF bin_code_neq] by simp
  with pref_prod_le[OF bin_lcp_pref_all[OF \<open>ws \<in> lists {u\<^sub>0, u\<^sub>1}\<close>]] bin_lcp_short
  show ?thesis
    by simp
qed

lemma bin_lcp_conjug_morph:
  assumes "u \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>" and "v \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>"
  shows "\<alpha>\<inverse>\<^sup>>(u \<cdot> \<alpha>) \<cdot> \<alpha>\<inverse>\<^sup>>(v \<cdot> \<alpha>) = \<alpha>\<inverse>\<^sup>>((u \<cdot> v) \<cdot> \<alpha>)"
  unfolding lq_reassoc[OF bin_lcp_pref_all_hull[OF \<open>u \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close>]] rassoc
    lq_pref[OF bin_lcp_pref_all_hull[OF \<open>v \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close>]]..

lemma lcp_bin_conjug_prim_iff:
  "set ws = {u\<^sub>0,u\<^sub>1} \<Longrightarrow> primitive (\<alpha>\<inverse>\<^sup>>(concat ws) \<cdot> \<alpha>) \<longleftrightarrow> primitive (concat ws)"
  using conjug_prim_iff[OF root_conjug[OF pref_ext[OF bin_lcp_pref_all_set]], symmetric]
  unfolding lq_reassoc[OF bin_lcp_pref_all_set] by simp

lemma bin_lcp_conjug_inj_on: "inj_on (\<lambda>u. \<alpha>\<inverse>\<^sup>>(u \<cdot> \<alpha>)) \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>"
  unfolding inj_on_def using bin_lcp_pref_all_hull cancel_right lq_pref
  by metis

lemma bin_code_lcp_marked: assumes "us \<in> lists {u\<^sub>0,u\<^sub>1}" and "vs \<in> lists {u\<^sub>0,u\<^sub>1}" and "hd us \<noteq> hd vs"
  shows "concat us \<cdot> \<alpha> \<and>\<^sub>p concat vs \<cdot> \<alpha> = \<alpha>"
proof (cases "us = \<epsilon> \<or> vs = \<epsilon>")
  assume "us = \<epsilon> \<or> vs = \<epsilon>"
  thus ?thesis
    using append_self_conv2 assms(1) assms(2) bin_lcp_pref_all concat.simps(1) lcp_pref_conv lcp_sym by metis
next
  assume "\<not> (us = \<epsilon> \<or> vs = \<epsilon>)" hence "us \<noteq> \<epsilon>" and "vs \<noteq> \<epsilon>" by blast+
  have spec_case: "concat us \<cdot> \<alpha> \<and>\<^sub>p concat vs \<cdot> \<alpha> = \<alpha>" if "us \<in> lists {u\<^sub>0,u\<^sub>1}" and "vs \<in> lists {u\<^sub>0,u\<^sub>1}" and "hd us = u\<^sub>0" and "hd vs = u\<^sub>1" and "us \<noteq> \<epsilon>" and "vs \<noteq> \<epsilon>" for us vs
  proof-
    have "concat us = u\<^sub>0 \<cdot> concat (tl us)"
      unfolding hd_concat_tl[OF \<open>us \<noteq> \<epsilon>\<close>, symmetric] \<open>hd us = u\<^sub>0\<close>..
    from bin_fst_mismatch_all[OF tl_in_lists[OF \<open>us \<in> lists {u\<^sub>0,u\<^sub>1}\<close>], folded rassoc this]
    have pref1: "\<alpha> \<cdot> [c\<^sub>0] \<le>p concat us \<cdot> \<alpha>".
    have "concat vs = u\<^sub>1 \<cdot> concat (tl vs)"
      unfolding hd_concat_tl[OF \<open>vs \<noteq> \<epsilon>\<close>, symmetric] \<open>hd vs = u\<^sub>1\<close>..
    from bin_snd_mismatch_all[OF tl_in_lists[OF \<open>vs \<in> lists {u\<^sub>0,u\<^sub>1}\<close>], folded rassoc this]
    have pref2: "\<alpha> \<cdot> [c\<^sub>1] \<le>p concat vs \<cdot> \<alpha>".
    show ?thesis
      using  lcp_first_mismatch_pref[OF pref1 pref2 bin_mismatch_neq].
  qed
  have "hd us \<in>  {u\<^sub>0,u\<^sub>1}" and "hd vs \<in>  {u\<^sub>0,u\<^sub>1}" using
      lists_hd_in_set[OF \<open>us \<noteq> \<epsilon>\<close> \<open>us \<in> lists {u\<^sub>0, u\<^sub>1}\<close>] lists_hd_in_set[OF \<open>vs \<noteq> \<epsilon>\<close> \<open>vs \<in> lists {u\<^sub>0, u\<^sub>1}\<close>].
  then consider "hd us = u\<^sub>0 \<and> hd vs = u\<^sub>1" | "hd us = u\<^sub>1 \<and> hd vs = u\<^sub>0"
    using \<open>hd us \<noteq> hd vs\<close> by fastforce
  then show ?thesis
    using spec_case[rule_format] \<open>us \<noteq> \<epsilon>\<close> \<open>vs \<noteq> \<epsilon>\<close> assms lcp_sym by metis
qed

\<comment> \<open>ALT PROOF\<close>
lemma  assumes "us \<in> lists {u\<^sub>0,u\<^sub>1}" and "vs \<in> lists {u\<^sub>0,u\<^sub>1}" and "hd us \<noteq> hd vs"
  shows "concat us \<cdot> \<alpha> \<and>\<^sub>p concat vs \<cdot> \<alpha> = \<alpha>"
  using assms
proof (induct us vs rule: list_induct2')
  case (2 x xs)
  show ?case
    using bin_lcp_pref_all[OF \<open>x # xs \<in> lists {u\<^sub>0, u\<^sub>1}\<close>, folded lcp_pref_conv, unfolded lcp_sym[of \<alpha>]] by simp
next
  case (3 y ys)
  show ?case
    using bin_lcp_pref_all[OF \<open>y # ys \<in> lists {u\<^sub>0, u\<^sub>1}\<close>, folded lcp_pref_conv] by simp
next
  case (4 x xs y ys)
  interpret i: binary_code x y
    using "4.prems"(1) "4.prems"(2) "4.prems"(3) non_comm binary_code.intro by auto
  have alph: "{u\<^sub>0,u\<^sub>1} = {x,y}"
    using "4.prems"(1) "4.prems"(2) "4.prems"(3) by auto
  from disjE[OF this[unfolded doubleton_eq_iff]]
  have "i.bin_code_lcp = \<alpha>"
    using i.bin_lcp_swap[symmetric] by blast
  have c0: "i.bin_code_lcp \<cdot> [i.bin_code_mismatch_fst] \<le>p x \<cdot> concat xs \<cdot> i.bin_code_lcp"
    using  i.bin_lcp_pref_all[of xs] \<open>x # xs \<in> lists {u\<^sub>0, u\<^sub>1}\<close>[unfolded Cons_in_lists_iff alph]
      pref_prolong[OF i.bin_fst_mismatch] by blast
  have c1: "i.bin_code_lcp \<cdot> [i.bin_code_mismatch_snd] \<le>p y \<cdot> concat ys \<cdot> i.bin_code_lcp"
    using pref_prolong[OF conjunct2[OF \<open>y # ys \<in> lists {u\<^sub>0, u\<^sub>1}\<close>[unfolded      Cons_in_lists_iff alph],
          THEN i.bin_snd_mismatch_all[of ys]], OF self_pref].
  have "i.bin_code_lcp\<cdot>[i.bin_code_mismatch_fst] \<and>\<^sub>p i.bin_code_lcp\<cdot>[i.bin_code_mismatch_snd] = i.bin_code_lcp"
    by (simp add: i.bin_mismatch_neq lcp_first_mismatch')
  from lcp_rulers[OF c0 c1, unfolded this, unfolded bin_lcp_swap]
  show ?case
    unfolding concat.simps(2) rassoc using i.bin_mismatch_neq
      \<open>i.bin_code_lcp = \<alpha>\<close> by force
qed simp

lemma bin_code_lcp_concat: assumes "us \<in> lists {u\<^sub>0,u\<^sub>1}" and "vs \<in> lists {u\<^sub>0,u\<^sub>1}" and "\<not> us \<bowtie> vs"
  shows "concat us \<cdot> \<alpha> \<and>\<^sub>p concat vs \<cdot> \<alpha> = concat (us \<and>\<^sub>p vs) \<cdot> \<alpha>"
proof-
  obtain us' vs' where us: "(us \<and>\<^sub>p vs) \<cdot> us' = us" and vs: "(us \<and>\<^sub>p vs) \<cdot> vs' = vs" and "us' \<noteq> \<epsilon>" and "vs' \<noteq> \<epsilon>" and "hd us' \<noteq> hd vs'"
    using lcp_mismatchE[OF \<open>\<not> us \<bowtie> vs\<close>].
  have cu: "concat us \<cdot> \<alpha> = concat (us \<and>\<^sub>p vs) \<cdot> concat us' \<cdot> \<alpha>"
    unfolding lassoc concat_morph[symmetric] us..
  have cv: "concat vs \<cdot> \<alpha> = concat (us \<and>\<^sub>p vs) \<cdot> concat vs' \<cdot> \<alpha>"
    unfolding lassoc concat_morph[symmetric] vs..
  have "us' \<in> lists {u\<^sub>0,u\<^sub>1}"
    using \<open>us \<in> lists {u\<^sub>0,u\<^sub>1}\<close> us by inlists
  have "vs' \<in> lists {u\<^sub>0,u\<^sub>1}"
    using \<open>vs \<in> lists {u\<^sub>0,u\<^sub>1}\<close> vs by inlists
  show "concat us \<cdot> \<alpha> \<and>\<^sub>p concat vs \<cdot> \<alpha> = concat (us \<and>\<^sub>p vs) \<cdot> \<alpha>"
    unfolding cu cv
    using bin_code_lcp_marked[OF \<open>us' \<in> lists {u\<^sub>0,u\<^sub>1}\<close> \<open>vs' \<in> lists {u\<^sub>0,u\<^sub>1}\<close> \<open>hd us' \<noteq> hd vs'\<close>]
    unfolding lcp_ext_left by fast
qed

lemma bin_code_lcp_concat': assumes "us \<in> lists {u\<^sub>0,u\<^sub>1}" and "vs \<in> lists {u\<^sub>0,u\<^sub>1}" and "\<not> concat us \<bowtie> concat vs"
  shows "concat us \<and>\<^sub>p concat vs = concat (us \<and>\<^sub>p vs) \<cdot> \<alpha>"
  using bin_code_lcp_concat[OF assms(1-2)] assms(3) lcp_ext_right_conv pref_concat_pref prefix_comparable_def by metis

lemma bin_lcp_pows:  "0 < k \<Longrightarrow> 0 < l \<Longrightarrow> u\<^sub>0\<^sup>@k \<cdot> u\<^sub>1 \<cdot> z \<and>\<^sub>p u\<^sub>1\<^sup>@l \<cdot> u\<^sub>0 \<cdot> z' = \<alpha>"
  using lcp_first_mismatch_pref[OF bin_lcp_fst_pow_pref bin_lcp_snd_pow_pref bin_mismatch_neq].

theorem bin_code: assumes "us \<in> lists {u\<^sub>0,u\<^sub>1}" and "vs \<in> lists {u\<^sub>0,u\<^sub>1}" and "concat us = concat vs"
  shows "us = vs"
  using assms
proof (induct us vs rule: list_induct2')
  case (4 x xs y ys)
  then show ?case
  proof-
    have "x =y"
      using bin_code_lcp_marked[OF \<open>x # xs \<in> lists {u\<^sub>0, u\<^sub>1}\<close> \<open>y # ys \<in> lists {u\<^sub>0, u\<^sub>1}\<close>] \<open>y # ys \<in> lists {u\<^sub>0, u\<^sub>1}\<close> non_comm
      unfolding \<open>concat (x # xs) = concat (y # ys)\<close> unfolding concat.simps(2) lcp_self list.sel(1)
      by auto
     thus "x # xs = y # ys"
      using "4.hyps" \<open>concat (x # xs) = concat (y # ys)\<close>[unfolded  concat.simps(2) \<open>x = y\<close>, unfolded cancel]
        \<open>y # ys \<in> lists {u\<^sub>0, u\<^sub>1}\<close>[unfolded Cons_in_lists_iff] \<open>x # xs \<in> lists {u\<^sub>0, u\<^sub>1}\<close>[unfolded Cons_in_lists_iff]
      by simp
  qed
qed (auto simp: bin_fst_nemp bin_snd_nemp)

lemma code_bin_roots: "binary_code (\<rho> u\<^sub>0) (\<rho> u\<^sub>1)"
  using non_comm comp_primroot_conv' by unfold_locales blast

sublocale code "{u\<^sub>0,u\<^sub>1}"
  using bin_code by unfold_locales

lemma primroot_dec: "(Dec {\<rho> u\<^sub>0, \<rho> u\<^sub>1} u\<^sub>0) = [\<rho> u\<^sub>0]\<^sup>@e\<^sub>\<rho> u\<^sub>0" "(Dec {\<rho> u\<^sub>0, \<rho> u\<^sub>1} u\<^sub>1) = [\<rho> u\<^sub>1]\<^sup>@e\<^sub>\<rho> u\<^sub>1"
proof-
  interpret rs: binary_code "\<rho> u\<^sub>0" "\<rho> u\<^sub>1"
    by (simp add: code_bin_roots)
  from primroot_exp_eq
  have "concat ([\<rho> u\<^sub>0]\<^sup>@e\<^sub>\<rho> u\<^sub>0) = u\<^sub>0" "concat ([\<rho> u\<^sub>1]\<^sup>@e\<^sub>\<rho> u\<^sub>1) = u\<^sub>1"
    by force+
  from rs.code_unique_dec[OF _ this(1)] rs.code_unique_dec[OF _ this(2)]
  show "(Dec {\<rho> u\<^sub>0, \<rho> u\<^sub>1} u\<^sub>0) = [\<rho> u\<^sub>0]\<^sup>@e\<^sub>\<rho> u\<^sub>0" "(Dec {\<rho> u\<^sub>0, \<rho> u\<^sub>1} u\<^sub>1) = [\<rho> u\<^sub>1]\<^sup>@e\<^sub>\<rho> u\<^sub>1"
    by (simp_all add: sing_pow_lists)
qed

lemma bin_code_prefs: assumes "w \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>" and "p \<le>p w" "w' \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>" and "\<^bold>|u\<^sub>1\<^bold>| \<le> \<^bold>|p\<^bold>|"
  shows " \<not> u\<^sub>0 \<cdot>  p \<le>p u\<^sub>1 \<cdot> w'"
proof
  assume contr: "u\<^sub>0 \<cdot> p  \<le>p u\<^sub>1 \<cdot> w'"
  have "\<^bold>|\<alpha>\<^bold>| < \<^bold>|u\<^sub>0 \<cdot> p\<^bold>|"
    using \<open>\<^bold>|u\<^sub>1\<^bold>| \<le> \<^bold>|p\<^bold>|\<close> bin_lcp_short by auto
  hence "\<alpha> \<cdot> [c\<^sub>0] \<le>p u\<^sub>0 \<cdot> p"
    using \<open>p \<le>p w\<close>  \<open>w \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close> bin_lcp_mismatch_pref_all_fst by auto
  from pref_ext[OF pref_trans[OF this contr], unfolded rassoc]
  have "\<alpha> \<cdot> [c\<^sub>0] \<le>p u\<^sub>1 \<cdot> w' \<cdot> \<alpha>".
  from bin_mismatch_neq same_sing_pref[OF bin_snd_mismatch_all_hull[OF \<open>w' \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close>] this]
  show False
    by simp
qed

lemma bin_code_rev: "binary_code (rev u\<^sub>0) (rev u\<^sub>1)"
  by (unfold_locales, unfold comm_rev_iff, simp add: non_comm)

lemma bin_mismatch_pows: "\<not> u\<^sub>0\<^sup>@Suc k \<cdot> u\<^sub>1 \<cdot> z = u\<^sub>1\<^sup>@Suc l \<cdot> u\<^sub>0 \<cdot> z'"
proof (rule notI)
  assume eq: "u\<^sub>0 \<^sup>@ Suc k \<cdot> u\<^sub>1 \<cdot> z = u\<^sub>1 \<^sup>@ Suc l \<cdot> u\<^sub>0 \<cdot> z'"
  have pref1: "\<alpha> \<cdot> [c\<^sub>0] \<le>p u\<^sub>0\<^sup>@Suc k \<cdot> u\<^sub>1" and pref2: "\<alpha> \<cdot> [c\<^sub>1] \<le>p u\<^sub>1\<^sup>@Suc l \<cdot> u\<^sub>0"
    using bin_lcp_fst_pow_pref[of "Suc k" \<epsilon>, unfolded emp_simps] bin_lcp_snd_pow_pref[of "Suc l" \<epsilon>, unfolded emp_simps] by blast+
  from ruler[OF pref_ext[OF pref1, unfolded rassoc, of z, unfolded eq] pref_ext[OF pref2, unfolded rassoc, of z', unfolded eq]] bin_mismatch_neq
  show False by simp
qed

lemma bin_lcp_pows_lcp:  "0 < k \<Longrightarrow> 0 < l \<Longrightarrow> u\<^sub>0\<^sup>@k \<cdot> u\<^sub>1\<^sup>@l \<and>\<^sub>p u\<^sub>1\<^sup>@l \<cdot> u\<^sub>0\<^sup>@k = u\<^sub>0 \<cdot> u\<^sub>1 \<and>\<^sub>p u\<^sub>1 \<cdot> u\<^sub>0"
   using bin_lcp_pows unfolding bin_lcp_def using pow_pos by metis

lemma bin_mismatch: "u\<^sub>0 \<cdot> \<alpha> \<and>\<^sub>p u\<^sub>1 \<cdot> \<alpha> = \<alpha>"
  using lcp_first_mismatch_pref[OF bin_fst_mismatch bin_snd_mismatch bin_mismatch_neq].

lemma not_comp_bin_fst_snd: "\<not> u\<^sub>0 \<cdot> \<alpha> \<bowtie> u\<^sub>1 \<cdot>  \<alpha>"
  using ruler_comp[OF bin_fst_mismatch bin_snd_mismatch] bin_mismatch_neq
  unfolding prefix_comparable_def pref_cancel_conv by force

theorem bin_bounded_delay: assumes "z \<le>p u\<^sub>0 \<cdot> w\<^sub>0" and "z \<le>p u\<^sub>1 \<cdot> w\<^sub>1"
  and "w\<^sub>0 \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>" and "w\<^sub>1 \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>"
shows "\<^bold>|z\<^bold>| \<le> \<^bold>|\<alpha>\<^bold>|"
proof (rule leI, rule notI)
  assume "\<^bold>|\<alpha>\<^bold>| < \<^bold>|z\<^bold>|"
  hence "\<^bold>|\<alpha> \<cdot> [a]\<^bold>| \<le> \<^bold>|z\<^bold>|" for a
    unfolding lenmorph sing_len by simp
  have "z \<le>p u\<^sub>0 \<cdot> w\<^sub>0 \<cdot> \<alpha>" and "z \<le>p u\<^sub>1 \<cdot> w\<^sub>1 \<cdot> \<alpha>"
    using  pref_prolong[OF \<open>z \<le>p u\<^sub>0 \<cdot> w\<^sub>0\<close> triv_pref] pref_prolong[OF \<open>z \<le>p u\<^sub>1 \<cdot> w\<^sub>1\<close> triv_pref].
  have "\<alpha> \<cdot> [c\<^sub>0] \<le>p u\<^sub>0 \<cdot> w\<^sub>0 \<cdot> \<alpha>" and "\<alpha> \<cdot> [c\<^sub>1] \<le>p u\<^sub>1 \<cdot> w\<^sub>1 \<cdot> \<alpha>"
    using bin_fst_mismatch_all_hull[OF \<open>w\<^sub>0 \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close>] bin_snd_mismatch_all_hull[OF \<open>w\<^sub>1 \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close>].
  from \<open>z \<le>p u\<^sub>0 \<cdot> w\<^sub>0 \<cdot> \<alpha>\<close> \<open>\<alpha> \<cdot> [c\<^sub>0] \<le>p u\<^sub>0 \<cdot> w\<^sub>0 \<cdot> \<alpha>\<close> \<open>\<^bold>|\<alpha> \<cdot> [c\<^sub>0]\<^bold>| \<le> \<^bold>|z\<^bold>|\<close>
  have "\<alpha> \<cdot> [c\<^sub>0] \<le>p z"
    using prefix_length_prefix by blast
  from \<open>z \<le>p u\<^sub>1 \<cdot> w\<^sub>1 \<cdot> \<alpha>\<close> \<open>\<alpha> \<cdot> [c\<^sub>1] \<le>p u\<^sub>1 \<cdot> w\<^sub>1 \<cdot> \<alpha>\<close> \<open>\<^bold>|\<alpha> \<cdot> [c\<^sub>1]\<^bold>| \<le> \<^bold>|z\<^bold>|\<close>
  have "\<alpha> \<cdot> [c\<^sub>1] \<le>p z"
    using prefix_length_prefix by blast
  from \<open>\<alpha> \<cdot> [c\<^sub>1] \<le>p z\<close> \<open>\<alpha> \<cdot> [c\<^sub>0] \<le>p z\<close> bin_mismatch_neq
  show False
    unfolding prefix_def by force
qed

thm binary_code.bin_lcp_pows_lcp

lemma prim_roots_lcp: "\<rho> u\<^sub>0 \<cdot> \<rho> u\<^sub>1 \<and>\<^sub>p \<rho> u\<^sub>1 \<cdot> \<rho> u\<^sub>0 = \<alpha>"
proof-
  obtain k where "\<rho> u\<^sub>0\<^sup>@k = u\<^sub>0" "0 < k"
    using primroot_expE.
  obtain m where "\<rho> u\<^sub>1\<^sup>@m = u\<^sub>1" "0 < m"
    using primroot_expE.
  have "\<rho> u\<^sub>0 \<cdot> \<rho> u\<^sub>1 \<noteq> \<rho> u\<^sub>1 \<cdot> \<rho> u\<^sub>0"
    using non_comm[unfolded comp_primroot_conv'[of u\<^sub>0]].
  then interpret r: binary_code "\<rho> u\<^sub>0" "\<rho> u\<^sub>1" by unfold_locales
  from r.bin_lcp_pows_lcp[OF \<open>0 < k\<close> \<open>0 < m\<close>, unfolded \<open>\<rho> u\<^sub>1\<^sup>@m = u\<^sub>1\<close> \<open>\<rho> u\<^sub>0\<^sup>@k = u\<^sub>0\<close>, symmetric]
  show ?thesis
     unfolding bin_lcp_def.
qed

subsubsection \<open>Maximal r-prefixes\<close>

lemma bin_lcp_per_root_max_pref_short:  assumes "\<alpha> <p u\<^sub>0 \<cdot> u\<^sub>1 \<and>\<^sub>p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1" and "r \<noteq> \<epsilon>" and "q \<le>p w" and "w \<in> \<langle>{u\<^sub>0, u\<^sub>1}\<rangle>"
  shows "u\<^sub>1 \<cdot> q \<and>\<^sub>p r \<cdot> u\<^sub>1 \<cdot> q = take \<^bold>|u\<^sub>1 \<cdot> q\<^bold>| \<alpha>"
proof-
  have "q \<bowtie> \<alpha>"
    using bin_lcp_pref_all_hull[OF \<open>w \<in> \<langle>{u\<^sub>0, u\<^sub>1}\<rangle>\<close>] ruler_comp[OF \<open>q \<le>p w\<close>, of \<alpha> "w \<cdot> \<alpha>"] by blast
  hence comp1: "u\<^sub>1 \<cdot> q \<bowtie> \<alpha> \<cdot> [c\<^sub>1]"
    using  ruler_comp[OF self_pref bin_snd_mismatch, of "u\<^sub>1 \<cdot> q"] unfolding comp_cancel by blast

  from add_nth_pref[OF assms(1), THEN pref_lcp_pref] bin_fst_mismatch'
  have "(u\<^sub>0 \<cdot> u\<^sub>1 \<and>\<^sub>p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1) ! \<^bold>|\<alpha>\<^bold>| = c\<^sub>0"
    using same_sing_pref by fast

  from add_nth_pref[OF assms(1), unfolded this]
  have "\<alpha> \<cdot> [c\<^sub>0] \<le>p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1"
    by force

  have len: "\<^bold>|\<alpha> \<cdot> [c\<^sub>0]\<^bold>| \<le> \<^bold>|r \<cdot> \<alpha>\<^bold>|"
    using nemp_pos_len[OF \<open>r \<noteq> \<epsilon>\<close>] unfolding lenmorph sing_len by linarith

  have comp2: "r \<cdot> u\<^sub>1 \<cdot> q \<bowtie> \<alpha> \<cdot> [c\<^sub>0]"
  proof(rule ruler_comp[OF _ _ comp_refl])
    show "r \<cdot> u\<^sub>1 \<cdot> q \<le>p r \<cdot> u\<^sub>1 \<cdot> w \<cdot> \<alpha>"
      using \<open>q \<le>p w\<close> by fastforce
    show "\<alpha> \<cdot> [c\<^sub>0] \<le>p r \<cdot> u\<^sub>1 \<cdot> w \<cdot> \<alpha>"
    proof(rule pref_prolong)
      show "\<alpha> \<cdot> [c\<^sub>0] \<le>p r \<cdot> \<alpha>"
        using \<open>\<alpha> \<cdot> [c\<^sub>0] \<le>p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1\<close> bin_lcp_pref len pref_prod_pref_short by blast
      show "\<alpha> \<le>p u\<^sub>1 \<cdot> w \<cdot> \<alpha>"
        using \<open>w \<in> \<langle>{u\<^sub>0, u\<^sub>1}\<rangle>\<close> bin_lcp_pref_all_hull[of "u\<^sub>1 \<cdot> w"] by auto
    qed
  qed

  have min: "(min \<^bold>|u\<^sub>1 \<cdot> q\<^bold>| \<^bold>|r \<cdot> u\<^sub>1 \<cdot> q\<^bold>|) = \<^bold>|u\<^sub>1 \<cdot> q\<^bold>|"
    unfolding lenmorph by simp

  show ?thesis
    using  bin_mismatch_neq double_ruler[OF comp1 comp2,unfolded min]
    by (simp add: lcp_mismatch_eq_len mismatch_incopm)
qed

lemma bin_per_root_max_pref_short:  assumes "(u\<^sub>0 \<cdot> u\<^sub>1) <p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1" and "q \<le>p w" and "w \<in> \<langle>{u\<^sub>0, u\<^sub>1}\<rangle>"
  shows "u\<^sub>1 \<cdot> q \<and>\<^sub>p r \<cdot> u\<^sub>1 \<cdot> q = take \<^bold>|u\<^sub>1 \<cdot> q\<^bold>| \<alpha>"
proof (rule bin_lcp_per_root_max_pref_short[OF _ _ assms(2-3)])
  show "\<alpha> <p u\<^sub>0 \<cdot> u\<^sub>1 \<and>\<^sub>p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1"
    unfolding lcp.absorb3[OF assms(1)] using bin_fst_mismatch'[THEN prefix_snocD].
qed (use assms(1) in blast)

lemma bin_root_max_pref_long:  assumes "r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1 =  u\<^sub>0 \<cdot> u\<^sub>1 \<cdot> r" and "q \<le>p w" and "w \<in> \<langle>{u\<^sub>0, u\<^sub>1}\<rangle>" and "\<^bold>|\<alpha>\<^bold>| \<le> \<^bold>|q\<^bold>|"
  shows "u\<^sub>0 \<cdot> \<alpha>  \<le>p u\<^sub>0 \<cdot> q \<and>\<^sub>p r \<cdot> u\<^sub>0 \<cdot> q"
proof (rule pref_pref_lcp)
  have len: " \<^bold>|u\<^sub>0 \<cdot> \<alpha>\<^bold>| \<le> \<^bold>|r \<cdot> u\<^sub>0 \<cdot> \<alpha>\<^bold>|"
    by simp
  from bin_lcp_pref_all_len[OF assms(2-4)]
  show "u\<^sub>0 \<cdot> \<alpha> \<le>p u\<^sub>0 \<cdot> q"
    unfolding pref_cancel_conv.
  have "u\<^sub>0 \<cdot> \<alpha> \<le>p r \<cdot> u\<^sub>0 \<cdot> \<alpha>"
  proof(rule ruler_le[OF _ _ len])
    show "u\<^sub>0 \<cdot> \<alpha> \<le>p (r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1) \<cdot> u\<^sub>0 \<cdot> u\<^sub>1"
      unfolding assms(1) unfolding rassoc pref_cancel_conv assms(1)
      using pref_ext[OF pref_ext[OF bin_lcp_pref'], unfolded rassoc].
    show "r \<cdot> u\<^sub>0 \<cdot> \<alpha> \<le>p (r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1) \<cdot> u\<^sub>0 \<cdot> u\<^sub>1"
      unfolding rassoc pref_cancel_conv using pref_ext[OF bin_lcp_pref', unfolded rassoc].
  qed
  from pref_prolong[OF this[unfolded lassoc], OF \<open>\<alpha> \<le>p q\<close>, unfolded rassoc]
  show "u\<^sub>0 \<cdot> \<alpha> \<le>p r \<cdot> u\<^sub>0 \<cdot> q".
qed

lemma per_root_lcp_per_root: "u\<^sub>0 \<cdot> u\<^sub>1 <p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1 \<Longrightarrow> \<alpha> \<cdot> [c\<^sub>0] \<le>p r \<cdot> \<alpha>"
  using per_root_pref_sing[OF _ bin_fst_mismatch'].

lemma per_root_bin_fst_snd_lcp:  assumes "u\<^sub>0 \<cdot> u\<^sub>1 <p r \<cdot> u\<^sub>0 \<cdot> u\<^sub>1" and
                   "q \<le>p w" and "w \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>" and "\<^bold>|\<alpha>\<^bold>| < \<^bold>|u\<^sub>1 \<cdot> q\<^bold>|"
                   "q' \<le>p w'" and "w' \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>" and "\<^bold>|\<alpha>\<^bold>| \<le> \<^bold>|q'\<^bold>|"
  shows "u\<^sub>1 \<cdot> q \<and>\<^sub>p r \<cdot> q' = \<alpha>"
proof-
  have pref1: "\<alpha> \<cdot> [c\<^sub>1] \<le>p u\<^sub>1 \<cdot> q"
    using \<open>\<^bold>|\<alpha>\<^bold>| < \<^bold>|u\<^sub>1 \<cdot> q\<^bold>|\<close> \<open>q \<le>p w\<close> bin_snd_mismatch_all_len[of "u\<^sub>1 \<cdot> q", OF _ \<open>w \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close>]
    unfolding pref_cancel_conv by blast

  have "\<alpha> \<le>p q'"
    using \<open>\<^bold>|\<alpha>\<^bold>| \<le> \<^bold>|q'\<^bold>|\<close> \<open>q' \<le>p w'\<close> \<open>w' \<in> \<langle>{u\<^sub>0,u\<^sub>1}\<rangle>\<close> bin_lcp_pref_all_len by blast
  have pref2: "\<alpha> \<cdot> [c\<^sub>0] \<le>p r \<cdot> \<alpha>"
    using assms(1) per_root_lcp_per_root by auto
  hence pref2: "\<alpha> \<cdot> [c\<^sub>0] \<le>p r \<cdot> q'"
    using \<open>\<alpha> \<le>p q'\<close> pref_prolong by blast

  show ?thesis
    using lcp_first_mismatch_pref[OF pref1 pref2 bin_mismatch_neq[symmetric]].

qed



end
lemmas no_comm_bin_code = binary_code.bin_code[unfolded binary_code_def]

theorem bin_code_code: assumes "u \<cdot> v \<noteq> v \<cdot> u" shows "code {u, v}"
  unfolding code_def using no_comm_bin_code[OF assms] by blast

lemma code_bin_code: "u \<noteq> v \<Longrightarrow> code {u,v} \<Longrightarrow> u \<cdot> v \<noteq> v \<cdot> u"
  by (elim code.code_not_comm) simp_all

lemma lcp_roots_lcp: "x \<cdot> y \<noteq> y \<cdot> x \<Longrightarrow> x \<cdot> y \<and>\<^sub>p y \<cdot> x = \<rho> x \<cdot> \<rho> y \<and>\<^sub>p \<rho> y \<cdot> \<rho> x"
  using binary_code.prim_roots_lcp[unfolded binary_code_def bin_lcp_def, symmetric].

subsection \<open>Binary Mismatch tools\<close>

thm binary_code.bin_mismatch_pows[unfolded binary_code_def]

lemma bin_mismatch: "u\<^sup>@Suc k \<cdot> v \<cdot> z = v\<^sup>@Suc l \<cdot> u \<cdot> z' \<Longrightarrow> u \<cdot> v = v \<cdot> u"
  using binary_code.bin_mismatch_pows[unfolded binary_code_def] by blast

definition bin_mismatch_pref :: "'a list \<Rightarrow> 'a list \<Rightarrow> 'a list  \<Rightarrow> bool" where
  "bin_mismatch_pref x y w \<equiv> \<exists> k. x\<^sup>@k \<cdot> y \<le>p w"

\<comment> \<open>Binary mismatch elims\<close>

lemma bm_pref_letter: assumes "x \<cdot> y \<noteq> y \<cdot> x" and "bin_mismatch_pref x y (w1 \<cdot> y)"
  shows "bin_lcp x y \<cdot> [bin_mismatch x y] \<le>p x \<cdot> w1 \<cdot> bin_lcp x y"
proof-
  interpret binary_code x y
    using assms(1) by unfold_locales
  from assms[unfolded bin_mismatch_pref_def prefix_def rassoc]
  obtain k1 z1 where eq1: "w1 \<cdot> y = x\<^sup>@k1 \<cdot> y \<cdot> z1"
    by blast
  have "bin_lcp x y \<cdot> [bin_mismatch x y] \<le>p x \<cdot> w1 \<cdot> y \<cdot> bin_lcp x y"
    unfolding lassoc \<open>w1 \<cdot> y = x\<^sup>@k1 \<cdot> y \<cdot> z1\<close> pow_Suc[symmetric] unfolding rassoc using bin_lcp_fst_pow_pref by blast
  have "\<^bold>|bin_lcp x y \<cdot> [bin_mismatch x y]\<^bold>| \<le> \<^bold>|(x \<cdot> w1) \<cdot> bin_lcp x y\<^bold>|"
    unfolding lenmorph sing_len using nemp_len[OF bin_fst_nemp] by linarith
  from ruler_le[OF \<open>bin_lcp x y \<cdot> [bin_mismatch x y] \<le>p x \<cdot> w1 \<cdot> y \<cdot> bin_lcp x y\<close> _ this]
  show "bin_code_lcp \<cdot> [bin_mismatch x y] \<le>p x \<cdot> w1 \<cdot> bin_code_lcp"
    unfolding shifts using bin_lcp_snd_lcp.
qed

lemma bm_eq_hard: assumes "x \<cdot> w1 = y \<cdot> w2" and  "bin_mismatch_pref x y (w1 \<cdot> y)" and "bin_mismatch_pref y x (w2 \<cdot> x)"
  shows "x \<cdot> y = y \<cdot> x"
proof(rule classical)
  assume "x \<cdot> y \<noteq> y \<cdot> x"
  note bm_pref_letter[OF this assms(2)] bm_pref_letter[OF this[symmetric] assms(3)]
  from ruler_eq_len[OF this[unfolded lassoc \<open>x\<cdot>w1 = y\<cdot>w2\<close> bin_lcp_sym[of y]]]
  have "bin_mismatch x y = bin_mismatch y x"
    unfolding lenmorph sing_len cancel by blast
  thus "x \<cdot> y = y \<cdot> x"
    unfolding  bin_mismatch_comm.
qed



lemma bm_hard_lcp: assumes "x \<cdot> y \<noteq> y \<cdot> x" and "bin_mismatch_pref x y w1" and "bin_mismatch_pref y x w2"
  shows "x \<cdot> w1 \<and>\<^sub>p y \<cdot> w2 = x \<cdot> y \<and>\<^sub>p y \<cdot> x"
proof-
  interpret binary_code x y
    using \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> by unfold_locales
  write bin_code_lcp  (\<open>\<alpha>\<close>)
  from assms[unfolded bin_mismatch_pref_def]
  obtain k m where "x\<^sup>@k \<cdot> y \<le>p w1"  "y\<^sup>@m \<cdot> x \<le>p w2"
    by blast
  hence prefs: "x \<cdot> x\<^sup>@k \<cdot> y \<le>p x \<cdot> w1" "y \<cdot> y\<^sup>@m \<cdot> x \<le>p y \<cdot> w2"
    unfolding pref_cancel_conv.
  have l_less: "\<^bold>|\<alpha>\<^bold>| < \<^bold>|x \<cdot> x\<^sup>@k \<cdot> y\<^bold>|" "\<^bold>|\<alpha>\<^bold>| < \<^bold>|y \<cdot> y\<^sup>@m \<cdot> x\<^bold>|"
    using bin_lcp_short unfolding lenmorph by simp_all
  from bin_code_delay[OF less_imp_le less_imp_le, OF this self_pref self_pref]
  have aux: "x \<cdot> x\<^sup>@k \<cdot> y \<and>\<^sub>p y \<cdot> y\<^sup>@ m \<cdot> x = \<alpha>"
    by blast+
  have "\<not> x \<cdot> x \<^sup>@ k \<cdot> y \<bowtie> y \<cdot> y \<^sup>@ m \<cdot> x"
    unfolding prefix_comparable_def  lcp_pref_conv'[symmetric] aux aux[unfolded lcp_sym[of "x \<cdot> _"]]
    using l_less by fastforce
  thus ?thesis
    using lcp_rulers[OF prefs] unfolding bin_lcp_def aux by blast
qed

lemma bm_pref_hard: assumes "x \<cdot> w1 \<le>p y \<cdot> w2" and  "bin_mismatch_pref x y w1"
  and "bin_mismatch_pref y x (w2 \<cdot> x)"
shows "x \<cdot> y = y \<cdot> x"
proof(rule classical)
  assume "x \<cdot> y \<noteq> y \<cdot> x"
  then interpret binary_code x y
    by unfold_locales
  from assms[unfolded bin_mismatch_pref_def prefix_def rassoc]
  obtain k1 z1 where eq1: "w1 = x\<^sup>@k1 \<cdot> y \<cdot> z1"
    by blast
  have "bin_lcp x y \<cdot> [bin_mismatch x y] \<le>p x \<cdot> w1"
    unfolding lassoc \<open>w1 = x\<^sup>@k1 \<cdot> y \<cdot> z1\<close> pow_Suc[symmetric] unfolding rassoc using bin_lcp_fst_pow_pref by blast
  note pref_ext[OF pref_trans[OF this assms(1)], unfolded rassoc] bm_pref_letter[OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close>[symmetric] assms(3), unfolded bin_lcp_sym[of y]]
  from ruler_eq_len[OF this]
  have "bin_mismatch x y = bin_mismatch y x"
    unfolding lenmorph sing_len cancel by blast
  thus "x \<cdot> y = y \<cdot> x"
    unfolding  bin_mismatch_comm.
qed





named_theorems bm_elims
lemmas [bm_elims] = bm_eq_hard bm_eq_hard[symmetric] bm_pref_hard bm_pref_hard[symmetric]
                  bm_hard_lcp bm_hard_lcp[symmetric]
                  arg_cong2[of _ _ _ _ "\<lambda> x y. x \<and>\<^sub>p y"]

named_theorems bm_elims_rev
lemmas [bm_elims_rev] = bm_elims[reversed]

\<comment> \<open>Binary mismatch predicate evaluation\<close>
named_theorems bm_simps
lemma [bm_simps]: " bin_mismatch_pref x y (y \<cdot> v)"
  unfolding bin_mismatch_pref_def using  append_Nil pow_zero[of x] by blast
lemma [bm_simps]: " bin_mismatch_pref x y y"
  unfolding bin_mismatch_pref_def using  append_Nil pow_zero[of x] self_pref by metis
lemma [bm_simps]:
  "w1 \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> bin_mismatch_pref x y w \<Longrightarrow> bin_mismatch_pref x y (w1 \<cdot> w)"
  unfolding bin_mismatch_pref_def
proof (induct w1 arbitrary: w rule: hull.induct)
  case (prod_cl w1 w2)
  from prod_cl.hyps(3)[OF prod_cl.prems]
  obtain k s where "w2 \<cdot> w = x \<^sup>@ k \<cdot> y \<cdot> s" by (auto simp add: prefix_def)
  consider "w1 = x" | "w1 = y" using \<open>w1 \<in> {x,y}\<close> by blast
  then show ?case
  proof (cases)
    assume "w1 = x"
    show ?thesis
      unfolding rassoc \<open>w2 \<cdot> w = x \<^sup>@ k \<cdot> y \<cdot> s\<close> \<open>w1 = x\<close>
      unfolding lassoc pow_Suc[symmetric] unfolding rassoc
      using same_prefix_prefix by blast
  next
    assume "w1 = y"
    have "x\<^sup>@0 \<cdot> y \<le>p y \<cdot> w2 \<cdot> w" by auto
    thus ?thesis
      unfolding rassoc \<open>w1 = y\<close> by blast
  qed
qed simp

lemmas [bm_simps] = lcp_ext_left

named_theorems bm_simps_rev
lemmas [bm_simps_rev] =  bm_simps[reversed]

\<comment> \<open>Binary hull membership evaluation\<close>

named_theorems bin_hull_in
lemma[bin_hull_in]: "x \<in> \<langle>{x,y}\<rangle>"
  by blast
lemma[bin_hull_in]: "y \<in> \<langle>{x,y}\<rangle>"
  by blast
lemma[bin_hull_in]: "w \<in> \<langle>{x,y}\<rangle> \<longleftrightarrow> w \<in> \<langle>{y,x}\<rangle>"
  by (simp add: insert_commute)
lemmas[bin_hull_in] = hull_closed power_in rassoc

named_theorems bin_hull_in_rev
lemmas [bin_hull_in_rev] =  bin_hull_in[reversed]

method mismatch0 =
  ((simp only: shifts bm_simps)?,
    (elim bm_elims)?;
    (simp_all only: bm_simps bin_hull_in))


method mismatch_rev =
  ((simp only: shifts_rev bm_simps_rev)?,
    (elim bm_elims_rev)?;
    (simp_all only: bm_simps_rev bin_hull_in_rev))

method mismatch =
  (insert method_facts, use nothing in
    \<open>(mismatch0;fail)|(mismatch_rev)\<close>
  )


thm bm_elims

subsubsection "Mismatch method demonstrations"

lemma "y \<cdot> x \<le>p x\<^sup>@k \<cdot> x \<cdot> y \<cdot> w \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma "w1 \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> w2 \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> x \<cdot> w2 \<cdot> y \<cdot> z = y \<cdot> w1 \<cdot> x \<cdot> v \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma "w1 \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> y \<cdot> x \<cdot> w2 \<cdot> z = x \<cdot> w1 \<Longrightarrow> x \<cdot> y = y \<cdot> x"
       by mismatch

lemma "w1 \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> w2 \<in> \<langle>{x,y}\<rangle> \<Longrightarrow> x \<cdot> y \<cdot> w2 \<cdot> x \<le>s x \<cdot> w1 \<cdot> y \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma assumes "x \<cdot> y \<cdot> z = y \<cdot> y \<cdot> x \<cdot> v" shows "x \<cdot> y = y \<cdot> x"
  using assms by mismatch

lemma assumes "y \<cdot> x \<cdot> x \<cdot> y \<cdot> z = y \<cdot> x \<cdot> y \<cdot> y \<cdot> x \<cdot> v" shows "x \<cdot> y = y \<cdot> x"
  using assms by mismatch

lemma "y \<cdot> y \<cdot> x \<cdot> v = x \<cdot> x \<cdot> y \<cdot> z \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma "x \<cdot> x \<cdot> y \<cdot> z = y \<cdot> y \<cdot> x \<cdot> z' \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma "z \<cdot> x \<cdot> y \<cdot> x \<cdot> x  = v \<cdot> x \<cdot> y \<cdot> y \<Longrightarrow> y \<cdot> x = x \<cdot> y"
  by mismatch

lemma "x \<cdot> y \<le>p y \<cdot> y \<cdot> x \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma "y \<cdot> x \<cdot> x \<cdot> x \<cdot> y \<le>p y \<cdot> x \<cdot> x \<cdot> y \<cdot> y \<cdot> x \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma "x \<cdot> y \<le>p y \<cdot> y \<cdot> x \<cdot> z \<Longrightarrow> y \<cdot> x = x \<cdot> y"
  by mismatch

lemma "x \<cdot> x \<cdot> y \<cdot> y \<cdot> y \<le>s z\<cdot> y \<cdot> y \<cdot> x \<cdot> x \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma assumes "x \<cdot> x \<cdot> y \<cdot> y \<cdot> y \<cdot> y \<le>s z\<cdot> y \<cdot> y \<cdot> x \<cdot> x" shows "x \<cdot> y = y \<cdot> x"
  using assms by mismatch

lemma "k \<noteq> 0 \<Longrightarrow> j \<noteq> 0 \<Longrightarrow> (x \<^sup>@ j \<cdot> y \<^sup>@ ka) \<cdot> y = y\<^sup>@k \<cdot> x \<^sup>@ j \<cdot> y \<^sup>@ (k - 1) \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma "dif \<noteq> 0 \<Longrightarrow> j \<noteq> 0 \<Longrightarrow> (x \<^sup>@ j \<cdot> y \<^sup>@ ka) \<cdot> y \<^sup>@ dif = y \<^sup>@ dif \<cdot> x \<^sup>@ j \<cdot> y \<^sup>@ ka \<Longrightarrow> x \<cdot> y = y \<cdot> x"
  by mismatch

lemma assumes "x \<cdot> y \<noteq> y \<cdot> x"
  shows "x \<cdot> x \<cdot> y \<and>\<^sub>p y \<cdot> y \<cdot> x = (x \<cdot> y \<and>\<^sub>p y \<cdot> x)"
  using assms by mismatch

lemma assumes "x \<cdot> y \<noteq> y \<cdot> x"
  shows "w \<cdot> z \<cdot> x \<cdot> x \<cdot> y \<and>\<^sub>p w \<cdot> z \<cdot> y \<cdot> y \<cdot> x = (w \<cdot> z) \<cdot> (x \<cdot> y \<and>\<^sub>p y \<cdot> x)"
  using assms by mismatch

subsection \<open>Applied mismatch\<close>

lemma pows_comm_comm:  assumes  "u\<^sup>@k \<cdot> v\<^sup>@m = u\<^sup>@l \<cdot> v\<^sup>@n" "k \<noteq> l" shows "u \<cdot> v = v \<cdot> u"
proof-
  have aux: "u\<^sup>@k \<cdot> v\<^sup>@m \<cdot> v \<cdot> u = u\<^sup>@l \<cdot> v\<^sup>@n \<cdot> v \<cdot> u \<Longrightarrow> k \<noteq> l \<Longrightarrow> u \<cdot> v = v \<cdot> u"
  by (induct k l rule: diff_induct)  mismatch+
  from this[unfolded lassoc cancel_right, OF assms]
  show "u \<cdot> v = v \<cdot> u".
qed

section \<open>Two words hull (not necessarily a code)\<close>

lemma bin_lists_len_count: assumes "x \<noteq> y" and "ws \<in> lists {x,y}" shows
  "count_list ws x + count_list ws y = \<^bold>|ws\<^bold>|"
proof-
  have "finite {x,y}" by simp
  have "set ws \<subseteq> {x,y}" using \<open>ws \<in> lists{x,y}\<close> by blast
  show ?thesis
    using sum_count_set[OF \<open>set ws \<subseteq> {x,y}\<close> \<open>finite {x,y}\<close>] \<open>x \<noteq> y\<close> by simp
qed

lemma two_elem_first_block: assumes "w \<in> \<langle>{u,v}\<rangle>"
  obtains m where "u\<^sup>@m \<cdot> v \<bowtie> w"
  using assms
proof-
  obtain ws where "concat ws = w" and "ws \<in> lists {u,v}"
    using concat_dec[OF \<open>w \<in> \<langle>{u,v}\<rangle>\<close>] dec_in_lists[OF \<open>w \<in> \<langle>{u,v}\<rangle>\<close>] by simp
  consider (only_u) "takeWhile (\<lambda> x. x = u) ws = ws" | (some_v) "takeWhile (\<lambda> x. x = u) ws \<noteq> ws \<and> hd (dropWhile (\<lambda> x. x = u) ws) \<noteq> u"
    using hd_dropWhile[of "(\<lambda> x. x = u)" ws] by auto
  then show thesis
  proof (cases)
    case only_u
    hence "ws = [u]\<^sup>@\<^bold>|ws\<^bold>|"
      unfolding takeWhile_sing_pow by metis
    hence "w = u\<^sup>@\<^bold>|ws\<^bold>|"
      using \<open>concat ws = w\<close> concat_sing_pow by metis
    then show thesis
      using that by blast
  next
    case some_v
    note some_v = conjunct1[OF this] conjunct2[OF this]
    hence "dropWhile (\<lambda> x. x = u) ws \<noteq> \<epsilon>" by force
    from lists_hd_in_set[OF this]
    have "hd (dropWhile (\<lambda>x. x = u) ws) \<in> {u,v}"
      using \<open>ws \<in> lists {u,v}\<close> append_in_lists_conv  takeWhile_dropWhile_id by metis
    hence "hd (dropWhile (\<lambda>x. x = u) ws) = v"
      using some_v(2) by simp
    from dropWhile_distinct[of ws u, unfolded this] some_v(1)
    have "(takeWhile (\<lambda>x. x = u) ws)\<cdot>[v] \<le>p ws"
      unfolding takeWhile_letter_pref_exp by simp
    from pref_concat_pref[OF this, unfolded concat_morph, unfolded \<open>concat ws = w\<close> concat_takeWhile_sing[unfolded this]]
    have "u\<^sup>@\<^bold>|takeWhile (\<lambda>x. x = u) ws\<^bold>|\<cdot> v \<le>p w"
      by simp
    with that
    show thesis
      by blast
  qed
qed

lemmas two_elem_last_block = two_elem_first_block[reversed]

lemma two_elem_pref: assumes  "v \<le>p u \<cdot> p" and "p \<in> \<langle>{u,v}\<rangle>"
  shows "v \<le>p u \<cdot> v"
proof-
  obtain m where "u\<^sup>@m \<cdot> v \<bowtie> p"
    using two_elem_first_block[OF \<open>p \<in> \<langle>{u,v}\<rangle>\<close>].
  have "v \<le>p u\<^sup>@(Suc m) \<cdot> v"
    using pref_prolong_comp[OF \<open>v \<le>p u \<cdot> p\<close> \<open>u\<^sup>@m \<cdot> v \<bowtie> p\<close>, unfolded lassoc, folded pow_Suc].
  thus "v \<le>p u \<cdot> v"
    using per_drop_exp' by blast
qed

lemmas two_elem_suf = two_elem_pref[reversed]

lemma gen_drop_exp: assumes "p \<in> \<langle>{u,v\<^sup>@(Suc k)}\<rangle>" shows "p \<in> \<langle>{u,v}\<rangle>"
  by (rule hull.induct[OF assms], simp, blast)

lemma gen_drop_exp_pos: assumes "p \<in> \<langle>{u,v\<^sup>@k}\<rangle>" "0 < k" shows "p \<in> \<langle>{u,v}\<rangle>"
  using gen_drop_exp[of _ _ _ "k-1", unfolded Suc_minus_pos[OF \<open>0 < k\<close>], OF \<open>p \<in> \<langle>{u,v\<^sup>@k}\<rangle>\<close>].

lemma gen_prim: "p \<in> \<langle>{u,v}\<rangle> \<Longrightarrow> p \<in> \<langle>{u,\<rho> v}\<rangle>"
  using gen_drop_exp_pos primroot_expE by metis

lemma roots_hull: assumes "w \<in> \<langle>{u\<^sup>@k,v\<^sup>@m}\<rangle>" shows "w \<in> \<langle>{u,v}\<rangle>"
proof-
  have "u\<^sup>@k \<in> \<langle>{u,v}\<rangle>" and "v\<^sup>@m \<in> \<langle>{u,v}\<rangle>"
    by (simp_all add: gen_in power_in)
  hence "{u\<^sup>@k,v\<^sup>@m} \<subseteq> \<langle>{u,v}\<rangle>"
    by blast
  from hull_mono'[OF this]
  show "w \<in> \<langle>{u,v}\<rangle>"
    using \<open>w \<in> \<langle>{u\<^sup>@k,v\<^sup>@m}\<rangle>\<close> by blast
qed

lemma roots_hull_sub: "\<langle>{u\<^sup>@k,v\<^sup>@m}\<rangle> \<subseteq> \<langle>{u,v}\<rangle>"
  using roots_hull by blast

lemma primroot_gen[intro]: "v \<in> \<langle>{u, \<rho> v}\<rangle>"
  using power_in[of "\<rho> v" "{u,\<rho> v}"]
  by (cases "v = \<epsilon>", simp) (metis primroot_expE gen_in insert_iff)

lemma primroot_gen'[intro]: "u \<in> \<langle>{\<rho> u, v}\<rangle>"
  using primroot_gen insert_commute by metis

lemma set_lists_primroot: "set ws \<subseteq> {x,y} \<Longrightarrow> ws \<in> lists \<langle>{\<rho> x, \<rho> y}\<rangle>"
  by blast

section \<open>Free hull\<close>

text\<open>While not every set $G$ of generators is a code, there is a unique minimal
 free monoid containing it, called the \emph{free hull} of $G$.
It can be defined inductively using the property known as the \emph{stability condition}.
\<close>

inductive_set free_hull :: "'a list set \<Rightarrow> 'a list set" (\<open>\<langle>_\<rangle>\<^sub>F\<close>)
  for G where
    "\<epsilon> \<in> \<langle>G\<rangle>\<^sub>F"
  | free_gen_in: "w \<in> G \<Longrightarrow> w \<in> \<langle>G\<rangle>\<^sub>F"
  | "w1 \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> w2 \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> w1 \<cdot> w2 \<in> \<langle>G\<rangle>\<^sub>F"
  | "p \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> q \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> p \<cdot> w \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> w \<cdot> q \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> w \<in> \<langle>G\<rangle>\<^sub>F" \<comment> \<open>the stability condition\<close>

lemmas [intro] = free_hull.intros

text\<open>The defined set indeed is a hull.\<close>

lemma free_hull_hull[simp]: "\<langle>\<langle>G\<rangle>\<^sub>F\<rangle> = \<langle>G\<rangle>\<^sub>F"
  by (intro antisym subsetI, rule hull.induct) blast+

text\<open>The free hull is always (non-strictly) larger than the hull.\<close>

lemma hull_sub_free_hull: "\<langle>G\<rangle> \<subseteq> \<langle>G\<rangle>\<^sub>F"
proof
  fix x assume "x \<in> \<langle>G\<rangle>"
  then show "x \<in> \<langle>G\<rangle>\<^sub>F"
    using free_hull.intros(3)
      hull_induct[of x G "\<lambda> x. x \<in> \<langle>G\<rangle>\<^sub>F", OF \<open>x \<in> \<langle>G\<rangle>\<close> free_hull.intros(1)[of G] free_hull.intros(2)]
    by auto
qed

text\<open>On the other hand, it can be proved that the \emph{free basis}, defined as the basis of the free hull,  has a (non-strictly) smaller cardinality than the ordinary basis.\<close>

definition free_basis ::  "'a list set \<Rightarrow> 'a list set" (\<open>\<BB>\<^sub>F _\<close> [54] 55)
  where  "free_basis G \<equiv> \<BB> \<langle>G\<rangle>\<^sub>F"

lemma basis_gen_hull_free: "\<langle>\<BB>\<^sub>F G\<rangle> = \<langle>G\<rangle>\<^sub>F"
  unfolding free_basis_def using basis_gen_hull free_hull_hull by blast

lemma genset_sub_free: "G \<subseteq> \<langle>G\<rangle>\<^sub>F"
  by (simp add: free_hull.free_gen_in subsetI)

text
  \<open>We have developed two points of view on freeness:
\<^item> being a free hull, that is, to satisfy the stability condition;
\<^item> being generated by a code.\<close>

text\<open>We now show their equivalence\<close>

text\<open>First, basis of a free hull is a code.\<close>

lemma free_basis_code[simp]: "code (\<BB>\<^sub>F G)"
proof
  fix xs ys
  show "xs \<in> lists (\<BB>\<^sub>F G) \<Longrightarrow> ys \<in> lists (\<BB>\<^sub>F G) \<Longrightarrow> concat xs = concat ys \<Longrightarrow> xs = ys"
  proof(induction xs ys rule: list_induct2')
    case (2 x xs)
    show ?case
      using listsE[OF \<open>x # xs \<in> lists (\<BB>\<^sub>F G)\<close>, of "x \<in> \<BB>\<^sub>F G", unfolded free_basis_def, THEN emp_not_basis]
        concat.simps(2)[of x xs, unfolded \<open>concat (x # xs) = concat \<epsilon>\<close>[unfolded concat.simps(1)], symmetric, unfolded append_is_Nil_conv[of x "concat xs"]]
      by blast
  next
    case (3 y ys)
    show ?case
      using  listsE[OF \<open>y # ys \<in> lists (\<BB>\<^sub>F G)\<close>, of "y \<in> \<BB>\<^sub>F G", unfolded free_basis_def, THEN emp_not_basis]
        concat.simps(2)[of y ys, unfolded \<open>concat \<epsilon> = concat (y # ys)\<close>[unfolded concat.simps(1),symmetric],symmetric, unfolded append_is_Nil_conv[of y "concat ys"]]
      by blast
  next
    case (4 x xs y ys)
    have "\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|"
    proof(rule ccontr)
      assume "\<^bold>|x\<^bold>| \<noteq> \<^bold>|y\<^bold>|"
      have "x \<cdot> concat xs = y \<cdot> concat ys"
        using \<open>concat (x # xs) = concat (y # ys)\<close> by simp
      then obtain t where or: "x = y \<cdot> t \<and> t \<cdot> concat xs = concat ys \<or> x \<cdot> t = y \<and> concat xs = t \<cdot> concat ys"
        using append_eq_append_conv2[of x "concat xs" y "concat ys"]  by blast
      hence "t \<noteq> \<epsilon>"
        using \<open>\<^bold>|x\<^bold>| \<noteq> \<^bold>|y\<^bold>|\<close> by auto
      have "x \<in> \<BB>\<^sub>F G" and "y \<in> \<BB>\<^sub>F G"
        using  listsE[OF \<open>x # xs \<in> lists (\<BB>\<^sub>F G)\<close>, of "x \<in> \<BB>\<^sub>F G" ] listsE[OF \<open>y # ys \<in> lists (\<BB>\<^sub>F G)\<close>, of "y \<in> \<BB>\<^sub>F G" ] by blast+
      hence "x \<noteq> \<epsilon>" and "y \<noteq> \<epsilon>"
        unfolding free_basis_def using emp_not_basis by blast+
      have  "x \<in> \<langle>G\<rangle>\<^sub>F" and "y \<in> \<langle>G\<rangle>\<^sub>F"
        using basis_sub[of "\<langle>G\<rangle>\<^sub>F", unfolded free_basis_def[symmetric] ] \<open>x # xs \<in> lists (\<BB>\<^sub>F G)\<close>
          \<open>y # ys \<in> lists (\<BB>\<^sub>F G)\<close> by auto
      have "concat xs \<in> \<langle>G\<rangle>\<^sub>F" and "concat ys \<in> \<langle>G\<rangle>\<^sub>F"
        using concat_tl_basis[OF \<open>x # xs \<in> lists (\<BB>\<^sub>F G)\<close>[unfolded free_basis_def]]
          concat_tl_basis[OF \<open>y # ys \<in> lists (\<BB>\<^sub>F G)\<close>[unfolded free_basis_def]] unfolding free_hull_hull.
      have "t \<in> \<langle>G\<rangle>\<^sub>F"
        using or free_hull.intros(4) \<open>x \<in> \<langle>G\<rangle>\<^sub>F\<close> \<open>y \<in> \<langle>G\<rangle>\<^sub>F\<close> \<open>concat xs \<in> \<langle>G\<rangle>\<^sub>F\<close> \<open>concat ys \<in> \<langle>G\<rangle>\<^sub>F\<close> by metis
      thus False
        using or basis_dec[of x "\<langle>G\<rangle>\<^sub>F" t, unfolded free_hull_hull, OF \<open>x \<in> \<langle>G\<rangle>\<^sub>F\<close> \<open>t \<in> \<langle>G\<rangle>\<^sub>F\<close>]
          basis_dec[of y "\<langle>G\<rangle>\<^sub>F" t, unfolded free_hull_hull, OF \<open>y \<in> \<langle>G\<rangle>\<^sub>F\<close> \<open>t \<in> \<langle>G\<rangle>\<^sub>F\<close>]
        using  \<open>t \<noteq> \<epsilon>\<close> \<open>x \<noteq> \<epsilon>\<close> \<open>y \<noteq> \<epsilon>\<close> \<open>x \<in> \<BB>\<^sub>F G\<close> \<open>y \<in> \<BB>\<^sub>F G\<close> unfolding free_basis_def
        by auto
    qed
    thus "x # xs = y # ys"
      using "4.IH" \<open>x # xs \<in> lists (\<BB>\<^sub>F G)\<close> \<open>y # ys \<in> lists (\<BB>\<^sub>F G)\<close> \<open>concat (x # xs) = concat (y # ys)\<close>
      by auto
  next
  qed simp
qed

lemma gen_in_free_hull: "x \<in> G \<Longrightarrow> x \<in> \<langle>\<BB>\<^sub>F G\<rangle>"
  using free_hull.free_gen_in[folded basis_gen_hull_free].

text\<open>Second, a code generates its free hull.\<close>

lemma (in code) code_gen_free_hull: "\<langle>\<C>\<rangle>\<^sub>F = \<langle>\<C>\<rangle>"
proof
  show "\<langle>\<C>\<rangle> \<subseteq> \<langle>\<C>\<rangle>\<^sub>F"
    using hull_mono[of \<C> "\<langle>\<C>\<rangle>\<^sub>F"]
      free_gen_in[of _ \<C>]  subsetI[of \<C> "\<langle>\<C>\<rangle>\<^sub>F"]
    unfolding free_hull_hull by auto
  show "\<langle>\<C>\<rangle>\<^sub>F \<subseteq> \<langle>\<C>\<rangle>"
  proof
    fix x assume "x \<in> \<langle>\<C>\<rangle>\<^sub>F"
    have "\<epsilon> \<in> \<langle>\<C>\<rangle>"
      by simp
    show "x \<in> \<langle>\<C>\<rangle>"
    proof(rule free_hull.induct[of x \<C>])
      fix p q w assume "p \<in> \<langle>\<C>\<rangle>" "q \<in> \<langle>\<C>\<rangle>" "p \<cdot> w \<in> \<langle>\<C>\<rangle>" "w \<cdot> q \<in> \<langle>\<C>\<rangle>"
      have eq: "(Dec \<C> p) \<cdot> (Dec \<C> w \<cdot> q) = (Dec \<C> p \<cdot> w) \<cdot> (Dec \<C> q)"
        using code_dec_morph[OF \<open>p \<in> \<langle>\<C>\<rangle>\<close> \<open>w \<cdot> q \<in> \<langle>\<C>\<rangle>\<close>, unfolded lassoc]
        unfolding code_dec_morph[OF \<open>p \<cdot> w \<in> \<langle>\<C>\<rangle>\<close> \<open>q \<in> \<langle>\<C>\<rangle>\<close>, symmetric].
      have "Dec \<C> p \<bowtie>  Dec \<C> p \<cdot> w"
        using eqd_comp[OF eq].
      hence "Dec \<C> p \<le>p  Dec \<C> p \<cdot> w"
        using \<open>p \<cdot> w \<in> \<langle>\<C>\<rangle>\<close> \<open>p \<in> \<langle>\<C>\<rangle>\<close> concat_morph concat_dec prefD pref_antisym triv_pref
        unfolding prefix_comparable_def
        by metis
      then obtain ts where "(Dec \<C> p) \<cdot> ts =  Dec \<C> p \<cdot> w"
        using lq_pref by blast
      hence  "ts \<in> lists \<C>"
        using \<open>p \<cdot> w \<in> \<langle>\<C>\<rangle>\<close> by inlists
      hence "concat ts = w"
        using  concat_morph[of "Dec \<C> p" ts]
        unfolding \<open>(Dec \<C> p) \<cdot> ts =  Dec \<C> p \<cdot> w\<close> concat_dec[OF \<open>p \<cdot> w \<in> \<langle>\<C>\<rangle>\<close>]  concat_dec[OF \<open>p \<in> \<langle>\<C>\<rangle>\<close>] by auto
      thus "w \<in> \<langle>\<C>\<rangle>"
        using \<open>ts \<in> lists \<C>\<close> by auto
    qed (simp_all add: \<open>x \<in> \<langle>\<C>\<rangle>\<^sub>F\<close> hull_closed gen_in)
  qed
qed

text\<open>That is, a code is its own free basis\<close>

lemma (in code) code_free_basis: "\<C> = \<BB>\<^sub>F \<C>"
  using basis_of_hull[of \<C>, unfolded code_gen_free_hull[symmetric]
      code_is_basis, symmetric] unfolding free_basis_def.

text\<open>This allows to use the introduction rules of the free hull to prove one of the basic characterizations
 of the code, called the stability condition\<close>

lemma (in code) stability: "p \<in> \<langle>\<C>\<rangle> \<Longrightarrow> q \<in> \<langle>\<C>\<rangle> \<Longrightarrow> p \<cdot> w \<in> \<langle>\<C>\<rangle> \<Longrightarrow> w \<cdot> q \<in> \<langle>\<C>\<rangle> \<Longrightarrow> w \<in> \<langle>\<C>\<rangle>"
  unfolding code_gen_free_hull[symmetric] using free_hull.intros(4) by auto

text\<open>Moreover, the free hull of G is the smallest code-generated hull containing G.
In other words, the term free hull is appropriate.\<close>



text\<open>First, several intuitive monotonicity and closure results.\<close>

lemma free_hull_mono: "G \<subseteq> H \<Longrightarrow> \<langle>G\<rangle>\<^sub>F \<subseteq> \<langle>H\<rangle>\<^sub>F"
proof
  assume "G \<subseteq> H"
  fix x assume "x \<in> \<langle>G\<rangle>\<^sub>F"
  have el: "\<And> w. w \<in> G \<Longrightarrow> w \<in> \<langle>H\<rangle>\<^sub>F"
    using \<open>G \<subseteq> H\<close> free_hull.free_gen_in by auto
  show "x \<in> \<langle>H\<rangle>\<^sub>F"
    by (rule free_hull.induct[of x G]) (auto simp add: \<open>x \<in> \<langle>G\<rangle>\<^sub>F\<close> el)
qed

lemma free_hull_idem: "\<langle>\<langle>G\<rangle>\<^sub>F\<rangle>\<^sub>F = \<langle>G\<rangle>\<^sub>F"
proof
  show "\<langle>\<langle>G\<rangle>\<^sub>F\<rangle>\<^sub>F \<subseteq> \<langle>G\<rangle>\<^sub>F"
  proof
    fix x assume "x \<in> \<langle>\<langle>G\<rangle>\<^sub>F\<rangle>\<^sub>F"
    show "x \<in> \<langle>G\<rangle>\<^sub>F"
    proof (rule free_hull.induct[of x "\<langle>G\<rangle>\<^sub>F"])
      show "\<And>p q w. p \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow>  q \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow>  p \<cdot> w \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow>  w \<cdot> q \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> w \<in> \<langle>G\<rangle>\<^sub>F"
        using free_hull.intros(4) by auto
    qed (simp_all add: \<open>x \<in> \<langle>\<langle>G\<rangle>\<^sub>F\<rangle>\<^sub>F\<close> free_hull.intros(1), simp add: free_hull.intros(2), simp add: free_hull.intros(3))
  qed
next
  show "\<langle>G\<rangle>\<^sub>F \<subseteq> \<langle>\<langle>G\<rangle>\<^sub>F\<rangle>\<^sub>F"
    using free_hull_hull hull_sub_free_hull by auto
qed

lemma hull_gen_free_hull: "\<langle>\<langle>G\<rangle>\<rangle>\<^sub>F = \<langle>G\<rangle>\<^sub>F"
proof
  show " \<langle>\<langle>G\<rangle>\<rangle>\<^sub>F \<subseteq> \<langle>G\<rangle>\<^sub>F"
    using free_hull_idem free_hull_mono hull_sub_free_hull by metis
next
  show "\<langle>G\<rangle>\<^sub>F \<subseteq> \<langle>\<langle>G\<rangle>\<rangle>\<^sub>F"
    by (simp add: free_hull_mono)
qed

text \<open>Code is also the free basis of its hull.\<close>

lemma (in code) code_free_basis_hull: "\<C> = \<BB>\<^sub>F \<langle>\<C>\<rangle>"
  unfolding free_basis_def using code_free_basis[unfolded free_basis_def]
  unfolding  hull_gen_free_hull.

text\<open>The minimality of the free hull easily follows.\<close>

theorem (in code) free_hull_min: assumes "G \<subseteq> \<langle>\<C>\<rangle>" shows "\<langle>G\<rangle>\<^sub>F \<subseteq> \<langle>\<C>\<rangle>"
  using free_hull_mono[OF \<open>G \<subseteq> \<langle>\<C>\<rangle>\<close>] unfolding hull_gen_free_hull
  unfolding code_gen_free_hull.

theorem free_hull_inter: "\<langle>G\<rangle>\<^sub>F = \<Inter> {M. G \<subseteq> M \<and> M = \<langle>M\<rangle>\<^sub>F}"
proof
  have "X \<in> {M. G \<subseteq> M \<and> M = \<langle>M\<rangle>\<^sub>F} \<Longrightarrow> \<langle>G\<rangle>\<^sub>F \<subseteq> X" for X
    unfolding mem_Collect_eq[of _ "\<lambda> M. G \<subseteq> M \<and> M = \<langle>M\<rangle>\<^sub>F"]
    using free_hull_mono[of G X] by simp
  from Inter_greatest[of "{M. G \<subseteq> M \<and> M = \<langle>M\<rangle>\<^sub>F}", OF this]
  show "\<langle>G\<rangle>\<^sub>F \<subseteq> \<Inter> {M. G \<subseteq> M \<and> M = \<langle>M\<rangle>\<^sub>F}"
    by blast
next
  show " \<Inter> {M. G \<subseteq> M \<and> M = \<langle>M\<rangle>\<^sub>F} \<subseteq> \<langle>G\<rangle>\<^sub>F"
    by (simp add: Inter_lower free_hull_idem genset_sub_free)
qed

text\<open>Decomposition into the free basis is a morphism.\<close>

lemma free_basis_dec_morph: "u \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow> v \<in> \<langle>G\<rangle>\<^sub>F \<Longrightarrow>
    Dec (\<BB>\<^sub>F G) (u \<cdot> v) = (Dec (\<BB>\<^sub>F G) u) \<cdot> (Dec (\<BB>\<^sub>F G) v)"
  using code.code_dec_morph[OF free_basis_code, of u G v, symmetric,
      unfolded  basis_gen_hull_free[of G]].

section \<open>Reversing hulls and decompositions\<close>

lemma basis_rev_commute[reversal_rule]: "\<BB> (rev ` G) = rev ` (\<BB> G)"
proof
  have "\<langle>rev ` \<BB> G\<rangle> = \<langle>rev ` G\<rangle>" and *: "\<langle>rev ` \<BB> (rev ` G)\<rangle> = \<langle>rev ` rev `G\<rangle>"
    unfolding rev_hull[symmetric] basis_gen_hull by blast+
  from basis_sub_gen[OF this(1)]
  show "\<BB> (rev ` G) \<subseteq> rev ` \<BB> G".
  from image_mono[OF basis_sub_gen[OF *], of rev]
  show "rev ` (\<BB> G) \<subseteq> \<BB> (rev ` G)"
    unfolding rev_rev_image_eq.
qed

lemma rev_free_hull_comm: "\<langle>rev ` X\<rangle>\<^sub>F = rev ` \<langle>X\<rangle>\<^sub>F"
proof-
  have "rev ` \<langle>X\<rangle>\<^sub>F \<subseteq> \<langle>rev ` X\<rangle>\<^sub>F" for X :: "'a list set"
  proof
    fix x assume "x \<in> rev ` \<langle>X\<rangle>\<^sub>F"
    hence  "rev x \<in> \<langle>X\<rangle>\<^sub>F"
      by (simp add: rev_in_conv)
    have "rev x \<in> rev ` \<langle>rev ` X\<rangle>\<^sub>F"
      by (induct rule: free_hull.induct[OF \<open>rev x \<in> \<langle>X\<rangle>\<^sub>F\<close>])
        (auto simp add: rev_in_conv[symmetric])
    then show "x \<in> \<langle>rev ` X\<rangle>\<^sub>F"
      by blast
  qed
  from this
    image_mono[OF this[of "rev ` X", unfolded rev_rev_image_eq], of rev, unfolded rev_rev_image_eq]
  show  "\<langle>rev ` X\<rangle>\<^sub>F = rev ` \<langle>X\<rangle>\<^sub>F"
    by blast
qed

lemma free_basis_rev_commute [reversal_rule]: "\<BB>\<^sub>F rev ` X =  rev ` (\<BB>\<^sub>F X)"
  unfolding free_basis_def basis_rev_commute free_basis_def rev_free_hull_comm..

lemma rev_dec[reversal_rule]: assumes "x \<in> \<langle>X\<rangle>\<^sub>F" shows "Dec rev ` (\<BB>\<^sub>F X) (rev x) = map rev (rev (Dec (\<BB>\<^sub>F X) x))"
proof-
  have "x \<in> \<langle>\<BB>\<^sub>F X\<rangle>"
    using \<open>x \<in> \<langle>X\<rangle>\<^sub>F\<close> by (simp add: basis_gen_hull_free)
  from concat_dec[OF this]
  have "concat (map rev (rev (Dec \<BB>\<^sub>F X x))) = rev x"
    unfolding rev_concat[symmetric] by blast
  from  rev_image_eqI[OF rev_in_lists[OF dec_in_lists[OF \<open>x \<in> \<langle>\<BB>\<^sub>F X\<rangle>\<close>]], of _ "map rev"]
  have "map rev (rev (Dec \<BB>\<^sub>F X x)) \<in> lists (rev ` (\<BB>\<^sub>F X))"
    unfolding lists_image by blast
  from code.code_unique_dec'[OF code.code_rev_code[OF free_basis_code] this]
  show ?thesis
    unfolding \<open>concat (map rev (rev (Dec \<BB>\<^sub>F X x))) = rev x\<close>.
qed

lemma rev_hd_dec_last_eq[reversal_rule]: assumes "x \<in> X" and  "x \<noteq> \<epsilon>" shows
  "rev (hd (Dec (rev ` (\<BB>\<^sub>F X)) (rev x))) = last (Dec \<BB>\<^sub>F X x)"
proof-
  have "rev (Dec \<BB>\<^sub>F X x) \<noteq> \<epsilon>"
    using \<open>x \<in> X\<close> basis_gen_hull_free dec_nemp'[OF \<open>x \<noteq> \<epsilon>\<close>] by blast
  show ?thesis
    unfolding hd_rev rev_dec[OF free_gen_in[OF \<open>x \<in> X\<close>]] hd_map[OF \<open>rev (Dec \<BB>\<^sub>F X x) \<noteq> \<epsilon>\<close>]
    by simp
qed

lemma rev_hd_dec_last_eq'[reversal_rule]: assumes "x \<in> X" and  "x \<noteq> \<epsilon>" shows
  "(hd (Dec (rev ` (\<BB>\<^sub>F X)) (rev x))) = rev (last (Dec \<BB>\<^sub>F X x))"
  using assms(1) assms(2) rev_hd_dec_last_eq rev_swap by blast

section \<open>Lists as the free hull of singletons\<close>

text\<open>A crucial property of free monoids of words is that they can be seen as lists over the free basis,
instead as lists over the original alphabet.\<close>

abbreviation sings where "sings B \<equiv> {[b] | b. b \<in> B}"

term "Set.filter P A"

lemma sings_image: "sings B =  (\<lambda> x. [x]) ` B"
  using Setcompr_eq_image.

lemma lists_sing_map_concat_ident: "xs \<in> lists (sings B) \<Longrightarrow> xs = map (\<lambda> x. [x]) (concat xs)"
  by (induct xs, simp, auto)

lemma code_sings: "code (sings B)"
proof
  fix xs ys assume xs: "xs \<in> lists (sings B)" and ys: "ys \<in> lists (sings B)"
    and eq: "concat xs = concat ys"
  from lists_sing_map_concat_ident[OF xs, unfolded eq]
  show "xs = ys" unfolding  lists_sing_map_concat_ident[OF ys, symmetric].
qed

lemma sings_gen_lists: "\<langle>sings B\<rangle> = lists B"
  unfolding hull_concat_lists
proof(intro equalityI subsetI, standard)
  fix xs
  show "xs \<in> concat ` lists (sings B) \<Longrightarrow> \<forall>x\<in>set xs. x \<in> B"
    by force
  assume "xs \<in> lists B"
  hence "map (\<lambda>x. x # \<epsilon>) xs \<in> lists (sings B)"
    by force
  from imageI[OF this, of concat]
  show "xs \<in> concat ` lists (sings B)"
    unfolding concat_map_sing_ident[of xs].
qed

lemma sing_gen_lists: "lists {x} = \<langle>{[x]}\<rangle>"
  using sings_gen_lists[of "{x}"] by simp

lemma bin_gen_lists: "lists {x, y} = \<langle>{[x],[y]}\<rangle>"
  using sings_gen_lists[of "{x,y}"] unfolding Setcompr_eq_image by simp

lemma "sings B = \<BB>\<^sub>F (lists B)"
  using code.code_free_basis_hull[OF code_sings, of B, unfolded sings_gen_lists].

lemma map_sings: "xs \<in> lists B \<Longrightarrow> map (\<lambda>x. x # \<epsilon>) xs \<in> lists (sings B)"
  by (induct xs) auto

lemma dec_sings: "xs \<in> lists B \<Longrightarrow> Dec (sings B) xs = map (\<lambda> x. [x]) xs"
  using code.code_unique_dec'[OF code_sings, of "map (\<lambda> x. [x]) xs" B, OF map_sings]
  unfolding concat_map_sing_ident.

lemma sing_lists_exp: assumes "ws \<in> lists {x}"
  obtains k where "ws = [x]\<^sup>@k"
  using  unique_letter_wordE''[OF assms[folded in_lists_conv_set_subset]].

lemma sing_lists_exp_len: "ws \<in> lists {x} \<Longrightarrow> [x]\<^sup>@\<^bold>|ws\<^bold>| = ws"
  by  (induct ws, auto)

lemma sing_lists_exp_count: "ws \<in> lists {x} \<Longrightarrow> [x]\<^sup>@(count_list ws x) = ws"
  by  (induct ws, auto)

lemma sing_set_pow_count_list: "set ws \<subseteq> {a} \<Longrightarrow> [a]\<^sup>@(count_list ws a) = ws"
  unfolding in_lists_conv_set_subset using  sing_lists_exp_count.

lemma sing_set_pow: "set ws \<subseteq> {a} \<Longrightarrow> [a]\<^sup>@\<^bold>|ws\<^bold>| = ws"
  by auto

lemma count_sing_exp[simp]: "count_list ([a]\<^sup>@k) a = k"
  by (induct k, simp_all)

lemma count_sing_exp'[simp]: "count_list ([a]) a = 1"
  by simp

lemma count_sing_distinct[simp]: "a \<noteq> b \<Longrightarrow> count_list ([a]\<^sup>@k) b = 0"
  by (induct k, simp, auto)

lemma count_sing_distinct'[simp]: "a \<noteq> b \<Longrightarrow> count_list ([a]) b = 0"
  by simp

lemma sing_letter_imp_prim: assumes "count_list w a = 1" shows "primitive w"
proof
  fix r k
  assume "r \<^sup>@ k = w"
  have "count_list w a = k * count_list r a"
    by (simp only: count_list_pow flip: \<open>r \<^sup>@ k = w\<close>)
  then show "k = 1"
    unfolding \<open>count_list w a = 1\<close> by simp
qed

lemma prim_abk: "a \<noteq> b \<Longrightarrow> primitive ([a] \<cdot> [b] \<^sup>@ k)"
  by (intro sing_letter_imp_prim[of _ a]) simp

lemma sing_code: "x \<noteq> \<epsilon> \<Longrightarrow> code {x}"
proof (rule code.intro)
  fix xs ys
  assume "x \<noteq> \<epsilon>" "xs \<in> lists {x}" "ys \<in> lists {x}" "concat xs = concat ys"
  show "xs = ys"
    using \<open>concat xs = concat ys\<close>
      [unfolded concat_sing_list_pow'[OF \<open>xs \<in> lists {x}\<close>]
        concat_sing_list_pow'[OF \<open>ys \<in> lists {x}\<close>]
        eq_pow_exp[OF \<open>x \<noteq> \<epsilon>\<close>]]
      sing_lists_exp_len[OF \<open>xs \<in> lists {x}\<close>]
      sing_lists_exp_len[OF \<open>ys \<in> lists {x}\<close>] by argo
qed

lemma sings_card: "card A = card (sings A)"
  by(rule bij_betw_same_card, rule bij_betwI'[of _ "\<lambda>x. [x]"], auto)

lemma sings_finite: "finite A = finite (sings A)"
  by(rule bij_betw_finite, rule bij_betwI'[of _ "\<lambda>x. [x]"], auto)

lemma sings_conv: "A = B \<longleftrightarrow> sings A = sings B"
proof(standard, simp)
  have "\<And>x A B. sings A = sings B \<Longrightarrow> x \<in> A \<Longrightarrow> x \<in> B"
  proof-
    fix x :: "'b" and A B
    assume "sings A = sings B" "x \<in> A"
    hence "[x] \<in> sings B"
      using \<open>sings A = sings B\<close> by blast
    thus "x \<in> B"
      by blast
  qed
  from this[of A B] this[of B A, OF sym]
  show "sings A = sings B \<Longrightarrow> A = B"
    by blast
qed

section \<open>Various additional lemmas\<close>

subsection \<open>Roots of binary set\<close>

lemma two_roots_code: assumes "x \<noteq> \<epsilon>" and  "y \<noteq> \<epsilon>" shows "code {\<rho> x, \<rho> y}"
  using assms
proof (cases "\<rho> x = \<rho> y")
  assume "\<rho> x = \<rho> y"
  thus "code {\<rho> x, \<rho> y}" using sing_code[OF primroot_nemp[OF \<open>x \<noteq> \<epsilon>\<close>]] by simp
next
  assume "\<rho> x \<noteq> \<rho> y"
  hence "\<rho> x \<cdot> \<rho> y \<noteq> \<rho> y \<cdot> \<rho> x"
    using comm_prim[OF primroot_prim[OF \<open>x \<noteq> \<epsilon>\<close>] primroot_prim[OF \<open>y \<noteq> \<epsilon>\<close>]] by blast
  thus "code {\<rho> x, \<rho> y}"
    by (simp add: bin_code_code)
qed

lemma primroot_in_set_dec: assumes "x \<noteq> \<epsilon>" and  "y \<noteq> \<epsilon>" shows "\<rho> x \<in> set (Dec {\<rho> x, \<rho> y} x)"
proof-
  obtain k where "concat ([\<rho> x]\<^sup>@k) = x" "0 < k"
    using primroot_expE
      concat_sing_pow[symmetric, of "\<rho> x"] by metis
  from code.code_unique_dec'[OF two_roots_code[OF assms], of "[\<rho> x]\<^sup>@k", unfolded \<open>concat ([\<rho> x]\<^sup>@k) = x\<close>]
  have "Dec {\<rho> x, \<rho> y} x = [\<rho> x]\<^sup>@k"
    using insertI1 sing_pow_lists by metis
  show ?thesis
    unfolding \<open>Dec {\<rho> x, \<rho> y} x = [\<rho> x]\<^sup>@k\<close> using \<open>0 < k\<close> by simp
qed

lemma primroot_dec: assumes "x \<cdot> y \<noteq> y \<cdot> x"
  shows "(Dec {\<rho> x, \<rho> y} x) = [\<rho> x]\<^sup>@e\<^sub>\<rho> x" "(Dec {\<rho> x, \<rho> y} y) = [\<rho> y]\<^sup>@e\<^sub>\<rho> y"
  by (simp_all add: binary_code.intro[OF assms] binary_code.primroot_dec)

lemma (in binary_code) bin_roots_sings_code: "non_overlapping {Dec {\<rho> u\<^sub>0, \<rho> u\<^sub>1} u\<^sub>0, Dec {\<rho> u\<^sub>0, \<rho> u\<^sub>1} u\<^sub>1}"
  using code_roots_non_overlapping unfolding primroot_dec  by force

subsection Other

lemma bin_count_one_decompose: assumes "ws \<in> lists {x,y}" and "x \<noteq> y" and  "count_list ws y = 1"
  obtains k m where "[x]\<^sup>@k \<cdot> [y] \<cdot> [x]\<^sup>@m = ws"
proof-
  have "ws \<notin> [x]*"
    using count_sing_distinct[OF \<open>x \<noteq> y\<close>] \<open>count_list ws y = 1\<close> unfolding root_def by force
  from distinct_letter_in[OF this]
  obtain ws' k  b where "[x]\<^sup>@k \<cdot> [b] \<cdot> ws' = ws" and "b \<noteq> x" by blast
  hence "b = y"
    using \<open>ws \<in> lists {x,y}\<close>  by force
  have "ws' \<in> lists {x,y}"
    using \<open>ws \<in> lists {x,y}\<close>[folded \<open>[x]\<^sup>@k \<cdot> [b] \<cdot> ws' = ws\<close>] by simp
  have "count_list ws' y = 0"
    using arg_cong[OF \<open>[x]\<^sup>@k \<cdot> [b] \<cdot> ws' = ws\<close>, of "\<lambda> x. count_list x y"]
    unfolding count_list_append \<open>count_list ws y = 1\<close> \<open>b = y\<close>  by force
  from sing_lists_exp[OF bin_lists_count_zero'[OF \<open>ws' \<in> lists {x,y}\<close> this]]
  obtain m where "ws' = [x]\<^sup>@m".
  from that[OF \<open>[x]\<^sup>@k \<cdot> [b] \<cdot> ws' = ws\<close>[unfolded this \<open>b = y\<close>]]
  show thesis.
qed

lemma bin_count_one_conjug: assumes "ws \<in> lists {x,y}" and "x \<noteq> y" and "count_list ws y = 1"
  shows "ws \<sim> [x]\<^sup>@(count_list ws x) \<cdot> [y]"
proof-
  obtain e1 e2 where "[x]\<^sup>@e1 \<cdot> [y] \<cdot> [x]\<^sup>@e2 = ws"
    using bin_count_one_decompose[OF assms].
  from conjugI'[of "[x] \<^sup>@ e1 \<cdot> [y]" "[x]\<^sup>@e2", unfolded rassoc this]
  have "ws \<sim> [x]\<^sup>@(e2 + e1) \<cdot> [y]"
    unfolding add_exps rassoc.
  moreover have "count_list ([x]\<^sup>@(e2 + e1) \<cdot> [y]) x = e2 + e1"
    using \<open>x \<noteq> y\<close> by simp
  ultimately show ?thesis
    by (simp add: count_list_conjug)
qed

lemma bin_prim_long_set: assumes "ws \<in> lists {x,y}" and "primitive ws" and  "2 \<le> \<^bold>|ws\<^bold>|"
  shows "set ws = {x,y}"
proof-
  have "\<not> set ws \<subseteq> {c}" for c
    using \<open>primitive ws\<close> pow_nemp_imprim \<open>2 \<le> \<^bold>|ws\<^bold>|\<close>
      sing_lists_exp_len[folded in_lists_conv_set_subset] by metis
  then show "set ws = {x,y}"
    unfolding subset_singleton_iff using \<open>ws \<in> lists {x,y}\<close>[folded in_lists_conv_set_subset] doubleton_subset_cases by metis
qed

lemma bin_prim_long_pref: assumes "ws \<in> lists {x,y}" and "primitive ws" and  "2 \<le> \<^bold>|ws\<^bold>|"
  obtains ws' where "ws \<sim> ws'" and "[x,y] \<le>p ws'"
proof-
  from pow_nemp_imprim[OF \<open>2 \<le> \<^bold>|ws\<^bold>|\<close>, of "[x]"] sing_lists_exp_len[of ws x]
  have "\<not> ws \<in> lists {x}"
    using \<open>primitive ws\<close> \<open>2 \<le> \<^bold>|ws\<^bold>|\<close> by fastforce
  hence "x \<noteq> y"
    using \<open>ws \<in> lists {x,y}\<close> by fastforce
  from switch_fac[OF \<open>x \<noteq> y\<close> bin_prim_long_set[OF assms]]
  show thesis
    using \<open>2 \<le> \<^bold>|ws\<^bold>|\<close> rotate_into_pos_sq[of \<epsilon> "[x,y]" ws thesis, unfolded emp_simps, OF \<open>[x, y] \<le>f ws \<cdot> ws\<close> _ _ that, of id]
    by force
qed

end
