(*  Title:      Binary Code Imprimitive
    File:       Binary_Code_Imprimitive.Binary_Code_Imprimitive
    Author:     Štěpán Holub, Charles University
    Author:     Martin Raška, Charles University

Part of Combinatorics on Words Formalized. See https://gitlab.com/formalcow/combinatorics-on-words-formalized/
*)

theory Binary_Code_Imprimitive
  imports
    Combinatorics_Words_Graph_Lemma.Glued_Codes
    Binary_Square_Interpretation
begin

text \<open>This theory focuses on the characterization of imprimitive words which are concatenations
of copies of  two words (forming a binary code).
We follow the article @{cite lerest} (mainly Th\'eor\`eme 2.1 and Lemme 3.1),
while substantially optimizing the proof. See also @{cite spehner} for an earlier result on this question,
and @{cite Manuch} for another proof.\<close>

section \<open>General primitivity not preserving codes\<close>

context code

begin

text \<open>
Two nontrivially conjugate elements generated by a code induce a disjoint interpretation.\<close>

lemma shift_disjoint:
  assumes  "ws \<in> lists \<C>" and "ws' \<in> lists \<C>" and  "z \<notin> \<langle>\<C>\<rangle>" and "z \<cdot> concat ws = concat ws' \<cdot> z"
    "us \<le>p ws\<^sup>@n" and "vs \<le>p ws'\<^sup>@n"
  shows "z \<cdot> concat us \<noteq> concat vs"
  using \<open>z \<notin> \<langle>\<C>\<rangle>\<close>
proof (elim contrapos_nn)
  assume "z \<cdot> concat us = concat vs"
  have "z \<noteq> \<epsilon>"
    using \<open>z \<notin> \<langle>\<C>\<rangle>\<close> by blast
  obtain us' where "ws\<^sup>@n = us \<cdot> us'"
    using  prefixE[OF \<open>us \<le>p ws\<^sup>@n\<close>].
  obtain vs' where "ws'\<^sup>@n = vs \<cdot> vs'"
    using  prefixE[OF \<open>vs \<le>p ws'\<^sup>@n\<close>].
  from conjug_pow[OF \<open>z \<cdot> concat ws = concat ws' \<cdot> z\<close>[symmetric], symmetric]
  have "z \<cdot> concat (ws\<^sup>@n)  = concat (ws'\<^sup>@n) \<cdot> z"
    unfolding concat_pow.
  from this[ unfolded \<open>ws\<^sup>@n = us \<cdot> us'\<close> \<open>ws'\<^sup>@n = vs \<cdot> vs'\<close> concat_morph rassoc
      \<open>z \<cdot> concat us = concat vs\<close>[symmetric] cancel]
  have "concat vs' \<cdot> z = concat us'"..
  show "z \<in> \<langle>\<C>\<rangle>"
  proof (rule stability)
    have "us \<in> lists \<C>" and "us' \<in> lists \<C>" and "vs \<in> lists \<C>" and "vs' \<in> lists \<C>"
      using \<open>ws \<in> lists \<C>\<close> \<open>ws' \<in> lists \<C>\<close>  \<open>ws'\<^sup>@n = vs \<cdot> vs'\<close> \<open>ws\<^sup>@n = us \<cdot> us'\<close>
      by inlists
    thus "z \<cdot> concat us \<in> \<langle>\<C>\<rangle>" and "concat vs' \<in> \<langle>\<C>\<rangle>" and "concat us \<in> \<langle>\<C>\<rangle>" and "concat vs' \<cdot> z \<in> \<langle>\<C>\<rangle>"
      unfolding \<open>concat vs' \<cdot> z = concat us'\<close> \<open>z \<cdot> concat us = concat vs\<close>
      by (simp_all add: concat_in_hull')
  qed
qed

text\<open>This in particular yields a disjoint extendable interpretation of any prefix\<close>

lemma  shift_interp:
  assumes  "ws \<in> lists \<C>" and "ws' \<in> lists \<C>"  and "z \<notin> \<langle>\<C>\<rangle>" and
    conjug: "z \<cdot> concat ws = concat ws' \<cdot> z" and "\<^bold>|z\<^bold>| \<le> \<^bold>|concat ws'\<^bold>|"
    and "us \<le>p ws" and "us \<noteq> \<epsilon>"
  obtains p s vs ps where
     "p us s \<sim>\<^sub>\<D> vs" and "vs \<in> lists \<C>"
     and "s \<le>p concat (us\<inverse>\<^sup>>(ws \<cdot> ws))" and "p \<le>s concat ws" \<comment> \<open>extendable\<close>
     and "ps \<cdot> vs \<le>p ws' \<cdot> ws'" and "concat ps \<cdot> p = z"
proof-
  have "ws' \<cdot> ws' \<in> lists \<C>"
    using \<open>ws' \<in> lists \<C>\<close> by inlists
  have "concat us \<noteq> \<epsilon>"
    using \<open>us \<noteq> \<epsilon>\<close> unfolding code_concat_eq_emp_iff[OF pref_in_lists[OF \<open>us \<le>p ws\<close> \<open>ws \<in> lists \<C>\<close>]].
  have "\<^bold>|concat ws'\<^bold>| = \<^bold>|concat ws\<^bold>|"
    using lenarg[OF conjug, unfolded lenmorph] by linarith
  have "z \<cdot> concat(ws \<cdot> ws) = concat (ws' \<cdot> ws') \<cdot> z"
    unfolding rassoc concat_morph conjug[symmetric] unfolding lassoc cancel_right
    using conjug.
  hence "concat (ws' \<cdot> ws') \<le>p z \<cdot> concat (ws \<cdot> ws)"
    by blast
  have "z \<cdot> concat ws \<le>p concat (ws' \<cdot> ws')"
    unfolding concat_morph conjug pref_cancel_conv using eq_le_pref[OF conjug  \<open>\<^bold>|z\<^bold>| \<le> \<^bold>|concat ws'\<^bold>|\<close>].
  from prefixE[OF pref_shorten[OF pref_concat_pref[OF \<open>us \<le>p ws\<close>] this], unfolded rassoc]
  obtain su where fac_u[symmetric]: "concat (ws' \<cdot> ws') = z \<cdot> concat us \<cdot> su".

  from obtain_fac_interp[OF fac_u \<open>concat us \<noteq> \<epsilon>\<close>]
  obtain ps ss' p s vs where "p (concat us) s \<sim>\<^sub>\<I> vs" and
    "ps \<cdot> vs \<cdot> ss' = ws' \<cdot> ws'" and "concat ps \<cdot> p = z" and "s \<cdot> concat ss' = su".
  note fac_interpD[OF \<open>p (concat us) s \<sim>\<^sub>\<I> vs\<close>]

  let ?ss = "us\<inverse>\<^sup>>(ws \<cdot> ws)"
  have "us \<cdot> ?ss = ws \<cdot> ws"
    using \<open>us \<le>p ws\<close> by auto

  have "ps \<cdot> vs \<le>p ws' \<cdot> ws'"
    unfolding \<open>ps \<cdot> vs \<cdot> ss' = ws' \<cdot> ws'\<close>[symmetric] lassoc using triv_pref.

  hence "vs \<in> lists \<C>"
    using \<open>ws'\<in> lists \<C>\<close>
    by inlists

  have "s \<le>p concat ?ss"
    using \<open>concat (ws' \<cdot> ws') \<le>p z \<cdot> concat (ws \<cdot> ws)\<close>
    unfolding arg_cong[OF \<open>ps \<cdot> vs \<cdot> ss' = ws' \<cdot> ws'\<close>, of concat, symmetric] \<open>concat ps \<cdot> p = z\<close>[symmetric]
              arg_cong[OF \<open>us \<cdot> ?ss = ws \<cdot> ws\<close>, of concat, symmetric]
    unfolding concat_morph rassoc pref_cancel_conv
              \<open>p \<cdot> concat us \<cdot> s = concat vs\<close>[symmetric]
    using append_prefixD by auto

  have "\<^bold>|p\<^bold>| \<le> \<^bold>|concat ws\<^bold>|"
    using  \<open>\<^bold>|z\<^bold>| \<le> \<^bold>|concat ws'\<^bold>|\<close>[folded lenarg[OF \<open>concat ps \<cdot> p = z\<close>], unfolded \<open>\<^bold>|concat ws'\<^bold>| = \<^bold>|concat ws\<^bold>|\<close>]
    by simp

  with eqd[reversed, OF conjug[folded \<open>concat ps \<cdot> p = z\<close>, unfolded lassoc, symmetric] this]
  have "p \<le>s concat ws"
    by blast

  have disjoint: "p \<cdot> concat us' \<noteq> concat vs'" if "us' \<le>p us" "vs' \<le>p vs" for us' vs'
  proof
    have "us' \<le>p ws \<cdot> ws"
      using \<open>us \<le>p ws\<close> \<open>us' \<le>p us\<close> by auto
    have "ps \<cdot> vs' \<le>p ws' \<cdot> ws'"
      using \<open>vs' \<le>p vs\<close> \<open>ps \<cdot> vs \<le>p ws' \<cdot> ws'\<close> pref_trans same_prefix_prefix by metis
    assume "p \<cdot> concat us' = concat vs'"
    hence "z \<cdot> concat us' = concat (ps \<cdot> vs')"
      unfolding concat_morph \<open>concat ps \<cdot> p = z\<close>[symmetric] rassoc cancel.
    thus False
      using shift_disjoint[OF \<open>ws \<in> lists \<C>\<close> \<open>ws' \<in> lists \<C>\<close> \<open>z \<notin> \<langle>\<C>\<rangle>\<close>
          \<open>z \<cdot> concat ws = concat ws' \<cdot> z\<close> \<open>us' \<le>p ws \<cdot> ws\<close>[folded pow_two] \<open>ps \<cdot> vs' \<le>p ws' \<cdot> ws'\<close>[folded pow_two]] by fast
  qed
  from disjoint[of \<epsilon> \<epsilon>]
  have "p \<noteq> \<epsilon>" by blast
  have "s \<noteq> \<epsilon>"
    using \<open>p \<cdot> concat us \<cdot> s = concat vs\<close> disjoint by auto

  from disjoint_interpI[OF \<open>p (concat us) s \<sim>\<^sub>\<I> vs\<close>] disjoint
  have "p us s \<sim>\<^sub>\<D> vs"
    by blast

  from that[OF this \<open>vs \<in> lists \<C>\<close>
       \<open>s \<le>p concat ?ss\<close> \<open>p \<le>s concat ws\<close> \<open>ps \<cdot> vs \<le>p ws' \<cdot> ws'\<close> \<open>concat ps \<cdot> p = z\<close> ]
  show thesis.
qed

text\<open>The conditions are in particular met by imprimitivity witnesses\<close>

lemma imprim_witness_shift:
  assumes "ws \<in> lists \<C>" and "primitive ws" and "\<not> primitive (concat ws)"
  obtains z n  where "concat ws = z\<^sup>@n" "z \<notin> \<langle>\<C>\<rangle>" and
    "z \<cdot> concat ws = concat ws \<cdot> z" and "\<^bold>|z\<^bold>| < \<^bold>|concat ws\<^bold>|" and "2 \<le> n"
proof-
  have "concat ws \<noteq> \<epsilon>"
    using \<open>primitive ws\<close>  emp_concat_emp'[OF \<open>ws \<in> lists \<C>\<close>] emp_not_prim by blast
  obtain z n where [symmetric]: "z\<^sup>@n = concat ws" and "2 \<le> n"
    using not_prim_primroot_expE[OF \<open>\<not> primitive (concat ws)\<close>] by metis

  hence "z \<noteq> \<epsilon>"
    using \<open>concat ws \<noteq> \<epsilon>\<close>  by force

  have "z \<notin> \<langle>\<C>\<rangle>"
  proof
    assume "z \<in> \<langle>\<C>\<rangle>"
    then obtain zs where "zs \<in> lists \<C>" and "concat zs = z"
      using hull_concat_lists0 by blast
    from is_code[OF \<open>ws \<in> lists \<C>\<close> pow_in_lists[OF \<open>zs \<in> lists \<C>\<close>],
        unfolded concat_pow \<open>concat ws = z\<^sup>@n\<close> \<open>concat zs = z\<close>, of n]
    show False
      using \<open>primitive ws\<close> \<open>2 \<le> n\<close> pow_nemp_imprim by blast
  qed

  have "\<^bold>|z\<^bold>| < \<^bold>|concat ws\<^bold>|"
    unfolding lenarg[OF \<open>concat ws = z\<^sup>@n\<close>, unfolded lenmorph pow_len]
    using nemp_len[OF \<open>z \<noteq> \<epsilon>\<close>] \<open>2 \<le> n\<close> by simp

  from that[OF \<open>concat ws = z\<^sup>@n\<close> \<open>z \<notin> \<langle>\<C>\<rangle>\<close> _  this \<open>2 \<le> n\<close>]
  show thesis
    unfolding \<open>concat ws = z\<^sup>@n\<close>  pow_comm by blast

qed

end

section \<open>Covered uniform square\<close>

\<comment> \<open>Showing that two noncommuting words of the same length do not admit a non-trivial interpretation\<close>

lemma cover_xy_xxx: assumes "\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|" and "p \<cdot> x \<cdot> y \<cdot> s = x \<cdot> x \<cdot> x"
  shows "x = y"
  using append_assoc assms(1) assms(2) eq_le_pref le_refl long_pref lq_triv prefI pref_comm_eq' by metis

lemma cover_xy_yyy: assumes "\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|" and eq: "p \<cdot> x \<cdot> y \<cdot> s = y \<cdot> y \<cdot> y"
  shows "x = y"
  using cover_xy_xxx[reversed, unfolded rassoc, OF \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close>[symmetric] eq, symmetric].

lemma cover_xy_xxy: assumes "\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|" and "s \<noteq> \<epsilon>" and eq: "p \<cdot> x \<cdot> y \<cdot> s = x \<cdot> x \<cdot> y"
  shows "x = y"
proof-
  have "\<^bold>|p\<^bold>| < \<^bold>|x\<^bold>|"
    using lenarg[OF eq] nemp_pos_len[OF \<open>s \<noteq> \<epsilon>\<close>] unfolding lenmorph by linarith
  then obtain t where x: "x = p \<cdot> t" and "t \<noteq> \<epsilon>"
    using eqd[OF eq] by force
  from eq[unfolded this rassoc cancel]
  have "p \<cdot> t = t \<cdot> p"
    by mismatch
  hence "x \<le>p t \<cdot> x"
    unfolding x by auto
  from eq[unfolded x]
  have "y \<le>p t \<cdot> y"
    using \<open>p \<cdot> t = t \<cdot> p\<close> \<open>p \<cdot> t \<cdot> y \<cdot> s = t \<cdot> p \<cdot> t \<cdot> y\<close> pref_cancel' suf_marker_per_root triv_pref by metis
  show "x = y"
    using same_len_nemp_root_eq[OF per_rootI[OF \<open>x \<le>p t \<cdot> x\<close> \<open>t \<noteq> \<epsilon>\<close>]
          per_rootI[OF \<open>y \<le>p t \<cdot> y\<close> \<open>t \<noteq> \<epsilon>\<close>]  \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close>].
qed

lemma cover_xy_xyy: assumes "\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|" and "p \<noteq> \<epsilon>" and eq: "p \<cdot> x \<cdot> y \<cdot> s = x \<cdot> y \<cdot> y"
  shows "x = y"
  using cover_xy_xxy[reversed, unfolded rassoc, OF assms(1)[symmetric] assms(2) eq]..

lemma cover_xy_yyx: assumes "\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|" and eq: "p \<cdot> x \<cdot> y \<cdot> s = y \<cdot> y \<cdot> x"
  shows "x = y"
proof-
  have "\<^bold>|p\<^bold>| \<le> \<^bold>|y\<^bold>|"
    using lenarg[OF eq]  unfolding lenmorph by linarith
  then obtain t where y: "y = p \<cdot> t"
    using eqd[OF eq] by force
  from eqd_eq[OF _  \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close>[unfolded y swap_len[of p]], unfolded rassoc]  eq[unfolded this rassoc cancel]
  have x: "x = t \<cdot> p" by blast
  from eq[unfolded x y rassoc cancel]
  have "p \<cdot> t = t \<cdot> p"
    by mismatch
  thus "x = y"
    unfolding x y..
qed

lemma cover_xy_yxx: assumes "\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|" and eq: "p \<cdot> x \<cdot> y \<cdot> s = y \<cdot> x \<cdot> x"
  shows "x = y"
  using cover_xy_yyx[reversed, unfolded rassoc, OF assms(1)[symmetric] eq]..

lemma cover_xy_xyx: assumes "\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|" and "p \<noteq> \<epsilon>" and "s \<noteq> \<epsilon>" and eq: "p \<cdot> x \<cdot> y \<cdot> s = x \<cdot> y \<cdot> x"
  shows "\<not> primitive (x \<cdot> y)"
proof
  assume "primitive (x \<cdot> y)"
  have "p \<cdot> (x \<cdot> y) \<cdot> (s \<cdot> y) = (x \<cdot> y) \<cdot> (x \<cdot> y)"
    unfolding lassoc eq[unfolded lassoc]..
  from prim_overlap_sqE[OF \<open>primitive (x \<cdot> y)\<close> this]
  show False
    using \<open>p \<noteq> \<epsilon>\<close> \<open>s \<noteq> \<epsilon>\<close> by blast
qed

lemma cover_xy_yxy: assumes "\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|" and "p \<noteq> \<epsilon>" and \<open>s \<noteq> \<epsilon>\<close> and eq: "p \<cdot> x \<cdot> y \<cdot> s = y \<cdot> x \<cdot> y"
  shows "\<not> primitive (x \<cdot> y)"
  using cover_xy_xyx[reversed, unfolded rassoc, OF assms(1)[symmetric] assms(3) assms(2) eq].

theorem uniform_square_interp: assumes "x\<cdot>y \<noteq> y\<cdot>x" and  "\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|" and "vs \<in> lists {x,y}"
  and "p  (x \<cdot> y)  s \<sim>\<^sub>\<I> vs" and "p \<noteq> \<epsilon>"
shows "\<not> primitive (x\<cdot>y)" and  "vs = [x,y,x] \<or> vs = [y,x,y]"
proof-
  note fac_interpD[OF \<open>p  (x \<cdot> y)  s \<sim>\<^sub>\<I> vs\<close>]

  have "vs \<noteq> \<epsilon>"
    using \<open>p \<cdot> (x \<cdot> y) \<cdot> s = concat vs\<close> assms(5) by force
  have "\<^bold>|p\<^bold>| < \<^bold>|x\<^bold>|"
    using prefix_length_less[OF \<open>p <p hd vs\<close>] lists_hd_in_set[OF \<open>vs \<noteq> \<epsilon>\<close> \<open>vs \<in> lists {x,y}\<close>]
     \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close>
    by fastforce
  have "\<^bold>|s\<^bold>| < \<^bold>|x\<^bold>|"
    using suffix_length_less[OF \<open>s <s last vs\<close>] \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close> lists_hd_in_set[reversed, OF \<open>vs \<noteq> \<epsilon>\<close> \<open>vs \<in> lists {x,y}\<close>]
    by fastforce
  have "\<^bold>|concat vs\<^bold>| = \<^bold>|x\<^bold>| * \<^bold>|vs\<^bold>|"
    using assms(2-3)
  proof (induction vs)
    case (Cons a vs)
    have "\<^bold>|a\<^bold>| = \<^bold>|x\<^bold>|" and "\<^bold>|a # vs\<^bold>| = Suc \<^bold>|vs\<^bold>|" and
      "\<^bold>|concat (a # vs)\<^bold>| = \<^bold>|a\<^bold>| + \<^bold>|concat vs\<^bold>|" and "\<^bold>|concat vs\<^bold>| = \<^bold>|x\<^bold>| * \<^bold>|vs\<^bold>|"
      using \<open>a#vs \<in> lists {x,y}\<close> \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close> Cons.IH Cons.prems by auto
    then show ?case by force
  qed simp
  note leneq = lenarg[OF \<open>p \<cdot> (x \<cdot> y) \<cdot> s = concat vs\<close>, unfolded this lenmorph \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close>[symmetric]]
  hence "\<^bold>|x\<^bold>| * \<^bold>|vs\<^bold>| < \<^bold>|x\<^bold>| * 4" and  "2 * \<^bold>|x\<^bold>| < \<^bold>|x\<^bold>| * \<^bold>|vs\<^bold>| "
    using \<open>\<^bold>|p\<^bold>| < \<^bold>|x\<^bold>|\<close> \<open>\<^bold>|s\<^bold>| < \<^bold>|x\<^bold>|\<close> nemp_pos_len[OF \<open>p \<noteq> \<epsilon>\<close>] by linarith+
  hence "\<^bold>|vs\<^bold>| = 3"
    by force
  hence "s \<noteq> \<epsilon>"
    using leneq \<open>\<^bold>|p\<^bold>| < \<^bold>|x\<^bold>|\<close> by force

  have "x \<noteq> y"
    using assms(1) by blast
  with \<open>\<^bold>|vs\<^bold>| = 3\<close> \<open>vs \<in> lists {x,y}\<close> \<open>p \<cdot> (x \<cdot> y) \<cdot> s = concat vs\<close>
  have "(\<not> primitive (x\<cdot>y)) \<and> (vs = [x,y,x] \<or> vs = [y,x,y])"
  proof(list_inspection, simp_all)
    assume "p \<cdot> x \<cdot> y \<cdot> s = x \<cdot> x \<cdot> x"
    from cover_xy_xxx[OF \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close> this]
    show False
      using \<open>x \<noteq> y\<close> by blast
  next
    assume "p \<cdot> x \<cdot> y \<cdot> s = x \<cdot> x \<cdot> y"
    from cover_xy_xxy[OF \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close> \<open>s \<noteq> \<epsilon>\<close> this]
    show False
      using \<open>x \<noteq> y\<close> by blast
  next
    assume "p \<cdot> x \<cdot> y \<cdot> s = x \<cdot> y \<cdot> x"
    from cover_xy_xyx[OF \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close> \<open>p \<noteq> \<epsilon>\<close> \<open>s \<noteq> \<epsilon>\<close> this]
    show "\<not> primitive (x \<cdot> y)"
      by blast
  next
    assume "p \<cdot> x \<cdot> y \<cdot> s = x \<cdot> y \<cdot> y"
    from cover_xy_xyy[OF \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close> \<open>p \<noteq> \<epsilon>\<close> this]
    show False
      using \<open>x \<noteq> y\<close> by blast
  next
    assume "p \<cdot> x \<cdot> y \<cdot> s = y \<cdot> x \<cdot> x"
    from cover_xy_yxx[OF \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close> this]
    show False
      using \<open>x \<noteq> y\<close> by blast
  next
    assume "p \<cdot> x \<cdot> y \<cdot> s = y \<cdot> x \<cdot> y"
    from cover_xy_yxy[OF \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close> \<open>p \<noteq> \<epsilon>\<close> \<open>s \<noteq> \<epsilon>\<close> this]
    show "\<not> primitive (x \<cdot> y)"
      by blast
  next
    assume "p \<cdot> x \<cdot> y \<cdot> s = y \<cdot> y \<cdot> x"
    from cover_xy_yyx[OF \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close> this]
    show False
      using \<open>x \<noteq> y\<close> by blast
  next
    assume "p \<cdot> x \<cdot> y \<cdot> s = y \<cdot> y \<cdot> y"
    from cover_xy_yyy[OF \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close> this]
    show False
      using \<open>x \<noteq> y\<close> by blast
  qed
  thus "\<not> primitive (x\<cdot>y)" "vs = [x,y,x] \<or> vs = [y,x,y]"
    by blast+
qed

subsection \<open>Primitivity (non)preserving uniform binary codes\<close>

\<comment> \<open>This in particular implies the following characterization of uniform binary primitive codes. Cf. V. Mitrana, Primitive morphisms, Information Processing Letters 64 (1997), 277--281\<close>

theorem bin_uniform_prim_morph:
  assumes "x \<cdot> y \<noteq> y \<cdot> x" and "\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|" and "primitive (x \<cdot> y)"
         and "ws \<in> lists {x,y}" and "2 \<le> \<^bold>|ws\<^bold>|"
         shows "primitive ws \<longleftrightarrow>  primitive (concat ws)"
proof (standard, rule ccontr)
  assume \<open>primitive ws\<close> and \<open>\<not> primitive (concat ws)\<close>
  from bin_prim_long_pref[OF \<open>ws \<in> lists {x,y}\<close> \<open>primitive ws\<close> \<open>2 \<le> \<^bold>|ws\<^bold>|\<close>]
  obtain ws' where "ws \<sim> ws'" "[x, y] \<le>p ws'".
  have "ws' \<in> lists {x,y}"
    using conjug_in_lists'[OF \<open>ws \<sim> ws'\<close> \<open>ws \<in> lists {x,y}\<close>].
  have "primitive ws'"
    using prim_conjug[OF \<open>primitive ws\<close> \<open>ws \<sim> ws'\<close>].
  have "\<not> primitive (concat ws')"
    using conjug_concat_prim_iff \<open>\<not> primitive (concat ws)\<close> \<open>ws \<sim> ws'\<close> by auto
  interpret code "{x,y}"
    using bin_code_code[OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close>].

  have "[x,y] \<noteq> \<epsilon>" by blast
  from imprim_witness_shift[OF \<open>ws' \<in> lists {x,y}\<close> \<open>primitive ws'\<close> \<open>\<not> primitive (concat ws')\<close>]
  obtain z n where "concat ws' = z \<^sup>@ n" "z \<notin> \<langle>{x, y}\<rangle>" "z \<cdot> concat ws' = concat ws' \<cdot> z" "\<^bold>|z\<^bold>| < \<^bold>|concat ws'\<^bold>|".
  from shift_interp[OF \<open>ws' \<in> lists {x,y}\<close> \<open>ws' \<in> lists {x,y}\<close> this(2-3) less_imp_le[OF this(4)] \<open>[x,y] \<le>p ws'\<close> \<open>[x,y] \<noteq> \<epsilon>\<close>]
  obtain p s vs ps where "p [x, y] s \<sim>\<^sub>\<D> vs" "vs \<in> lists {x, y}" "s \<le>p concat ([x, y]\<inverse>\<^sup>>(ws' \<cdot> ws'))"
        "p \<le>s concat ws'" "ps \<cdot> vs \<le>p ws' \<cdot> ws'" "concat ps \<cdot> p = z".
  from uniform_square_interp(1)[OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> \<open>\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|\<close> \<open>vs \<in> lists {x,y}\<close> _ _]
       \<open>primitive (x \<cdot> y)\<close> disj_interpD[OF this(1), simplified] disj_interp_nemp(1)[OF this(1)]
  show False by force
qed (simp add: prim_concat_prim)

\<comment> \<open>A stronger version is implied by the following lemma.\<close>

lemma bin_uniform_imprim: assumes "x \<cdot> y \<noteq> y \<cdot> x" and "\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|" and "\<not> primitive (x \<cdot> y)"
  shows "primitive x"
proof-
  have "x \<cdot> y \<noteq> \<epsilon>" and "x \<noteq> \<epsilon>" and "y \<noteq> \<epsilon>"
    using \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> by blast+
  from not_prim_expE[OF \<open>\<not> primitive (x \<cdot> y)\<close> \<open>x \<cdot> y \<noteq> \<epsilon>\<close>]
  obtain z k where "primitive z" and  "2 \<le> k" and "z\<^sup>@k = x \<cdot> y".
  hence "0 < k"
    by simp
  from split_pow[OF \<open>z\<^sup>@k = x \<cdot> y\<close>[symmetric] \<open>0 < k\<close> \<open>y \<noteq> \<epsilon>\<close>]
  obtain u v l m where [symmetric]: "z = u \<cdot> v" and "v \<noteq> \<epsilon>" "x = (u\<cdot>v) \<^sup>@ l \<cdot> u"  "y = (v \<cdot> u) \<^sup>@ m \<cdot> v" "k = l + m + 1".
  have "u \<cdot> v \<noteq> v \<cdot> u"
    using  \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> unfolding \<open>x = (u\<cdot>v) \<^sup>@ l \<cdot> u\<close> \<open>y = (v \<cdot> u) \<^sup>@ m \<cdot> v\<close>
      shifts unfolding add_exps[symmetric] add.commute[of m] by force
  have "u \<noteq> \<epsilon>" and "v \<noteq> \<epsilon>" and "u \<noteq> v"
    using \<open>u \<cdot> v \<noteq> v \<cdot> u\<close> by blast+
  have "m = l" and "\<^bold>|u\<^bold>| = \<^bold>|v\<^bold>|"
    using almost_equal_equal[OF nemp_len[OF \<open>u \<noteq> \<epsilon>\<close>] nemp_len[OF \<open>v \<noteq> \<epsilon>\<close>], of l m] lenarg[OF \<open>x = (u\<cdot>v)\<^sup>@l \<cdot> u\<close>, unfolded \<open>\<^bold>|x\<^bold>| =\<^bold>|y\<^bold>|\<close>, unfolded lenarg[OF \<open>y = (v \<cdot> u) \<^sup>@ m \<cdot> v\<close>]]
    unfolding lenmorph pow_len lenarg[OF \<open>u \<cdot> v = z\<close>, symmetric] by algebra+
  from \<open>k = l + m + 1\<close>[folded Suc_eq_plus1, symmetric]
  have "l \<noteq> 0"
    using  \<open>2 \<le> k\<close>[folded \<open>Suc(l+m) = k\<close>, unfolded \<open>m = l\<close>] by force
  let ?w = "[u,v]\<^sup>@l \<cdot> [u]"
  have "?w \<in> lists {u,v}"
    by (induct l, simp_all)
  have "2 \<le> \<^bold>|?w\<^bold>|"
    using \<open>l \<noteq> 0\<close> unfolding lenmorph pow_len by fastforce
  have "concat ?w = x"
    using \<open>x = (u \<cdot> v) \<^sup>@ l \<cdot> u\<close> by simp
  from bin_uniform_prim_morph[OF \<open>u \<cdot> v \<noteq> v \<cdot> u\<close> \<open>\<^bold>|u\<^bold>| = \<^bold>|v\<^bold>|\<close> \<open>primitive z\<close>[folded \<open>u \<cdot> v = z\<close>] \<open>?w \<in> lists {u,v}\<close> \<open>2 \<le> \<^bold>|?w\<^bold>|\<close>]
  show "primitive x"
    unfolding \<open>concat ?w = x\<close> using alternate_prim[OF \<open>u \<noteq> v\<close>] by blast
qed


theorem bin_uniform_prim_morph':
  assumes "x \<cdot> y \<noteq> y \<cdot> x" and "\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|" and "primitive (x \<cdot> y) \<or> \<not> primitive x \<or> \<not> primitive y"
         and "ws \<in> lists {x,y}" and "2 \<le> \<^bold>|ws\<^bold>|"
       shows "primitive ws \<longleftrightarrow>  primitive (concat ws)"
  using bin_uniform_prim_morph[OF assms(1-2) _ assms(4-5)] bin_uniform_imprim[OF assms(1-2)]
        bin_uniform_imprim[OF assms(1-2)[symmetric], unfolded conjug_prim_iff'[of y]]
        assms(3) by blast

section \<open>The main theorem\<close>

subsection \<open>Imprimitive words with single y\<close>

text \<open>If the shorter word occurs only once, the result is straightforward from the parametric solution of the Lyndon-Schutzenberger
  equation.\<close>

lemma bin_imprim_single_y:
  assumes  non_comm: "x \<cdot> y \<noteq> y \<cdot> x" and
    "ws \<in> lists {x,y}" and
    "\<^bold>|y\<^bold>| \<le> \<^bold>|x\<^bold>|" and
    "2 \<le> count_list ws x" and
    "count_list ws y < 2" and
    "primitive ws" and
    "\<not> primitive (concat ws)"
  shows "ws \<sim> [x,x,y]" and "primitive x" and "primitive y"
proof-
  have "x \<noteq> y"
    using non_comm by blast
  have "count_list ws y \<noteq> 0"
  proof
    assume "count_list ws y = 0"
    from bin_lists_count_zero'[OF \<open>ws \<in> lists {x,y}\<close> this]
    have "ws \<in> lists {x}".
    from  prim_exp_one[OF \<open>primitive ws\<close> sing_lists_exp_count[OF this]]
    show False
      using \<open>2 \<le> count_list ws x\<close> by simp
  qed
  hence "count_list ws y = 1"
    using \<open>count_list ws y < 2\<close> by linarith

  from this bin_count_one_conjug[OF \<open>ws \<in> lists {x,y}\<close> _ this]
  have "ws \<sim> [x]\<^sup>@count_list ws x \<cdot> [y]"
    using  non_comm (1) by metis
  from conjug_concat_prim_iff[OF this]
  have "\<not> primitive (x\<^sup>@(count_list ws x) \<cdot> y)"
    using \<open>\<not> primitive (concat ws)\<close> by simp

  from not_prim_primroot_expE[OF this]
  obtain z l where [symmetric]: "z\<^sup>@l = x\<^sup>@(count_list ws x) \<cdot> y\<^sup>@1" and "2 \<le> l"
    unfolding pow_1.

  interpret LS_len_le x y "count_list ws x" 1 l z
    by (unfold_locales)
    (use \<open>2 \<le> count_list ws x\<close> \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> \<open> \<^bold>|y\<^bold>| \<le> \<^bold>|x\<^bold>|\<close>
     \<open>x \<^sup>@ count_list ws x \<cdot> y \<^sup>@ 1 = z \<^sup>@ l\<close> \<open>2 \<le> l\<close> in force)+

  from case_j2k1[OF \<open>2 \<le> count_list ws x\<close> refl]
  have "primitive x" and "primitive y" and "count_list ws x = 2" by blast+

  with \<open>ws \<sim> [x]\<^sup>@count_list ws x \<cdot> [y]\<close>[unfolded this(3) pow_two append_Cons append_Nil]
  show  "primitive x" and "primitive y" and "ws \<sim> [x,x,y]"
    by simp_all

qed

subsection \<open>Conjugate words\<close>

lemma bin_imprim_not_conjug:
  assumes "ws \<in> lists {x,y}" and
    "x \<cdot> y \<noteq> y \<cdot> x" and
    "2 \<le> \<^bold>|ws\<^bold>|" and
    "primitive ws" and
    "\<not> primitive (concat ws)"
  shows "\<not> x \<sim> y"
proof
  assume "x \<sim> y"
  hence "\<^bold>|x\<^bold>| = \<^bold>|y\<^bold>|" by force
  from bin_uniform_prim_morph[OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> this _ \<open>ws \<in> lists {x,y}\<close> \<open>2 \<le> \<^bold>|ws\<^bold>|\<close>]
  have "\<not> primitive (x\<cdot>y)"
    using \<open>primitive ws\<close> \<open>\<not> primitive (concat ws)\<close> by blast







  from Lyndon_Schutzenberger_conjug[OF \<open>x \<sim> y\<close> this]
  show False
    using \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> by blast
qed

subsection \<open>Square factor of the longer word and both words primitive (was all\_assms)\<close>

text\<open>The main idea of the proof is as follows: Imprimitivity of the concatenation yields
(at least) two overlapping factorizations into @{term "{x,y}"}.
Due to the presence of the square @{term "x \<cdot> x"}, these two can be synchronized, which yields that the
situation coincides with the canonical form.
\<close>

lemma bin_imprim_primitive:
  assumes "x \<cdot> y \<noteq> y \<cdot> x"
    and  "primitive x" and "primitive y"
    and "\<^bold>|y\<^bold>| \<le> \<^bold>|x\<^bold>|"
    and "ws \<in> lists {x, y}"
    and "primitive ws" and "\<not> primitive (concat ws)"
    and "[x, x] \<le>f ws \<cdot> ws"
  shows "ws \<sim> [x, x, y]"
proof-
  \<comment> \<open>Preliminaries\<close>
  have "x \<noteq> y"
    using assms(1) by blast
  have "\<^bold>|ws\<^bold>| \<noteq> 1"
    using  len_one_concat_in[OF \<open>ws \<in> lists {x, y}\<close>] \<open>\<not> primitive (concat ws)\<close> \<open>primitive x\<close> \<open>primitive y\<close>
    by blast
  with prim_nemp[OF \<open>primitive ws\<close>, THEN nemp_le_len]
  have "2 \<le> \<^bold>|ws\<^bold>|"
    by auto
  hence "\<^bold>|[x, x]\<^bold>| \<le> \<^bold>|ws\<^bold>|"
     by force
  have "\<not> x \<sim> y"
    by (rule bin_imprim_not_conjug) fact+
  have "primitive [x,x,y]"
    using \<open>x \<noteq> y\<close> by primitivity_inspection
  have "concat [x,x] = x \<cdot> x"
    by simp
  interpret xy: binary_code x y
    using \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> by (unfold_locales)

  \<comment> \<open>Rotate @{term ws} in order to obtain a list with a prefix @{term "[x\<cdot>x]"}\<close>
  obtain ws' where "ws \<sim> ws'" "[x,x] \<le>p ws'"
    using rotate_into_pos_sq[of \<epsilon> "[x,x]" _ thesis, unfolded emp_simps, OF \<open>[x, x] \<le>f ws \<cdot> ws\<close>
        le0 \<open>\<^bold>|[x, x]\<^bold>| \<le> \<^bold>|ws\<^bold>|\<close>] by blast

  have "ws' \<in> lists {x,y}" and "primitive ws'" and "\<not> primitive (concat ws')"
    using conjug_in_lists'[OF \<open>ws \<sim> ws'\<close> \<open>ws \<in> lists {x, y}\<close>]
      prim_conjug[OF \<open>primitive ws\<close>  \<open>ws \<sim> ws'\<close>]
      \<open>\<not> primitive (concat ws)\<close>[unfolded conjug_concat_prim_iff[OF \<open>ws \<sim> ws'\<close>]].
  have "2 \<le> \<^bold>|ws'\<^bold>|" and "[x,x] \<noteq> \<epsilon>" and "ws' \<noteq> \<epsilon>"
    using \<open>[x,x] \<le>p ws'\<close> unfolding prefix_def by auto
  have "concat ws' \<noteq> \<epsilon>"
    using \<open>primitive x\<close> \<open>[x,x] \<le>p ws'\<close> by (fastforce simp add: prefix_def)
  have "ws' \<cdot> ws' \<cdot> ws' \<in> lists {x, y}" and "ws' \<cdot> ws' \<in> lists {x, y}"
    using \<open>ws' \<in> lists {x,y}\<close> by inlists

  \<comment> \<open>The core of the proof\<close>
  have "ws' = [x,x,y]"
  proof(rule ccontr)
    assume "ws' \<noteq> [x,x,y]"

    from xy.imprim_witness_shift[OF \<open>ws' \<in> lists {x,y}\<close> \<open>primitive ws'\<close> \<open>\<not> primitive (concat ws')\<close>]
    obtain z n where con_ws: "concat ws' = z \<^sup>@ n" and "z \<notin> \<langle>{x, y}\<rangle>" and "z \<cdot> concat ws' = concat ws' \<cdot> z"
        and "\<^bold>|z\<^bold>| < \<^bold>|concat ws'\<^bold>|" and "2 \<le> n".
    have "0 < n"
        using \<open>2 \<le> n\<close> by simp
    from xy.shift_interp[OF \<open>ws' \<in> lists {x,y}\<close> \<open>ws' \<in> lists {x,y}\<close> \<open>z \<notin> \<langle>{x, y}\<rangle>\<close> \<open>z \<cdot> concat ws' = concat ws' \<cdot> z\<close>
            less_imp_le[OF \<open>\<^bold>|z\<^bold>| < \<^bold>|concat ws'\<^bold>|\<close>] \<open>[x,x] \<le>p ws'\<close> \<open>[x,x] \<noteq> \<epsilon>\<close>]
    obtain p s vs ps where dis: "p [x,x] s \<sim>\<^sub>\<D> vs" and \<open>vs \<in> lists {x, y}\<close> and
      "s \<le>p concat ([x,x]\<inverse>\<^sup>>(ws'\<cdot>ws'))" and "p \<le>s concat ws'" and "ps \<cdot> vs \<le>p ws' \<cdot> ws'" and "concat ps \<cdot> p = z".

    from disj_interp_nemp(1)[OF this(1)]
    have "p \<noteq> \<epsilon>" by simp

    have "p \<cdot> concat p1 \<noteq> concat p2" if  "p1 \<le>p [x, x]" and "p2 \<le>p vs" for p1 p2
        using \<open>p [x,x] s \<sim>\<^sub>\<D> vs\<close> disj_interpD1 that by blast

    have "ps \<in> lists {x,y}"
      using \<open>ps \<cdot> vs \<le>p ws' \<cdot> ws'\<close> \<open>ws' \<in> lists {x,y}\<close> \<open>ws' \<cdot> ws' \<in> lists {x, y}\<close> append_prefixD pref_in_lists by metis
    have "vs \<in> lists {x,y}"
      using \<open>ws' \<in> lists {x,y}\<close> pref_in_lists[OF \<open>ps \<cdot> vs \<le>p ws' \<cdot> ws'\<close>] by inlists
    have "[x,x]\<inverse>\<^sup>>(ws'\<cdot>ws') \<in> lists {x,y}"
      using \<open>ws' \<in> lists {x,y}\<close> by inlists
    have "p x \<cdot> x s \<sim>\<^sub>\<I> vs"
      using disj_interpD[OF \<open>p [x,x] s \<sim>\<^sub>\<D> vs\<close>] by simp

    interpret square_interp_ext x y p s vs
    proof (rule square_interp_ext.intro[OF square_interp.intro, unfolded square_interp_ext_axioms_def])
      show "(\<exists>pe. pe \<in> \<langle>{x, y}\<rangle> \<and> p \<le>s pe) \<and> (\<exists>se. se \<in> \<langle>{x, y}\<rangle> \<and> s \<le>p se)"
        using \<open>s \<le>p concat ([x,x]\<inverse>\<^sup>>(ws'\<cdot>ws'))\<close> \<open>p \<le>s concat ws'\<close>
         \<open>[x, x]\<inverse>\<^sup>>(ws' \<cdot> ws') \<in> lists {x, y}\<close> \<open>ws' \<in> lists {x, y}\<close> concat_in_hull' by meson
    qed fact+

\<comment> \<open>Establishing the connection between ws' = [x,x,y] and z = xp.\<close>

    define xp where "xp = x \<cdot> p"

    have "concat [x,x,y] = xp \<cdot> xp"
      by (simp add: xxy_root xp_def)

    hence "ws' \<cdot> [x,x,y] \<noteq> [x,x,y] \<cdot> ws'"
      using  comm_prim[OF \<open>primitive ws'\<close> \<open>primitive [x,x,y]\<close>] \<open>ws' \<noteq> [x,x,y] \<close>by force

    have "z \<cdot> xp \<noteq> xp \<cdot> z"
    proof
      assume "z \<cdot> xp = xp \<cdot> z"
      from comm_add_exp[symmetric, OF this[symmetric], of 2,
          THEN comm_add_exp, of n, unfolded pow_two]
      have "z\<^sup>@n \<cdot> xp \<cdot> xp = xp \<cdot> xp \<cdot> z\<^sup>@n"
        unfolding  rassoc.
      hence "concat ws' \<cdot> concat [x,x,y] = concat [x,x,y] \<cdot> concat ws'"
        unfolding con_ws \<open>concat [x,x,y] = xp \<cdot> xp\<close> rassoc  by simp
      from  xy.is_code[OF _ _ this[folded concat_morph]]
      have "ws' \<cdot> [x, x, y] = [x,x,y] \<cdot> ws'"
        using append_in_lists \<open>ws' \<in> lists {x,y}\<close> by simp
      thus False
        using \<open>ws' \<cdot> [x, x, y] \<noteq> [x,x,y] \<cdot> ws'\<close> by fastforce
    qed

    then interpret binary_code z xp
      by (unfold_locales)

    have "\<not> concat (ws' \<cdot> [x, x, y]) \<bowtie> concat ([x, x, y] \<cdot> ws')"
    proof (rule notI)
      assume "concat (ws' \<cdot> [x, x, y]) \<bowtie> concat ([x, x, y] \<cdot> ws')"
      from comm_comp_eq[OF this[unfolded concat_morph], unfolded \<open>concat [x,x,y] = xp \<cdot> xp\<close> con_ws]
      have "z \<^sup>@ n \<cdot> xp\<^sup>@Suc(Suc 0) = xp\<^sup>@Suc(Suc 0) \<cdot> z \<^sup>@ n"
        unfolding pow_Suc pow_zero emp_simps rassoc.
      from comm_drop_exps[OF this]
      show False
        using \<open>z \<cdot> xp \<noteq> xp \<cdot> z\<close> \<open>2 \<le> n\<close> by force
    qed

\<comment> \<open>How the xp/z mismatch is reflected by mismatch in lists {x,y}?\<close>
\<comment> \<open>Looking at the first occurrence of z:\<close>

    define lcp_ws where "lcp_ws = ws' \<cdot> [x,x,y] \<and>\<^sub>p [x,x,y] \<cdot> ws'"

    have "lcp_ws \<in> lists {x,y}"
      unfolding lcp_ws_def by inlists

    have lcp_xp_z: "concat (ws' \<cdot> [x,x,y]) \<and>\<^sub>p concat ([x,x,y] \<cdot> ws') = bin_lcp z (x \<cdot> p)"
      unfolding concat_morph con_ws \<open>concat [x,x,y] = xp \<cdot> xp\<close> add_exps[symmetric]
      using bin_lcp_pows[OF \<open>0 < n\<close>, of 2]
      unfolding pow_two pow_pos[OF \<open>0 < n\<close>] rassoc xp_def by force

    have "(concat lcp_ws) \<cdot> bin_lcp x y = bin_lcp z (x \<cdot> p)"
    proof (rule xy.bin_code_lcp_concat'[OF _ _  \<open>\<not> concat (ws' \<cdot> [x, x, y]) \<bowtie> concat ([x, x, y] \<cdot> ws')\<close>, folded lcp_ws_def, unfolded lcp_xp_z, symmetric])
      show "ws' \<cdot> [x, x, y] \<in> lists {x, y}" and  "[x, x, y] \<cdot> ws' \<in> lists {x, y}"
        by inlists
    qed

\<comment> \<open>Looking at the second occurrence of z:\<close>

    define ws'' where "ws'' = ps \<cdot> [x,y]"
    define lcp_ws' where "lcp_ws' = ws' \<cdot> ws'' \<and>\<^sub>p ws'' \<cdot> ws'"

    have "lcp_ws' \<in> lists {x,y}"
      unfolding lcp_ws'_def
      using \<open>ps \<in> lists {x, y}\<close> \<open>ws' \<in> lists {x, y}\<close> ws''_def by inlists

    have "concat ws'' = z \<cdot> xp"
      unfolding ws''_def xp_def using \<open>concat ps \<cdot> p = z\<close> xxy_root by fastforce

    have "ws' \<cdot> ws'' \<noteq> ws'' \<cdot> ws'"
    proof
      assume "ws' \<cdot> ws'' = ws'' \<cdot> ws'"
      from arg_cong[OF this, of concat, unfolded concat_morph con_ws
          \<open>concat ws'' = z \<cdot> xp\<close>,
          unfolded lassoc pow_comm, unfolded rassoc cancel]
      show False
        using \<open>z \<cdot> xp \<noteq> xp \<cdot> z\<close> comm_drop_exp'[OF _ \<open>0 < n\<close>] by blast
    qed

    have
      lcp_xp_z': "concat (ws' \<cdot> ws'') \<and>\<^sub>p concat (ws'' \<cdot> ws') = z \<cdot> bin_lcp z (x \<cdot> p)"
      unfolding concat_morph con_ws \<open>concat ws'' = z \<cdot> xp\<close> pow_Suc
      unfolding lcp_ext_left[symmetric] bin_lcp_def shifts
      unfolding rassoc lcp_ext_left cancel
      using bin_lcp_pows[OF \<open>0 < n\<close>, of 1 \<epsilon>  "z\<^sup>@(n-1)", unfolded pow_1, folded pow_pos[OF \<open>0 < n\<close>]]
      unfolding bin_lcp_def xp_def rassoc emp_simps by linarith

    have "z \<cdot> bin_lcp z (x \<cdot> p) = concat (lcp_ws') \<cdot> bin_lcp x y"
      unfolding lcp_xp_z'[symmetric] lcp_ws'_def
    proof (rule xy.bin_code_lcp_concat')
      show "ws' \<cdot> ws'' \<in> lists {x, y}"
        unfolding ws''_def using \<open>ws' \<cdot> ws' \<cdot> ws' \<in> lists {x, y}\<close> \<open>ps \<in> lists {x,y}\<close> by inlists
      thus "ws'' \<cdot> ws' \<in> lists {x, y}"
        by inlists
      show "\<not> concat (ws' \<cdot> ws'') \<bowtie> concat (ws'' \<cdot> ws')"
        unfolding concat_morph con_ws \<open>concat ws'' = z \<cdot> xp\<close> pow_pos[OF \<open>0 < n\<close>]
        unfolding rassoc comp_cancel
        unfolding lassoc pow_pos[OF \<open>0 < n\<close>, symmetric] pow_pos'[OF \<open>0 < n\<close>, symmetric]
           comm_comp_eq_conv
        using  comm_drop_exp'[OF _ \<open>0 < n\<close>, of z n xp] non_comm by argo
    qed

    have "concat lcp_ws' = z \<cdot> concat lcp_ws"
      unfolding cancel_right[of "concat lcp_ws'" "bin_lcp x y" "z \<cdot> concat lcp_ws", symmetric]
      unfolding rassoc[of z] \<open>concat (lcp_ws) \<cdot> bin_lcp x y = bin_lcp z (x \<cdot> p)\<close> \<open>z \<cdot> bin_lcp z (x \<cdot> p) = concat (lcp_ws') \<cdot> bin_lcp x y\<close>..

    have "lcp_ws \<le>p ws' \<cdot> [x,x,y]"
      unfolding lcp_ws_def using  longest_common_prefix_prefix1.
    have "lcp_ws \<noteq> ws' \<cdot> [x,x,y]"
      unfolding lcp_ws_def lcp_pref_conv
      using  \<open>ws' \<cdot> [x, x, y] \<noteq> [x, x, y] \<cdot> ws'\<close> pref_comm_eq by blast
    have "lcp_ws \<le>p ws' \<cdot> [x,x]"
      using spref_butlast_pref[OF  \<open>lcp_ws \<le>p ws' \<cdot> [x,x,y]\<close> \<open>lcp_ws \<noteq> ws' \<cdot> [x,x,y]\<close>]
      unfolding butlast_append by simp
    from  prefixE[OF pref_prolong[OF this \<open>[x,x] \<le>p ws'\<close>]]
    obtain ws''\<^sub>1 where "ws' \<cdot> ws' \<cdot> ws' = lcp_ws \<cdot> ws''\<^sub>1" using rassoc by metis

    have "ws' \<cdot> ps \<cdot> [x,y] \<le>p ws' \<cdot> ps \<cdot> [x,y,x]"
      by simp
    from pref_trans[OF pref_trans[OF longest_common_prefix_prefix1 this]]
    have "lcp_ws' \<le>p ws' \<cdot> ws' \<cdot> ws'"
      unfolding lcp_ws'_def ws''_def using \<open>ps \<cdot> vs \<le>p ws' \<cdot> ws'\<close>[unfolded cover_xyx, unfolded pref_cancel_conv]
      unfolding pref_cancel_conv[symmetric, of "ps \<cdot> [x,y,x]" "ws' \<cdot> ws'" ws'] by blast
    from prefixE[OF this]
    obtain ws''\<^sub>2 where "ws' \<cdot> ws' \<cdot> ws' = lcp_ws' \<cdot> ws''\<^sub>2".

    have "concat lcp_ws'\<cdot> concat ws''\<^sub>1 = z \<cdot> concat(lcp_ws) \<cdot> concat ws''\<^sub>1"
      unfolding lassoc \<open>concat lcp_ws' = z \<cdot> concat lcp_ws\<close>..
    also have "... = z \<cdot> concat (ws' \<cdot> ws' \<cdot> ws')"
      unfolding rassoc \<open>ws' \<cdot> ws' \<cdot> ws' = lcp_ws \<cdot> ws''\<^sub>1\<close> concat_morph..
    also have "... = concat (ws' \<cdot> ws' \<cdot> ws') \<cdot> z"
      unfolding concat_morph con_ws add_exps[symmetric]
        pow_Suc[symmetric] pow_Suc'[symmetric]..
    also have "... = concat lcp_ws'\<cdot> concat ws''\<^sub>2 \<cdot> z"
      unfolding \<open>ws' \<cdot> ws' \<cdot> ws' = lcp_ws' \<cdot> ws''\<^sub>2\<close> concat_morph rassoc..
    finally have "concat ws''\<^sub>1 = concat ws''\<^sub>2 \<cdot> z"
      unfolding cancel.

    from xy.stability[of "concat ws''\<^sub>2"  "concat lcp_ws" z,
        folded \<open>concat ws''\<^sub>1 = concat ws''\<^sub>2 \<cdot> z\<close> \<open>concat lcp_ws' = z \<cdot> concat lcp_ws\<close>]
    have "z \<in> \<langle>{x, y}\<rangle>"
      using \<open>ws' \<cdot> ws' \<cdot> ws' = lcp_ws \<cdot> ws''\<^sub>1\<close> \<open>ws' \<cdot> ws' \<cdot> ws' = lcp_ws' \<cdot> ws''\<^sub>2\<close> \<open>ws' \<cdot> ws' \<cdot> ws' \<in> lists {x, y}\<close>
        append_in_lists_dest append_in_lists_dest' concat_in_hull' by metis
    thus False
      using  \<open>z \<notin> \<langle>{x,y}\<rangle>\<close> by blast
  qed
  thus "ws \<sim> [x, x, y]"
    using \<open>ws \<sim> ws'\<close> by blast
qed

subsection \<open>Obtaining primitivity with two squares (refining)\<close>

lemma bin_imprim_both_squares_prim:
  assumes "x \<cdot> y \<noteq> y \<cdot> x"
    and "ws \<in> lists {x, y}"
    and "primitive ws" and "\<not> primitive (concat ws)"
    and "[x, x] \<le>f ws \<cdot> ws"
    and "[y, y] \<le>f ws \<cdot> ws"
    and  "primitive x" and "primitive y"
  shows False
proof-
  have "x \<noteq> y" using \<open>x \<cdot> y \<noteq> y \<cdot> x\<close>
    by blast
  from bin_imprim_primitive[OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> \<open>primitive x\<close> \<open>primitive y\<close>
      _ \<open>ws \<in> lists {x,y}\<close> \<open>primitive ws\<close> \<open>\<not> primitive (concat ws)\<close> \<open>[x, x] \<le>f ws \<cdot> ws\<close>]
    bin_imprim_primitive[OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close>[symmetric] \<open>primitive y\<close> \<open>primitive x\<close>
      _ \<open>ws \<in> lists {x,y}\<close>[unfolded insert_commute[of x]] \<open>primitive ws\<close> \<open>\<not> primitive (concat ws)\<close>
      \<open>[y, y] \<le>f ws \<cdot> ws\<close>]
  have "ws \<sim> [x, x, y] \<or> ws \<sim> [y, y, x]"
    using \<open>x \<cdot> y \<noteq> y \<cdot> x\<close>
    by force
  hence "\<^bold>|ws\<^bold>| = 3"
    using conjug_len by force
  note[simp] = sublist_code(3)
  from \<open>\<^bold>|ws\<^bold>| = 3\<close> \<open>ws \<in> lists {x,y}\<close> \<open>x \<noteq> y\<close>
       \<open>[x, x] \<le>f ws \<cdot> ws\<close> \<open>[y, y] \<le>f ws \<cdot> ws\<close>
  show False
   by list_inspection simp_all
qed

lemma bin_imprim_both_squares:
  assumes "x \<cdot> y \<noteq> y \<cdot> x"
    and "ws \<in> lists {x, y}"
    and "primitive ws" and "\<not> primitive (concat ws)"
    and "[x, x] \<le>f ws \<cdot> ws"
    and "[y, y] \<le>f ws \<cdot> ws"
  shows False
proof (rule bin_imprim_both_squares_prim)
  have "x \<noteq> \<epsilon>" and "y \<noteq> \<epsilon>" and "x \<noteq> y"
    using \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> by blast+
  let ?R = "\<lambda> x. [\<rho> x]\<^sup>@(e\<^sub>\<rho> x)"
  define ws' where "ws' = concat (map ?R ws)"
  show "\<rho> x \<cdot> \<rho> y \<noteq> \<rho> y \<cdot> \<rho> x"
    using  \<open>x \<cdot> y \<noteq> y \<cdot> x\<close>[unfolded comp_primroot_conv'[of x y]].
  have [simp]: "a = x \<or> a = y \<Longrightarrow> [\<rho> a] \<^sup>@ e\<^sub>\<rho> a \<in> lists {\<rho> x, \<rho> y}" for a
    using insert_iff sing_pow_lists[of _ "{\<rho> x, \<rho> y}"] by metis
  show "ws' \<in> lists {\<rho> x, \<rho> y}"
    unfolding ws'_def using \<open>ws \<in> lists {x,y}\<close>
    by (induction ws, simp_all)

\<comment> \<open>The primitivity of ws' is obtained from the fact that the decompositions into
    roots is a primitive morphism\<close>
  interpret binary_code x y
    using \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> by unfold_locales
  note[simp] = sublist_code(3)
  have "\<^bold>|ws\<^bold>| \<le> 3 \<Longrightarrow> ws \<in> lists {x,y} \<Longrightarrow> x \<noteq> y \<Longrightarrow> [x, x] \<le>f ws \<cdot> ws \<Longrightarrow> [y, y] \<le>f ws \<cdot> ws \<Longrightarrow> False"
    by list_inspection simp_all
  from this[OF _ \<open> ws \<in> lists {x,y}\<close> \<open>x \<noteq> y\<close> \<open>[x, x] \<le>f ws \<cdot> ws\<close> \<open>[y, y] \<le>f ws \<cdot> ws\<close>]
      roots_prim_morph[OF \<open>ws \<in> lists {x,y}\<close> _ \<open>primitive ws\<close>]
  show "primitive ws'"
    unfolding ws'_def  by fastforce

  show "\<not> primitive (concat ws')"
    unfolding ws'_def concat_root_dec_eq_concat[OF \<open>ws \<in> lists{x,y}\<close>] by fact

  have "concat(map ?R [x,x]) \<le>f ws' \<cdot> ws'" and "concat(map ?R [y,y]) \<le>f ws' \<cdot> ws'"
    unfolding ws'_def
    using concat_mono_fac[OF map_mono_sublist[OF \<open>[x,x] \<le>f ws \<cdot> ws\<close>]]
          concat_mono_fac[OF map_mono_sublist[OF \<open>[y,y] \<le>f ws \<cdot> ws\<close>]]
    unfolding concat_morph  map_append.

  have "Suc (Suc (e\<^sub>\<rho> x + e\<^sub>\<rho> x - 2)) = e\<^sub>\<rho> x + e\<^sub>\<rho> x"
    using Suc_minus2 primroot_exp_nemp[OF \<open>x \<noteq> \<epsilon>\<close>] by simp
  have "concat (map ?R [x,x]) = [\<rho> x] \<^sup>@ (Suc (e\<^sub>\<rho> x -1) + Suc (e\<^sub>\<rho> x - 1))"
    unfolding  Suc_minus_pos[OF primroot_exp_nemp[OF \<open>x \<noteq> \<epsilon>\<close>]] by (simp add: add_exps)
  hence "[\<rho> x, \<rho> x] \<le>f concat (map ?R [x,x])"
    by auto
  thus "[\<rho> x, \<rho> x] \<le>f ws' \<cdot> ws'"
    using fac_trans[OF _ \<open>concat(map ?R [x,x]) \<le>f ws' \<cdot> ws'\<close>] by blast

  have "Suc (Suc (e\<^sub>\<rho> y + e\<^sub>\<rho> y - 2)) = e\<^sub>\<rho> y + e\<^sub>\<rho> y"
    using Suc_minus2 primroot_exp_nemp[OF \<open>y \<noteq> \<epsilon>\<close>] by simp
  have "concat (map ?R [y,y]) = [\<rho> y] \<^sup>@ (Suc (e\<^sub>\<rho> y -1) + Suc (e\<^sub>\<rho> y - 1))"
    unfolding  Suc_minus_pos[OF primroot_exp_nemp[OF \<open>y \<noteq> \<epsilon>\<close>]] by (simp add: add_exps)
  hence "[\<rho> y, \<rho> y] \<le>f concat (map ?R [y,y])"
    by auto
  thus "[\<rho> y, \<rho> y] \<le>f ws' \<cdot> ws'"
    using fac_trans[OF _ \<open>concat(map ?R [y,y]) \<le>f ws' \<cdot> ws'\<close>] by blast

  show "primitive (\<rho> x)" and "primitive (\<rho> y)"
    using  primroot_prim \<open>x \<noteq> \<epsilon>\<close> \<open>y \<noteq> \<epsilon>\<close> by blast+
qed

subsection \<open>Obtaining the square of the longer word (gluing)\<close>

lemma bin_imprim_longer_twice:
  \<comment> \<open>
1. If there are both squares, then contradiction;
2. If a square is missing:
   a) if y appears once: the positive conclusion
   b) if y appears twice, then gluing preserves presence of the longer word at least twice (because both appear twice)
      and induction yields [x',x',y'] where y' is a suffix of x',
      a contradiction with primitivity of words of the form xyxyy;
\<close>
  assumes "x \<cdot> y \<noteq> y \<cdot> x"
    and "ws \<in> lists {x, y}"
    and "\<^bold>|y\<^bold>| \<le> \<^bold>|x\<^bold>|"
    and "count_list ws x \<ge> 2"
    and "primitive ws" and "\<not> primitive (concat ws)"
  shows "ws \<sim> [x,x,y] \<and> primitive x \<and> primitive y"
  using assms proof (induction "\<^bold>|ws\<^bold>|" arbitrary: x y ws rule: less_induct)
  case less
  then show ?case
  proof (cases)
    assume "[x, x] \<le>f ws \<cdot> ws \<and> [y, y] \<le>f ws \<cdot> ws"
    with bin_imprim_both_squares[OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> \<open>ws \<in> lists {x,y}\<close> \<open>primitive ws\<close> \<open> \<not> primitive (concat ws)\<close>]
    have False by blast
    thus ?case by blast
  next
    assume missing_sq: "\<not> ([x, x] \<le>f ws \<cdot> ws \<and> [y, y] \<le>f ws \<cdot> ws)"
    then show ?case
    proof (cases)
      assume "count_list ws y < 2"
      with bin_imprim_single_y[OF less.prems(1-4) this less.prems(5-6)]
      show "ws \<sim> [x,x,y] \<and> primitive x \<and> primitive y"
        by blast
    next
      assume "\<not> count_list ws y < 2" hence "2 \<le> count_list ws y" by simp

\<comment> \<open>Missing square and two y's allow gluing\<close>
      define x' where "x' = (if \<not> [x, x] \<le>f ws \<cdot> ws then x else y)"
      define y' where "y' = (if \<not> [x, x] \<le>f ws \<cdot> ws then y else x)"

      have "{x', y'} = {x, y}"
        by (simp add: doubleton_eq_iff x'_def y'_def)
      note cases = disjE[OF this[unfolded doubleton_eq_iff]]
      have "\<not> [x', x'] \<le>f ws \<cdot> ws"
        using missing_sq x'_def by presburger
      have "count_list ws x' \<ge> 2" and "count_list ws y' \<ge> 2"
        unfolding x'_def y'_def using \<open>2 \<le> count_list ws x\<close> \<open>2 \<le> count_list ws y\<close> by presburger+
      have "x' \<cdot>  y' \<noteq> y' \<cdot> x'"
        by (rule cases, simp_all add: \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> \<open>x \<cdot> y \<noteq> y \<cdot> x\<close>[symmetric])
      have "x' \<noteq> \<epsilon>" and "x' \<noteq> y'" and "x' \<cdot> y' \<noteq> y'"
        using \<open>x' \<cdot> y' \<noteq> y' \<cdot> x'\<close> by auto

\<comment> \<open>rotating last if necessary for successful gluing\<close>
      note prim_nemp[OF \<open>primitive ws\<close>]
      hence rot: "last ws = x' \<Longrightarrow> hd ws = x' \<Longrightarrow> butlast ws \<cdot> [x',x'] \<cdot> tl ws = ws \<cdot> ws"
        using append_butlast_last_id hd_tl hd_word rassoc by metis
      from this[THEN facI']
      have "last ws = x' \<Longrightarrow> hd ws \<noteq> x'"
        using \<open>\<not> [x', x'] \<le>f ws \<cdot> ws\<close>  by blast
      define ws' where "ws' = (if last ws \<noteq> x' then ws else tl ws \<cdot> [hd ws])"
      have cond: "ws' = \<epsilon> \<or> last ws' \<noteq> x'" \<comment> \<open>gluing condition\<close>
        unfolding ws'_def using \<open>last ws = x' \<Longrightarrow> hd ws \<noteq> x'\<close> by simp
      have "ws' \<sim> ws"
        unfolding ws'_def using \<open>ws \<noteq> \<epsilon>\<close> by fastforce
      hence counts': "count_list ws' x' \<ge> 2" "count_list ws' y' \<ge> 2"
        by (simp_all add: \<open>2 \<le> count_list ws x'\<close> \<open>2 \<le> count_list ws y'\<close> count_list_conjug)

\<comment> \<open>verify induction assumptions of the glued word\<close>
      let ?ws = "glue x' ws'"
      have c1: "\<^bold>|?ws\<^bold>| < \<^bold>|ws\<^bold>|"
        using len_glue[OF cond] conjug_len[OF \<open>ws' \<sim> ws\<close>] \<open>count_list ws' x' \<ge> 2\<close> by linarith
      hence c2: "(x' \<cdot> y') \<cdot> y' \<noteq> y' \<cdot> x' \<cdot> y'"
        using \<open>x' \<cdot> y' \<noteq> y' \<cdot> x'\<close> by force

      have "ws' \<le>f ws \<cdot> ws"
        using conjugE[OF \<open>ws' \<sim> ws\<close>] rassoc sublist_appendI by metis
      hence "\<not> [x', x'] \<le>f ws'"
        using \<open>\<not> [x',x'] \<le>f ws \<cdot> ws\<close> by blast
      have "ws' \<in> lists {x',y'}"
        using conjug_in_lists[OF \<open>ws' \<sim> ws\<close> \<open>ws \<in> lists {x,y}\<close>[folded \<open>{x',y'} = {x,y}\<close>]].
      have  c3: "?ws \<in> lists {x' \<cdot> y', y'}"
        using single_bin_glue_in_lists[OF cond \<open>\<not> [x', x'] \<le>f ws'\<close> \<open>ws' \<in> lists {x',y'}\<close>].

      have c4: "2 \<le> count_list (glue x' ws') (x' \<cdot> y')"
        using \<open>2 \<le> count_list ws' x'\<close>
        unfolding count_list_single_bin_glue(1)[OF \<open>x' \<noteq> \<epsilon>\<close> \<open>x' \<noteq> y'\<close> cond \<open>ws' \<in> lists {x',y'}\<close> \<open>\<not> [x',x'] \<le>f ws'\<close>].

      from \<open>primitive ws\<close>[folded conjug_prim_iff[OF \<open>ws' \<sim> ws\<close>]]
      have c5: "primitive (glue x' ws')"
        using prim_bin_glue [OF \<open>ws' \<in> lists {x',y'}\<close> \<open>x' \<noteq> \<epsilon>\<close> cond] by blast

      have "count_list ws' x' \<ge> 2"
        using \<open>count_list ws x \<ge> 2\<close> \<open>count_list ws y \<ge> 2\<close> \<open>{x', y'} = {x, y}\<close>
          count_list_conjug[OF \<open>ws' \<sim> ws\<close>] x'_def by metis

      have "concat (glue x' ws') = concat ws'"
        by (simp add: cond)
      have c6: "\<not> primitive (concat (glue x' ws'))"
        unfolding \<open>concat (glue x' ws') = concat ws'\<close> using \<open>\<not> primitive (concat ws)\<close> \<open>ws' \<sim> ws\<close>
          conjug_concat_conjug prim_conjug by metis
          \<comment> \<open>The claim holds by induction\<close>
      from less.hyps[OF c1 c2 c3 _ c4 c5 c6]
      have "glue x' ws' \<sim> [x' \<cdot> y', x' \<cdot> y', y']" by simp
          \<comment> \<open>Which is impossible after gluing\<close>
      from prim_xyxyy[OF \<open>x' \<cdot> y' \<noteq> y' \<cdot> x'\<close>] conjug_prim_iff[OF conjug_concat_conjug[OF this]]
      have False
        using \<open>\<not> primitive (concat (glue x' ws'))\<close> by simp
      thus ?case by blast
    qed
  qed
qed

lemma bin_imprim_both_twice:
  assumes "x \<cdot> y \<noteq> y \<cdot> x"
    and "ws \<in> lists {x, y}"
    and "count_list ws x \<ge> 2"
    and "count_list ws y \<ge> 2"
    and "primitive ws" and "\<not> primitive (concat ws)"
  shows False
proof-
  have "x \<noteq> y"
    using \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> by blast
  from bin_imprim_longer_twice[OF assms(1-2) _ assms(3) assms(5-6)]
    bin_imprim_longer_twice[OF assms(1)[symmetric] assms(2)[unfolded insert_commute[of x]] _ assms(4) assms(5-6)]
  have or: "ws \<sim> [x, x, y] \<or> ws \<sim> [y, y, x]" by linarith
  thus False
  proof (rule disjE)
    assume "ws \<sim> [x, x, y]"
    from \<open>count_list ws y \<ge> 2\<close>[unfolded count_list_conjug[OF this]]
    show False
      using \<open>x \<noteq> y\<close> by force
  next
    assume "ws \<sim> [y, y, x]"
    from \<open>count_list ws x \<ge> 2\<close>[unfolded count_list_conjug[OF this]]
    show False
      using \<open>x \<noteq> y\<close> by force
  qed
qed

section \<open>Examples\<close>

lemma "x \<noteq> \<epsilon> \<Longrightarrow>  \<epsilon> (x\<cdot>x) \<epsilon> \<sim>\<^sub>\<I> [x,x]"
  unfolding factor_interpretation_def
  by simp

lemma assumes "x = [(0::nat),1,0,1,0]" and "y = [1,0,0,1]"
  shows "[0,1] (x\<cdot>x) [1,0] \<sim>\<^sub>\<I> [x,y,x]"
  unfolding factor_interpretation_def assms by (simp add: suffix_def)

section "Primitivity non-preserving binary code"

text\<open>In this section, we give the final form of imprimitive words over a given
binary code @{term "{x,y}"}.
We start with a lemma, then we show that the only possibility is that such
word is conjugate with @{term "x\<^sup>@j \<cdot> y\<^sup>@k"}.\<close>

lemma bin_imprim_expsE_y: assumes "x \<cdot> y \<noteq> y \<cdot> x" and
  "ws \<in> lists {x,y}" and
  "2 \<le> \<^bold>|ws\<^bold>|" and
  "primitive ws" and
  "\<not> primitive (concat ws)" and
  "count_list ws y = 1"
obtains j k where "1 \<le> j" "1 \<le> k" "j = 1 \<or> k = 1"
  "ws \<sim> [x]\<^sup>@j \<cdot> [y]\<^sup>@k"
proof-
  have "x \<noteq> y" using \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> by blast
  obtain j1 j2 where "[x]\<^sup>@j1\<cdot>[y]\<cdot>[x]\<^sup>@j2 = ws"
    using bin_count_one_decompose[OF \<open>ws \<in> lists {x,y}\<close> \<open>x \<noteq> y\<close> \<open>count_list ws y = 1\<close>].
  have "1 \<le> j2 + j1"
    using \<open>[x] \<^sup>@ j1 \<cdot> [y] \<cdot> [x] \<^sup>@ j2 = ws\<close> \<open>2 \<le> \<^bold>|ws\<^bold>|\<close> not_less_eq_eq  by fastforce
  have "ws \<sim> [x]\<^sup>@(j2+j1)\<cdot>[y]\<^sup>@1"
    using conjugI'[of "[x] \<^sup>@ j1 \<cdot> [y]" "[x] \<^sup>@ j2"]
    unfolding \<open>[x] \<^sup>@ j1 \<cdot> [y] \<cdot> [x] \<^sup>@ j2 = ws\<close>[symmetric] add_exps rassoc pow_1.
  from that[OF \<open>1 \<le> j2 + j1\<close> _ _ this]
  show ?thesis
    by blast
qed

lemma bin_imprim_expsE: assumes "x \<cdot> y \<noteq> y \<cdot> x" and
  "ws \<in> lists {x,y}" and
  "2 \<le> \<^bold>|ws\<^bold>|" and
  "primitive ws" and
  "\<not> primitive (concat ws)"
obtains j k where "1 \<le> j" "1 \<le> k" "j = 1 \<or> k = 1"
  "ws \<sim> [x]\<^sup>@j \<cdot> [y]\<^sup>@k"
proof-
  note \<open>ws \<in> lists {x,y}\<close>[unfolded insert_commute[of x]]

  from bin_lists_count_zero[OF \<open>ws \<in> lists {x,y}\<close>]
    sing_lists_exp_len[of ws y]
    prim_exp_one[OF \<open>primitive ws\<close>, of "[y]" "\<^bold>|ws\<^bold>|"]
  have "count_list ws x \<noteq> 0"
    using \<open>2 \<le> \<^bold>|ws\<^bold>|\<close> by fastforce

  from bin_lists_count_zero[OF \<open>ws \<in> lists {y,x}\<close>]
    sing_lists_exp_len[of ws x]
    prim_exp_one[OF \<open>primitive ws\<close>, of "[x]" "\<^bold>|ws\<^bold>|"]
  have "count_list ws y \<noteq> 0"
    using \<open>2 \<le> \<^bold>|ws\<^bold>|\<close> by fastforce

  consider "count_list ws x = 1" | "count_list ws y = 1"
    using bin_imprim_both_twice[OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> \<open>ws \<in> lists {x,y}\<close> _ _
        \<open>primitive ws\<close> \<open>\<not> primitive (concat ws)\<close>]
      \<open>count_list ws x \<noteq> 0\<close> \<open>count_list ws y \<noteq> 0\<close>
    unfolding One_less_Two_le_iff[symmetric] less_one[symmetric] by fastforce
  thus thesis
  proof(cases)
    assume \<open>count_list ws x = 1\<close>
    from bin_imprim_expsE_y[reversed, OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> \<open>ws \<in> lists {y, x}\<close> \<open>2 \<le> \<^bold>|ws\<^bold>|\<close>
        \<open>primitive ws\<close> \<open>\<not> primitive (concat ws)\<close> \<open>count_list ws x = 1\<close>]
    show thesis
      using that by metis
  next
    assume \<open>count_list ws y = 1\<close>
    from bin_imprim_expsE_y[OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> \<open>ws \<in> lists {x, y}\<close> \<open>2 \<le> \<^bold>|ws\<^bold>|\<close>
        \<open>primitive ws\<close> \<open>\<not> primitive (concat ws)\<close> \<open>count_list ws y = 1\<close>]
    show ?thesis
      using that.
  qed
qed

subsection \<open>The target theorem\<close>

text\<open>Given a binary code @{term "{x,y}"} such that there is a primitive factorisation
@{term ws} over it whose concatenation is imprimitive, we finally show that there are
integers @{term j} and @{term k} (depending only on @{term "{x,y}"}) such that
any other such factorisation @{term ws'} is conjugate to @{term "[x]\<^sup>@j \<cdot> [y]\<^sup>@k"}.\<close>

theorem bin_imprim_code: assumes "x \<cdot> y \<noteq> y \<cdot> x"  and "ws \<in> lists {x,y}" and
  "2 \<le> \<^bold>|ws\<^bold>|" and "primitive ws" and  "\<not> primitive (concat ws)"
obtains j k where "1 \<le> j" and "1 \<le> k" and "j = 1 \<or> k = 1"
  "\<And>ws. ws \<in> lists {x,y} \<Longrightarrow>  2 \<le> \<^bold>|ws\<^bold>| \<Longrightarrow>
       (primitive ws \<and> \<not> primitive (concat ws) \<longleftrightarrow> ws \<sim> [x]\<^sup>@j \<cdot> [y]\<^sup>@k)" and
  "\<^bold>|y\<^bold>| \<le> \<^bold>|x\<^bold>| \<Longrightarrow> 2 \<le> j \<Longrightarrow> j = 2 \<and> primitive x \<and> primitive y" and
  "\<^bold>|y\<^bold>| \<le> \<^bold>|x\<^bold>| \<Longrightarrow> 2 \<le> k \<Longrightarrow> j = 1 \<and> primitive x"
proof-
  obtain j k where "1 \<le> j" "1 \<le> k" "j = 1 \<or> k = 1"
    "ws \<sim> [x]\<^sup>@j \<cdot> [y]\<^sup>@k"
    using bin_imprim_expsE[OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close>]
    using assms by metis

  have "\<not> primitive (x\<^sup>@j \<cdot> y\<^sup>@k)"
    using \<open>\<not> primitive (concat ws)\<close>
    unfolding concat_morph  concat_sing_pow
      conjug_prim_iff[OF conjug_concat_conjug[OF \<open>ws \<sim> [x] \<^sup>@ j \<cdot> [y] \<^sup>@ k\<close>]].

  from not_prim_primroot_expE[OF this]
  obtain z l where [symmetric]: "z\<^sup>@l = x\<^sup>@j \<cdot> y\<^sup>@k" and "2 \<le> l".

  show thesis
  proof (rule that[of j k ])
    show "1 \<le> j" "1 \<le> k" "j = 1 \<or> k = 1" by fact+

    fix ws'
    assume hyps: "ws' \<in> lists {x,y}" "2 \<le> \<^bold>|ws'\<^bold>|"
    show  "primitive ws' \<and> \<not> primitive (concat ws') \<longleftrightarrow> ws' \<sim> [x]\<^sup>@j \<cdot> [y]\<^sup>@k"
    proof
      assume " primitive ws' \<and> \<not> primitive (concat ws')"
      hence prems: "primitive ws'" "\<not> primitive (concat ws')" by blast+
      obtain j' k' where "1 \<le> j'" "1 \<le> k'" "j' = 1 \<or> k' = 1"
        "ws' \<sim> [x]\<^sup>@j' \<cdot> [y]\<^sup>@k'"
        using bin_imprim_expsE[OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close> hyps prems].

      have "\<not> primitive (x \<^sup>@ j'\<cdot> y \<^sup>@ k')"
        using  \<open>\<not> primitive (concat ws')\<close>
        unfolding concat_morph  concat_sing_pow
          conjug_prim_iff[OF conjug_concat_conjug[OF \<open>ws' \<sim> [x] \<^sup>@ j' \<cdot> [y] \<^sup>@ k'\<close>]].

      have "j = j'" "k = k'"
        using LS_unique[OF \<open>x \<cdot> y \<noteq> y \<cdot> x\<close>
            \<open>1 \<le> j\<close> \<open>1 \<le> k\<close>  \<open>\<not> primitive (x \<^sup>@ j \<cdot> y \<^sup>@ k)\<close>
            \<open>1 \<le> j'\<close> \<open>1 \<le> k'\<close> \<open>\<not> primitive (x \<^sup>@ j'\<cdot> y \<^sup>@ k')\<close>].

      show "ws' \<sim> [x] \<^sup>@ j \<cdot> [y] \<^sup>@ k"
        unfolding \<open>j = j'\<close> \<open>k = k'\<close> by fact
    next
      assume  "ws' \<sim> [x]\<^sup>@j \<cdot> [y]\<^sup>@k"
      note conjug_trans[OF \<open>ws \<sim> [x]\<^sup>@j \<cdot> [y]\<^sup>@k\<close> conjug_sym[OF this]]
      from prim_conjug[OF \<open>primitive ws\<close> this]
           \<open>\<not> primitive (concat ws)\<close>[unfolded conjug_concat_prim_iff[OF \<open>ws \<sim> ws'\<close>]]
      show "primitive ws' \<and> \<not> primitive (concat ws')" by blast
    qed
  next
    assume "\<^bold>|y\<^bold>| \<le> \<^bold>|x\<^bold>|"
    interpret LS_len_le x y j k l z
    by unfold_locales fact+

    assume "2 \<le> j"
    with jk_small
    have "k = 1" by fastforce
    from case_j2k1[OF \<open>2 \<le> j\<close> this]
    show "j = 2 \<and> primitive x \<and> primitive y"
      by blast
  next
    assume "\<^bold>|y\<^bold>| \<le> \<^bold>|x\<^bold>|"
    interpret LS_len_le x y j k l z
      by unfold_locales fact+

    assume "2 \<le> k"
    show "j = 1 \<and> primitive x"
      using \<open>2 \<le> k\<close> \<open>j = 1 \<or> k = 1\<close>   case_j1k2_primitive by auto
  qed
qed

\<comment> \<open>Formulation in terms of (binary) primitive morphism\<close>

definition bin_imprim_code where "bin_imprim_code x y \<equiv> x \<cdot> y \<noteq> y \<cdot> x \<and> (\<not> bin_prim x y)"

theorem bin_imprim_code': assumes "bin_imprim_code x y"
obtains j k where "1 \<le> j" and "1 \<le> k" and "j = 1 \<or> k = 1"
  "\<And>ws. ws \<in> lists {x,y} \<Longrightarrow>  2 \<le> \<^bold>|ws\<^bold>| \<Longrightarrow>
       (primitive ws \<and> \<not> primitive (concat ws) \<longleftrightarrow> ws \<sim> [x]\<^sup>@j \<cdot> [y]\<^sup>@k)" and
  "\<^bold>|y\<^bold>| \<le> \<^bold>|x\<^bold>| \<Longrightarrow> 2 \<le> j \<Longrightarrow> j = 2 \<and> primitive x \<and> primitive y" and
  "\<^bold>|y\<^bold>| \<le> \<^bold>|x\<^bold>| \<Longrightarrow> 2 \<le> k \<Longrightarrow> j = 1 \<and> primitive x"
proof-
  thm bin_imprim_code
  obtain ws where "x \<cdot> y \<noteq> y \<cdot> x"
  and "ws \<in> lists {x,y}" and  "2 \<le> \<^bold>|ws\<^bold>|" and "primitive ws" and  "\<not> primitive (concat ws)"
    using assms  unfolding bin_imprim_code_def bin_prim_altdef2 by blast
  from bin_imprim_code[OF this] that
  show thesis
    by blast
qed


end
